#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
import logging
import os
import time

from . import slices
from .. import bundles
from .. import check
from .. import matrixnet
from .. import utils


def calc_formula_id_worker(path, binding):
    binding.SetProperty("formula-id", matrixnet.calc_id(path))


def annotate_matrixnet_worker(path, binding):
    binding.SetProperty("Slices", slices.calc_slices_in_binding(binding, path))


def save_meta(path, additional):
    meta = {
        'time': time.strftime('%Y-%m-%d %H:%M:%S'),
    }
    if additional:
        meta.update(additional)

    logging.info('Meta information: %s', meta)
    with open(path, 'w') as f:
        json.dump(meta, f)


def prepare(
    mxops_path,
    models_dir,
    production=True,
    check_l3=True,
    check_id=True,
    check_slices=True,
    streams=None,
    meta=None,
    fail_without_slices=False,
    check_mappings_prefix=None,
):
    """
    Prepare models before creating models.archive

    :param mxops_path: путь к mx_ops
    :param models_dir: директория с моделями
    :param production: проверять продакшн модели на наличие Fast/Hast
    :param check_l3: проверять L3 модели
    :param check_id: требовать наличия formula-id у всех моделей
    :param check_slices: требовать наличия Slices у моделей ранжирования и у click моделей
    :param streams: список потоков, которые должны обязательно присутствовать в архиве
    :param meta: дополнительная мета информация об архиве с моделями
    :param fail_without_slices: требовать наличия slices во всех моделях
    :return:
    """
    top_files = list(utils.walk_files(models_dir, recursive=False))
    logging.debug("Top files: %s", top_files)
    if check_mappings_prefix is not None:
        check.check_mappings(top_files, check_mappings_prefix)
    top_models = [path for path in top_files if matrixnet.is_matrixnet(path)]
    top_bundles = [path for path in top_files if bundles.is_bundle_config(path)]
    if production:
        check.check_production(top_models, top_bundles, l3=check_l3, streams=streams)
    check.check_experiment(top_models, top_bundles)

    all_models = matrixnet.models_from_dir(models_dir, recursive=True)
    # infos = dict(pool.map(mxops_info_worker, ((mxops_path, path) for path in all_models)))
    import libmxnet
    models_bingings = {model_path: libmxnet.TMXNetInfo(model_path) for model_path in all_models}
    # ids = {path: mxops.info_props(info).get("formula-id") for path, info in infos.iteritems()}
    ids = {path: binding.GetProperty("formula-id") for path, binding in models_bingings.iteritems()}

    if check_id:
        bad_ranking_models = [os.path.basename(path) for path in top_models if not ids.get(path)]
        if bad_ranking_models:
            raise utils.ModelsError('Ranking models without formula-id: {}'.format(bad_ranking_models))

    without_id = [path for path in all_models if not ids.get(path)]
    if without_id:
        logging.debug("Paths without ids: %s", without_id)
        for path in without_id:
            calc_formula_id_worker(mxops_path, models_bingings[path])

    if check_slices:
        logging.info("Check Slices")
        models_slices = {path: binding.GetProperty("Slices") for path, binding in models_bingings.iteritems()}
        click_dir = os.path.join(models_dir, "click")
        annotate_models = top_models + matrixnet.models_from_dir(click_dir, recursive=True)
        without_slices = [path for path in annotate_models if not models_slices.get(path)]
        if without_slices:
            if fail_without_slices:
                raise utils.ModelsError('Ranking models without factor slices: {}'.format(without_slices))
            else:
                logging.info("Models without Slices: %s", without_slices)
                for path in without_slices:
                    annotate_matrixnet_worker(path, models_bingings[path])

    production_streams = utils.streams(top_models) | utils.streams(top_bundles)
    bundles.check_bundles(top_files, ids={ids.get(path) for path in top_models}, production_streams=production_streams)
    logging.debug("do call of cpp checker")
    from quality.relev_tools.models_archive_ops.pybinds import check as CppChecker
    assert CppChecker.CheckArchivePy(models_dir)

    save_meta(os.path.join(models_dir, 'meta'), additional=meta)
