# -*- coding: utf-8 -*-
import itertools
import json
import logging
import os

from . import matrixnet
from . import utils


def is_bundle_config(path):
    if path.endswith("Snippets.json"):
        return False
    return os.path.isfile(path) and path.endswith(".json")


def bundles_from_dir(path, recursive=True):
    """
    Перечислить все .json модели в директории

    :param path: путь до директории
    :param recursive: обходить ли поддиректории рекурсивно
    :return: список путей к файлам моделей
    """

    if is_bundle_config(path):
        return [path]
    elif os.path.isdir(path):
        logging.info('Get bundles from dir: %s', path)
        return [f for f in utils.walk_files(path, recursive) if is_bundle_config(f)]
    else:
        return []


def get_bundles_info(models, short_path=True, recursive=True):
    """
        :param models: пути к файлам и директориям
        :return: итератор (file-name, config_info)
    """
    all_models = itertools.chain.from_iterable(bundles_from_dir(m, recursive=recursive) for m in models)
    return (
        (path if not short_path else path.rsplit("/", 1)[-1], parse_bundle_config(path))
        for path in all_models
    )


def parse_bundle_config(path):
    logging.debug('Reading bundle: {0}'.format(path))
    name = os.path.basename(path)
    try:
        with open(path) as fpath:
            config = json.load(fpath)
    except IOError:
        raise utils.ModelsError('Cannot open {}'.format(name))
    except ValueError as e:
        raise utils.ModelsError('Cannot parse {}: {}'.format(name, e))
    if not isinstance(config, dict):
        raise utils.ModelsError('{} must be dict'.format(name))

    if isinstance(config.get("sum"), list) and len(config["sum"]) == 1:
        config_sum = config["sum"][0]
    elif isinstance(config.get("sum"), dict):
        config_sum = config["sum"]
    else:
        raise utils.ModelsError('sum in {} must be list with single element or dict'.format(name))
    if not isinstance(config_sum, dict):
        raise utils.ModelsError('sum elements in {} must be dict'.format(name))
    for weight in config_sum.itervalues():
        if not isinstance(weight, (float, int)):
            raise utils.ModelsError('sum weights in {} must be numbers'.format(name))

    return config_sum, config.get("formula-id")


def bundle_formula_ids(config_sum):
    """
    Получить список используемых formula-id

    :param config_sum: конфиг
    :return: генератор с formula-id
    """
    return (formula[1:] for formula in config_sum.iterkeys() if formula.startswith('@'))


def bundle_streams(config_sum):
    """
    Получить список используемых потоков

    :param config_sum: конфиг
    :return: генератор с stream-name
    """
    return (formula for formula in config_sum.iterkeys() if formula.startswith('full.'))


def check_bundles(files, ids, production_streams):
    """
    Убедиться, что все бандлы корректны и ссылаются на существующие formula-id

    :param files: пути к файлам
    :param ids: set {formula-id}
    :param production_streams: set {stream-name}
    :return:
    """
    logging.info('Checking bundles')
    models = set(utils.file_name(path) for path in files if matrixnet.is_matrixnet(path))
    bundles = [path for path in files if is_bundle_config(path)]
    bad_bundles = []
    for path in bundles:
        bundle_name = utils.file_name(path)
        if bundle_name in models:
            bad_bundles.append("Bundle {} has the same name as matrixnet model".format(bundle_name))
        config_sum = parse_bundle_config(path)[0]
        bad_ids = set(bundle_formula_ids(config_sum)) - ids
        if bad_ids:
            bad_bundles.append("Bundle {} contains unknown models: {}".format(bundle_name, bad_ids))
        bad_streams = set(bundle_streams(config_sum)) - production_streams
        if bad_streams:
            bad_bundles.append("Bundle {} contains unknown streams: {}".format(bundle_name, bad_streams))
    if bad_bundles:
        raise utils.ModelsError("\n".join(bad_bundles))
