#!/usr/bin/env python
# -*- coding: utf-8 -*-
import argparse
import logging
import os
from collections import defaultdict

from . import matrixnet
from . import utils

logger = logging.getLogger(__name__)


def check_production(matrixnet_models, bundle_models, l3=True, streams=None):
    """
    Проверить, что нет потоков без Fast/Hast моделей

    :param matrixnet_models: пути к .info-моделям (matrixnet)
    :param bundle_models: пути к .json-бандлам
    :param l3: проверять наличие L3 моделей
    :param streams: список потоков, которые должны обязательно присутствовать в архиве
    :return:
    """
    production_names = check_names(bundle_models, matrixnet_models)

    logger.info('Checking production models: %s', production_names)
    names_by_stream = defaultdict(set)
    production_streams = list(matrixnet.parse_streams(production_names))
    for name, stream in production_streams:
        if stream not in {"Fastrank", "Geo", "Ussr", "De"}:
            names_by_stream[stream].add(name)
    if streams:
        missed_streams = set(streams) - set(stream for _, stream in production_streams)
        if missed_streams:
            raise utils.ModelsError('Streams missed: {}'.format(sorted(missed_streams)))
    bad_names = []
    templates = ["{0}"] if l3 else ["{0}Fast", "{0}L1"]
    for stream, names in names_by_stream.iteritems():
        check = (t.format(stream) for t in templates)
        bad_names.extend(name for name in check if name not in names)

    if bad_names:
        raise utils.ModelsError("There are some models missing: {0}".format(", ".join(bad_names)))


def _check_duplicates(models_intersection):
    if not models_intersection:
        return

    raise utils.ModelsError(
        "Model bundles {} (.json) has the same name as .info (matrixnet) models".format(
            list(models_intersection))
    )


def check_names(bundle_models, matrixnet_models):
    matrixnet_names = utils.production_names(matrixnet_models)
    bundle_names = utils.production_names(bundle_models)
    production_intersection = matrixnet_names & bundle_names
    _check_duplicates(production_intersection)
    return matrixnet_names | bundle_names


def check_experiment(matrixnet_models, bundle_models, only='tdi'):
    """
    Проверить, что нет экспериментов с несколькими моделями в одном потоке

    :param matrixnet_models: пути к .info-моделям (matrixnet)
    :param bundle_models: пути к .json-бандлам
    :param only: проверить только определенный эксперимент
    :return:
    """

    matrixnet_names = utils.experiment_names(matrixnet_models)
    bundle_names = utils.experiment_names(bundle_models)
    experiment_intersection = matrixnet_names & bundle_names
    _check_duplicates(experiment_intersection)

    experiment_names = matrixnet_names | bundle_names

    logger.info('Checking experiment models: %s', experiment_names)
    experiments = defaultdict(lambda: defaultdict(set))
    for name, stream, e in matrixnet.parse_experiments(experiment_names):
        experiments[e][stream].add(name)

    def filter_bad(ex):
        return [names for names in ex.values() if len(names) > 1]

    if only:
        bad_exp = filter_bad(experiments.get(only, {}))
    else:
        bad_exp = ((name, filter_bad(e)) for name, e in experiments.iteritems())
        bad_exp = {name: e for name, e in bad_exp if e}

    if bad_exp:
        raise utils.ModelsError('There is more than one model in same experiment: {}'.format(bad_exp))


def check_mappings(top_files, check_mappings_prefix):
    files = {os.path.basename(i): i for i in top_files}
    logger.info("Try to check mappings with prefix: %s", check_mappings_prefix)
    for file_basename, file_name in files.items():
        if file_basename.startswith(check_mappings_prefix):
            not_exist = list(check_one_mapping(file_name, files))
            if not_exist:
                raise utils.ModelsError("There are files, which are not exist though required: {}".format(not_exist))


def check_one_mapping(file_name, formulas_files):
    logger.info("Check mapping %s", file_name)
    with open(file_name) as f:
        for line in f:
            formula_name_parts = line.split()
            formula_name_parts[-1] = "_{}_".format(formula_name_parts[-1])
            formula_name_parts.append(".json")
            formula_name = "".join(formula_name_parts)
            logger.debug("Check %s", formula_name)
            if formula_name in formulas_files:
                continue
            # sometimes, fml redefines .json models with .info models for tests
            formula_name_parts[-1] = ".info"
            formula_name = "".join(formula_name_parts)
            logger.debug("Check %s", formula_name)
            if formula_name in formulas_files:
                continue
            logger.info("There is no %s in provided files", formula_name)
            yield formula_name


def parse_args():
    args = argparse.ArgumentParser(description='Check model names')
    args.add_argument(
        "directory",
        type=str,
        help="path to directory with models",
    )
    args.add_argument(
        "-v",
        "--verbose",
        action='store_true',
        help="print logs",
    )
    return args.parse_args()


def main():
    args = parse_args()
    logging.basicConfig(
        format='%(levelname)s\t%(message)s',
        level=logging.INFO if args.verbose else logging.WARNING,
    )
    matrixnet_models = matrixnet.models_from_dir(args.directory, recursive=False)
    check_production(matrixnet_models=matrixnet_models, bundle_models=list())
    check_experiment(matrixnet_models=matrixnet_models, bundle_models=list())


if __name__ == "__main__":
    main()
