#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import json
import logging
import sys

from . import archiver
from . import utils


def generate_diff_footer(diff, vs="last_released"):

    if not diff:
        return [{
            'helperName': '',
            'content': "No diff"
        }]
    sorted_keys = sorted(diff.iterkeys())
    head = [{"key": key, "title": "{} ({})".format(key, len(diff[key]))} for key in sorted_keys]
    diff = {key: ["<br/>".join(map(set_color, sorted(val)))] for key, val in diff.iteritems()}
    return [
        {
            'helperName': '',
            'content': {
                "<h3>Models changes (current vs {})<h3>".format(vs): {
                    "header": head,
                    "body": diff,
                },
            },
        }
    ]


def set_color(s):
    return "<span style='color:{}'>{}</span>".format(
        ["blue", "red"][utils.is_production(s)], s
    )


def compare_archives(archiver_path, old_archive, new_archive):
    """
    Посчитать какие файлы были изменены в архиве

    :param archiver_path: путь к архиватору
    :param old_archive: путь к первому архиву
    :param new_archive: путь ко второму архиву
    :return: словарь с именами моделей
    {
        'added': добавленные,
        'deleted': удаленные,
        'changed': измененные,
        'same': нетронутые,
    }
    """

    logging.info('Compare archives %s vs %s using %s', old_archive, new_archive, archiver_path)
    old_content = archiver.get_md5(archiver_path, old_archive) if old_archive else {}
    logging.info('Old archive contains %d models', len(old_content))
    new_content = archiver.get_md5(archiver_path, new_archive) if new_archive else {}
    logging.info('New archive contains %d models', len(new_content))
    return utils.compare_dicts(old_content, new_content)


def parse_args():
    args = argparse.ArgumentParser(description='Compare two models.archive files')
    args.add_argument(
        "archive_1",
        type=str,
        help="path to archive #1",
    )
    args.add_argument(
        "archive_2",
        type=str,
        help="path to archive #2",
    )
    args.add_argument(
        "-a",
        "--archiver",
        type=str,
        default="archiver",
        help="path to archiver executable",
    )
    args.add_argument(
        "-p",
        "--pretty",
        action='store_true',
        help="print json with in readable format",
    )
    args.add_argument(
        "-o",
        "--output-file",
        type=str,
        help="write results to this file",
    )
    return args.parse_args()


def main():
    args = parse_args()

    result = compare_archives(args.archiver, args.archive_1, args.archive_2)

    if args.output_file:
        with open(args.output_file, "w") as f:
            if args.pretty:
                json.dump(result, f, indent=4, sort_keys=True)
            else:
                json.dump(result, f)
    else:
        if args.pretty:
            json.dump(result, sys.stdout, indent=4, sort_keys=True)
        else:
            json.dump(result, sys.stdout)


if __name__ == "__main__":
    main()
