# -*- coding: utf-8 -*-
import logging
import os
import time
import urllib

from sandbox.common.share import skynet_get
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia

from . import utils


def svn(models_dir, svn_url, revision=None):
    """
    Загрузить файлы из svn

    :param models_dir: путь до директории, куда сохранять файлы
    :param svn_url: URL нужной директории в svn
    :param revision: ревизия svn
    :return: список загруженых файлов и директорий
    """

    logging.info('Get files from svn: "%s", revision %s.', svn_url, revision)
    Arcadia.export(svn_url, models_dir, revision=revision)
    return [os.path.join(models_dir, f) for f in os.listdir(models_dir)]


def sky(models_dir, torrent):
    """
    Загрузить файлы через skynet

    :param models_dir: путь до директории, куда сохранять файлы
    :param torrent: rbtorrent ссылка
    :return: cписок путей к файлам
    """

    if not torrent:
        return []

    if not torrent.startswith('rbtorrent:'):
        raise SandboxTaskFailureError("Invalid torrent: {}".format(torrent))

    if not os.path.exists(models_dir):
        os.makedirs(models_dir)

    logging.info('Downloading models from torrent "%s"', torrent)
    skynet_get(torrent, models_dir)

    return [os.path.join(models_dir, f) for f in os.listdir(models_dir)]


def download_urls(models_dir, urls):
    """
    Загрузить файлы по URL'ам.

    :param models_dir: путь до директории, куда сохранять файлы
    :param urls: список пар (file_name, URL)
    :return: список путей к файлам
    """

    if not urls:
        return []

    logging.info('Downloading models to directory "%s"', models_dir)
    if not os.path.exists(models_dir):
        os.makedirs(models_dir)

    logging.info('Links: "%s"', urls)
    return [download_url(models_dir, name, url) for name, url in urls]


def download_url(models_dir, name, url):
    path = os.path.join(models_dir, name)
    models_subdir = os.path.dirname(path)
    if models_subdir and not os.path.exists(models_subdir):
        os.makedirs(models_subdir)
    for attempt in range(4):
        try:
            _, response = urllib.urlretrieve(url, path)
            if response.maintype == "text":
                with open(path) as f:
                    content = f.read()
                if content.startswith("HTTP ERROR"):
                    raise SandboxTaskFailureError("HTTP ERROR {}:\n{}".format(url, content))
            return path
        except IOError as e:
            sleep_time = 2 ** attempt
            logging.debug("I/O error while retrieving url: %s\n%s\nWill sleep for %s sec", url, e, sleep_time)
            time.sleep(sleep_time)

    raise SandboxTaskFailureError("Unable to download model {} from url: {}".format(path, url))


def parse_fml_urls(text):
    """
    Извлечь из текста список пар file_name URL
    Формат описания:
        файл1.info : url1
        файл2.info : url2

    :param text: текст с описанием
    :return: список пар (file_name, URL)
    """

    if not text:
        return []

    logging.info('Parsing beta models urls.')

    def parse_line(line):
        parts = tuple(x.strip() for x in line.split(":", 1))
        if len(parts) != 2:
            raise utils.ModelsError("Can't parse model url line: {}".format(line))
        name, url = parts
        return name, url

    lines = (line.strip() for line in text.splitlines())
    return [parse_line(line) for line in lines if line]
