import datetime as dt
import traceback
import json
import logging
import os

from sandbox.projects.common import requests_wrapper


EXPERIMENTS_API_URL = 'http://ab.yandex-team.ru/api/expstorage/production/testids?form=full'
QUEUED_EXPERIMENTS_API_URL = (
    'http://ab.yandex-team.ru/api/testid/'
    '?task__state=INBOX&task__state=QUEUED&task__state=IN_CONFIG&form=full&task__queue_id=1'
)
CRON_DOWNLOADS_API_URL = 'http://metrics.yandex-team.ru/services/api/cron/serp/list/?regional=RU&evaluation=WEB'
DO_NOT_REMOVE = {
    "basename_match": {
        # opt SEARCH-3526#1495123860000
        "ByOpt.json",
        "ByTouchOpt.json",
        "KzOpt.json",
        "KzTouchOpt.json",
        "RuOpt.json",
        "RuTouchOpt.json",
        "TrOpt.json",
        "TrTouchOpt.json",
        "UaOpt.json",
        "UaTouchOpt.json",
        "ByOpt.info",
        "ByTouchOpt.info",
        "KzOpt.info",
        "KzTouchOpt.info",
        "RuOpt.info",
        "RuTouchOpt.info",
        "TrOpt.info",
        "TrTouchOpt.info",
        "UaOpt.info",
        "UaTouchOpt.info",
        # filterbanned SEARCH-4861
        "serp_data_texts.txt"
    },
    "relpath_contains": {
        "rearrange/lua",
        "rearrange/baden",
    }
}


def get_currently_used_models():
    experiments = (
        set(experiments_from_api(EXPERIMENTS_API_URL)) |
        set(experiments_from_api(QUEUED_EXPERIMENTS_API_URL)) |
        set(cron_downloads_from_api(CRON_DOWNLOADS_API_URL))
    )
    logging.info('Experiments from API %d: %s', len(experiments), ', '.join(sorted(experiments)))
    return experiments


def parse_experiments_from_rearr(value):
    parts = value.split('=', 1)
    if len(parts) > 1:
        key, value = parts
        if key == 'fon':
            for exp in value.replace('@', '|').split('|'):
                logging.info('fon: %s', exp)
                yield exp
        elif key == "scheme_Local/AuxiliaryFml/Fml":
            try:
                markers = json.loads(value)
                for marker in markers:
                    name = marker["Name"]
                    exp = name.split(":")[1]
                    logging.debug('AuxiliaryFml: %s', exp)
                    yield exp
            except Exception:
                logging.info("Unable to parse AuxiliaryFml: %s", traceback.format_exc())
        elif key == 'scheme_Local/Tdi4All/SelfCheck':
            fmls = value.find('Fmls:')
            left = value.find('{', fmls)
            right = value.find('}', left)
            if left > 0 and right > 0:
                formulas = value[left + 1:right].split(',')
                for f in formulas:
                    exp = f.split(':', 1)[0][1:-1]
                    logging.info('SelfCheck: %s', exp)
                    yield exp


def parse_experiments(node):
    if isinstance(node, list):
        for value in node:
            for exp in parse_experiments(value):
                yield exp
    elif isinstance(node, str):
        if '=' in node:
            node_split = node.split('=', 1)
            for exp in parse_experiments({node_split[0]: [node_split[1]]}):
                yield exp
    elif isinstance(node, dict):
        for key, values in node.iteritems():
            if key == 'pron':
                for value in values:
                    for v in value.split(";"):
                        if v.startswith('exp_'):
                            exp = v[4:]
                            logging.info('pron: %s', exp)
                            yield exp
                        elif v.endswith('exp'):
                            exp = v[:-3]
                            logging.info('pron: %s', exp)
                            yield exp
            elif key == 'rearr':
                for value in values:
                    for v in value.split(";"):
                        for exp in parse_experiments_from_rearr(v):
                            yield exp
            else:
                for exp in parse_experiments(values):
                    yield exp


def experiments_from_api(api_url):
    result = requests_wrapper.get_r(api_url).json()
    params = [x['params'] for x in result if 'params' in x]
    for p in params:
        logging.info('Params from API: %s', p)
        try:
            parsed = json.loads(p)
        except ValueError:
            parsed = None
        for exp in parse_experiments(parsed):
            yield exp
            if exp.endswith("wopol"):
                yield exp[:-len("wopol")]


def cron_downloads_from_api(api_url):
    result = requests_wrapper.get_r(api_url).json()
    download_params = [x['downloadParams'] for x in result if 'downloadParams' in x]
    for dw_param in download_params:
        for param in dw_param:
            if param['key'] == "custom-param":
                parsed = str(param['value'].encode('utf-8').strip()).split('&')
                if parsed:
                    for exp in parse_experiments(parsed):
                        yield exp


def experiments_from_name(model):
    parts = os.path.basename(model).split('_', 2)
    return parts[1].split('.') if len(parts) > 1 else []


def model_is_experiment(model, experiments):
    return any(e in experiments for e in experiments_from_name(model))


def model_is_older_than(model, days, today):
    fail = False
    model_date_str = model.props.get("date", None)
    if model_date_str is None:
        logging.error("Unable to get model date in props: %s", model.props)
        fail = True
    model_date = get_time_from_str(model_date_str)
    if model_date is None:
        logging.error("Unable to get model from str: %s", model_date_str)
        fail = True
    if fail:
        logging.warn("Failed to check oldness for model: %s", model.name)
        return False
    return today - model_date >= dt.timedelta(days=days)


def get_time_from_str(date_str):
    for template in ["%Y-%m-%d-%H-%M", "%d %b %Y %H:%M:%S", "%Y-%m-%dT%H:%M:%S.%fZ"]:
        try:
            return dt.datetime.strptime(date_str, template)
        except Exception:
            pass


def model_in_do_not_remove(model_path):
    if os.path.basename(model_path) in DO_NOT_REMOVE["basename_match"]:
        return True
    if any(s in os.path.relpath(model_path) for s in DO_NOT_REMOVE["relpath_contains"]):
        return True
    return False
