# -*- coding: utf-8 -*-
import itertools
import logging
import os
import re

from . import mxops
from . import utils


def is_matrixnet(path):
    """
        Проверить, что по данному пути лежит .info файл
        :param path: полный путь к существующему .info файлу
    """

    return os.path.isfile(path) and path.endswith(".info")


def models_from_dir(path, recursive=True):
    """
    Перечислить все .info модели в директории

    :param path: путь до директории
    :param recursive: обходить ли поддиректории рекурсивно
    :return: список путей к файлам моделей
    """

    if is_matrixnet(path):
        return [path]
    elif os.path.isdir(path):
        logging.info('Get models from dir: %s', path)
        return [f for f in utils.walk_files(path, recursive) if is_matrixnet(f)]
    else:
        return []


def parse_streams(names, stream_re=re.compile('^[A-Z][a-z]*')):
    """
    Определить имена потоков по именам моделей

    :param names: имена моделей
    :param stream_re: регулярнов выражение для определения имени потока
    :return: итератор пар (имя, поток)
    """
    for name in names:
        match = stream_re.search(name)
        if match:
            stream = match.group(0)
            yield name, stream


def parse_experiments(names):
    """
    Определить имена потоков и экспериментов по именам моделей

    :param names: имена моделей
    :return: итератор троек (имя модели, поток, эксперимент)
    """

    for name in names:
        parts = name.split('_')
        count = len(parts)
        if count == 3:
            stream, exp, _ = parts
            for e in exp.split('.'):
                yield name, stream, e
        elif count > 1:
            raise utils.ModelsError("Experiment model should contain exactly 2 underscores ('_'): {}".format(name))


def get_models_id(mxops_path, models, short_path=False, recursive=True):
    """
    Получить formula-id для всех .info файлов

    :param mxops_path: путь к mx_ops
    :param models: пути к файлам и директориям
    :return: итератор (file-name,formula-id)
    """

    all_models = itertools.chain.from_iterable(models_from_dir(m, recursive=recursive) for m in models)
    return (
        (path if not short_path else path.rsplit("/", 1)[-1], mxops.get_id(mxops_path, path))
        for path in all_models
    )


def calc_id(path):
    """
    Вычислить formula-id используя md5 файла

    :param path: путь к файлу
    :return: formula-id
    """

    md5 = utils.md5_for_file(path)
    return "md5-{}".format(md5)


def ensure_models_id(mxops_path, models):
    """
    Убедиться, что все .info файлы содержат поле formula-id.
    Если этого поля нет, то записать туда md5 файла.

    :param mxops_path: путь к mx_ops
    :param models: пути к файлам и директориям
    :return:
    """

    logging.info('Checking formula-id')
    without_id = [path for (path, formula_id) in get_models_id(mxops_path, models) if not formula_id]
    if without_id:
        names = sorted(os.path.basename(path) for path in without_id)
        logging.info('Models without formula-id: %s', names)
        for path in without_id:
            formula_id = calc_id(path)
            mxops.set_prop(mxops_path, path, "formula-id", formula_id)


def ensure_model_slices(mxops_path, models, slice_name):
    """
    Проверить свойство slices. Если не указано, записать '<slice_name>[<min_factor>;<max_factor + 1>)'.
    """

    logging.info('Checking model Slices')
    all_models = itertools.chain.from_iterable(models_from_dir(m) for m in models)
    without_slices = []
    for path in all_models:
        info = mxops.get_info(mxops_path, path)
        props = mxops.info_props(info)
        if 'Slices' not in props:
            without_slices.append(os.path.basename(path))
            used_factors = info['Used factors'].split(',')
            if len(used_factors) > 0:
                min_factor = int(used_factors[0])
                max_factor = int(used_factors[-1])
                slices = '{}[{};{})'.format(slice_name, min_factor, max_factor + 1)
                mxops.set_prop(mxops_path, path, 'Slices', slices)
    logging.info('Models without Slices: %s', sorted(without_slices))
