# -*- coding: utf-8 -*-
from __future__ import print_function

import argparse
import logging
import subprocess
import traceback

from . import utils


def parse_info(info):
    """
    Разобрать формат вывода mx_ops info

    :param info: вывод mx_ops info
    :return: итератор пар (имя поля, значение)
    """

    lines = info.splitlines()
    parent = None
    for line in lines:
        parts = line.strip().split(':', 1)
        if len(parts) > 1:
            key = parts[0].strip()
            value = parts[1].strip()
            if parent is not None and line.startswith('\t'):
                yield parent, (key, value)
            elif value:
                parent = None
                yield key, value
            else:
                parent = key


def get_info(mxops_path, path):
    """
    Получить вывод mx_ops info

    :param mxops_path: путь к mx_ops
    :param path: путь к .info файлу
    :return: словарь {имя поля: значение}
    """

    cmd = [mxops_path, "info", path]
    logging.debug('CMD: %s', ' '.join(cmd))
    try:
        proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        output, err = proc.communicate()
        retcode = proc.poll()
        if retcode:
            logging.error("Error:\n%s", err)
            raise utils.ModelsError('Invalid info file: {}'.format(path))
    except Exception:
        logging.error("Traceback:\n%s", traceback.format_exc())
        raise
    info = {}
    for key, value in parse_info(output):
        if isinstance(value, tuple):
            info.setdefault(key, {})[value[0]] = value[1]
        else:
            info[key] = value
    return info


def get_prop(mxops_path, path, name):
    """
    Получить вывод mx_ops getprop

    :param mxops_path: путь к mx_ops
    :param path: путь к .info файлу
    :param name: имя свойства
    :return: вывод
    """

    cmd = [mxops_path, "getprop", path, name]
    logging.debug('CMD: %s', ' '.join(cmd))
    try:
        proc = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        output, err = proc.communicate()
        retcode = proc.poll()
        if retcode:
            logging.error("Error mx_ops getprop {} {}:\n{}".format(path, name, str(err)))
            return ''
    except Exception:
        logging.error("Traceback:\n%s", traceback.format_exc())
        raise
    return output.strip()


def info_props(info):
    return info.get("Model props", {}) if info else {}


def get_props(mxops_path, path):
    """
    Получить метаданные .info файла

    :param mxops_path: путь к mx_ops
    :param path: путь к .info файлу
    :return: словарь {имя поля: значение}
    """

    return info_props(get_info(mxops_path, path))


def get_id(mxops_path, path):
    return get_prop(mxops_path, path, "formula-id")


def set_prop(mxops_path, path, name, value):
    """
    Задать поле в метаданных .info файла

    :param mxops_path: путь к mx_ops
    :param path: путь к .info файлу
    :param name: имя поля
    :param value: новое значение поля
    :return:
    """

    cmd = [mxops_path, "setprop", path, name, value]
    logging.info('%s', ' '.join(cmd))
    subprocess.check_output(cmd)


def parse_args():
    args = argparse.ArgumentParser(description='Get formula info')
    args.add_argument("-m", "--mxops", type=str, default="mx_ops", help="path to mx_ops executable")
    args.add_argument("-i", "--info", type=str, help="path to .info file")
    return args.parse_args()


def main():
    args = parse_args()
    logging.basicConfig(
        format='%(levelname)s\t%(message)s',
        level=logging.INFO
    )
    print(get_info(args.mxops, args.info))


if __name__ == "__main__":
    main()
