#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import json
import logging
import multiprocessing
import os
import time

from . import archiver
from . import bundles
from . import check
from . import matrixnet
from . import mxops
from . import slices
from . import utils


def mxops_info_worker(args):
    mxops_path, path = args
    return path, mxops.get_info(mxops_path, path)


def get_formula_id_worker(args):
    mxops_path, path = args
    return path, mxops.get_id(mxops_path, path)


def calc_formula_id_worker(args):
    mxops_path, path = args
    mxops.set_prop(mxops_path, path, "formula-id", matrixnet.calc_id(path))


def annotate_matrixnet_worker(args):
    mxops_path, path, info = args
    mxops.set_prop(mxops_path, path, "Slices", slices.calc_slices(info, path))


def save_meta(path, additional):
    meta = {
        'time': time.strftime('%Y-%m-%d %H:%M:%S'),
    }
    if additional:
        meta.update(additional)
    logging.info('Meta information: %s', meta)
    with open(path, 'w') as f:
        json.dump(meta, f)


def prepare(
    mxops_path,
    models_dir,
    production=True,
    check_l3=True,
    check_id=True,
    check_slices=True,
    streams=None,
    meta=None,
    fail_without_slices=False,
):
    """
    Prepare models before creating models.archive

    :param mxops_path: путь к mx_ops
    :param models_dir: директория с моделями
    :param production: проверять продакшн модели на наличие Fast/Hast
    :param check_l3: проверять L3 модели
    :param check_id: требовать наличия formula-id у всех моделей
    :param check_slices: требовать наличия Slices у моделей ранжирования и у click моделей
    :param streams: список потоков, которые должны обязательно присутствовать в архиве
    :param meta: дополнительная мета информация об архиве с моделями
    :return:
    """

    top_files = list(utils.walk_files(models_dir, recursive=False))
    logging.debug("Top files: %s", top_files)
    top_models = [path for path in top_files if matrixnet.is_matrixnet(path)]
    top_bundles = [path for path in top_files if bundles.is_bundle_config(path)]
    if production:
        check.check_production(top_models, top_bundles, l3=check_l3, streams=streams)
    check.check_experiment(top_models, top_bundles)

    pool = multiprocessing.Pool()
    all_models = matrixnet.models_from_dir(models_dir, recursive=True)
    infos = dict(pool.map(mxops_info_worker, ((mxops_path, path) for path in all_models)))

    ids = dict(pool.map(get_formula_id_worker, ((mxops_path, path) for path in all_models)))

    if check_id:
        bad_ranking_models = [os.path.basename(path) for path in top_models if not ids.get(path)]
        if bad_ranking_models:
            raise utils.ModelsError('Ranking models without formula-id: {}'.format(bad_ranking_models))

    without_id = [path for path in all_models if not ids.get(path)]
    if without_id:
        pool.map(calc_formula_id_worker, ((mxops_path, path) for path in without_id))

    if check_slices:
        logging.info("Check Slices")
        click_dir = os.path.join(models_dir, "click")
        annotate_models = top_models + matrixnet.models_from_dir(click_dir, recursive=True)
        without_slices = [path for path in annotate_models if not mxops.info_props(infos.get(path)).get("Slices")]
        if without_slices:
            if fail_without_slices:
                raise utils.ModelsError('Ranking models without factor slices: {}'.format(without_slices))
            else:
                logging.info("Models without Slices: %s", without_slices)
                pool.map(annotate_matrixnet_worker, ((mxops_path, path, infos.get(path)) for path in without_slices))

    production_streams = utils.streams(top_models) | utils.streams(top_bundles)
    bundles.check_bundles(top_files, ids={ids.get(path) for path in top_models}, production_streams=production_streams)

    save_meta(os.path.join(models_dir, 'meta'), additional=meta)


def parse_args():
    args = argparse.ArgumentParser(description='Prepare models for creating models.archive')
    args.add_argument(
        "directory",
        type=str,
        help="path to directory with models",
    )
    args.add_argument(
        "-o",
        "--output-file",
        type=str,
        help="write models.archive results to this file",
    )
    args.add_argument(
        "-a",
        "--archiver",
        type=str,
        default="archiver",
        help="path to archiver executable",
    )
    args.add_argument(
        "-m",
        "--mxops",
        type=str,
        default="mx_ops",
        help="path to mx_ops executable",
    )
    args.add_argument(
        "-v",
        "--verbose",
        action='store_true',
        help="print logs",
    )
    return args.parse_args()


def main():
    args = parse_args()
    logging.basicConfig(
        format='%(levelname)s\t%(message)s',
        level=logging.INFO if args.verbose else logging.WARNING,
    )
    prepare(args.mxops, args.directory)
    if args.output_file:
        archiver.create(args.archiver, args.output_file, False, args.directory)


if __name__ == "__main__":
    main()
