#!/usr/bin/env python
# -*- coding: utf-8 -*-

import argparse
import itertools
import logging
import os

from . import matrixnet
from . import mxops
from . import utils


def parse_strelkovsky(borders):
    if not borders:
        return
    slices = borders.split(';')
    slice_from = 0
    for s in slices:
        parts = s.strip().split(':')
        slice_name = parts[0]
        slice_to = int(parts[1])
        yield (slice_name, slice_from, slice_to)
        slice_from = slice_to


def calc_slices(info, name):
    """
    Вычислить необходимое значение поля Slices

    :param info: результат вызова функции mxops_info
    :param name: имя файла с моделью
    :return: необходимое значение поля Slices
    """

    logging.info("Calc Slices for: %s", name)
    props = mxops.info_props(info)
    borders = list(parse_strelkovsky(props.get('borders')))
    if borders:
        logging.info("Borders: %s", str(borders))
        return ' '.join('%s[%i;%i)' % b for b in borders)
    used_factors = info.get('Used factors')
    after_last_f = int(used_factors.split(',')[-1]) + 1 if used_factors else 0
    name = os.path.basename(name)
    if 'Fresh' in name and 'ExtRelev' in name:
        logging.info("Fresh: %i", after_last_f)
        return 'fresh[0;100) web_production[100;%i)' % after_last_f
    else:
        logging.info("WebProduction: %i", after_last_f)
        return 'web_production[0;%i)' % after_last_f


def annotate_matrixnet(mxops_path, path, dry_run=False):
    """
    Записать поле Slices если оно отсутствует

    :param mxops_path: путь к mx_ops
    :param path: путь к .info файлу
    :param dry_run: только проверить, но не записывать поле
    :return:
    """

    info = mxops.get_info(mxops_path, path)
    slices = mxops.info_props(info).get('Slices')
    if slices is None:
        slices = calc_slices(info, path)
        if not dry_run:
            try:
                mxops.set_prop(mxops_path, path, 'Slices', slices)
            except Exception as e:
                raise utils.ModelsError(e)


def annotate_matrixnets(mxops_path, paths, dry_run=False):
    """
    Записать поле Slices всем моделям, у которых оно отсутствует

    :param mxops_path: путь к mx_ops
    :param paths: пути к моделям
    :param dry_run: только проверить, но не записывать поле
    :return:
    """

    failed = []
    for path in paths:
        try:
            annotate_matrixnet(mxops_path, path, dry_run=dry_run)
        except utils.ModelsError:
            failed.append(path)
    if failed:
        raise utils.ModelsError('Failed to annotate: %s' % ', '.join(failed))


def annotate_dir(mxops_path, path, dry_run=False, recursive=False):
    """
    Записать поле Slices всем моделям, у которых оно отсутствует

    :param mxops_path: путь к mx_ops
    :param path: путь к директории с моделями
    :param dry_run: только проверить, но не записывать поле
    :param recursive: спускаться в поддиректории
    :return:
    """

    annotate_matrixnets(
        mxops_path,
        matrixnet.models_from_dir(path, recursive=recursive),
        dry_run=dry_run,
    )


def annotate_dirs(mxops_path, paths, dry_run=False, recursive=False):
    """
    Записать поле Slices всем моделям, у которых оно отсутствует

    :param mxops_path: путь к mx_ops
    :param paths: пути к директориям с моделями
    :param dry_run: только проверить, но не записывать поле
    :param recursive: спускаться в поддиректории
    :return:
    """

    annotate_matrixnets(
        mxops_path,
        itertools.chain.from_iterable(matrixnet.models_from_dir(path, recursive=recursive) for path in paths),
        dry_run=dry_run,
    )


def parse_args():
    args = argparse.ArgumentParser(description='Annotate matrixnet with slices')
    args.add_argument(
        "path",
        type=str,
        nargs="+",
        help="path to matrixnet file or directory",
    )
    args.add_argument(
        "-m",
        "--mxops",
        type=str,
        default="mx_ops",
        help="path to mx_ops executable",
    )
    args.add_argument(
        "--dry-run",
        action='store_true',
        help="only check slices (do not use 'mx_ops setprop')",
    )
    args.add_argument(
        "-v",
        "--verbose",
        action='store_true',
        help="print logs",
    )
    args.add_argument(
        "-r",
        "--recursive",
        action='store_true',
        help="walk directories recursively",
    )
    return args.parse_args()


def main():
    args = parse_args()
    logging.basicConfig(
        format='%(levelname)s\t%(message)s',
        level=logging.INFO if args.verbose else logging.WARNING,
    )
    annotate_dirs(
        args.mxops,
        args.path,
        dry_run=args.dry_run,
        recursive=args.recursive,
    )


if __name__ == "__main__":
    main()
