# -*- coding: utf-8 -*-
import filecmp
import hashlib
import logging
import os
import re


class ModelsError(Exception):
    pass


def walk_files(directory, recursive=True):
    """
    Перечислить все файлы в директории
    :param directory: путь к директории
    :param recursive: обходить ли поддиректории рекурсивно
    :return: итератор по всем файлам
    """

    if recursive:
        for root, dirnames, filenames in os.walk(directory):
            for f in filenames:
                yield os.path.join(root, f)
    else:
        for name in os.listdir(directory):
            f = os.path.join(directory, name)
            if os.path.isfile(f):
                yield f


def md5_for_file(file_path, block_size=2 ** 20):
    """
    Вычислить md5 сумму файла
    :param file_path: путь к файлу
    :param block_size: размер блока для вычисления суммы
    :return: значение md5 суммы
    """

    with open(file_path, "rb") as f:
        md5 = hashlib.md5()
        data = f.read(block_size)
        while data:
            md5.update(data)
            data = f.read(block_size)
        return md5.hexdigest()


def conflict_files(dir_a, dir_b):
    """
    Найти файлы с одинаковым именем, но разным содержимым (директории всегда считаются различными)

    :param paths: пути к файлам и директориям
    :return: список относительных путей конфликтующих файлов и директорий
    """

    dir_a = os.path.normpath(dir_a)
    dir_b = os.path.normpath(dir_b)

    conflicts = set()
    for a_fpath in walk_files(dir_a):
        rel_dir_path, fname_a = os.path.split(os.path.relpath(a_fpath, dir_a))
        b_fpath = os.path.join(dir_b, rel_dir_path, fname_a)
        b_fpath_json = b_fpath.replace(".info", ".json") if fname_a.endswith(".info") else None
        if os.path.exists(b_fpath):
            if not filecmp.cmp(a_fpath, b_fpath) and md5_for_file(a_fpath) != md5_for_file(b_fpath):
                conflicts.add(os.path.join(rel_dir_path, fname_a))
        elif b_fpath_json and os.path.exists(b_fpath_json):  # RMDEV-574
            conflicts.add(os.path.join(rel_dir_path, fname_a.replace(".info", ".json")))

    return sorted(conflicts)


def compare_dicts(old_content, new_content):
    """
    Вычислить разницу между двумя наборами данных.

    :param old_content: словарь старых данных {имя: id}
    :param new_content: словарь новых данных {имя: id}
    :return: словарь с именами
    {
        'added': добавленные,
        'deleted': удаленные,
        'changed': измененные,
        'same': нетронутые,
    }
    """

    new_names = set(new_content.keys())
    old_names = set(old_content.keys())

    added = sorted(new_names - old_names)
    deleted = sorted(old_names - new_names)
    common = sorted(old_names & new_names)

    changed = [x for x in common if new_content[x] != old_content[x]]
    same = [x for x in common if new_content[x] == old_content[x]]

    logging.info("Added: %d", len(added))
    logging.info("Deleted: %d", len(deleted))
    logging.info("Changed: %d", len(changed))
    logging.info("Same: %d", len(same))

    return {
        'added': added,
        'deleted': deleted,
        'changed': changed,
        'same': same,
    }


def file_name(path):
    full_name = os.path.basename(path)
    name, ext = os.path.splitext(full_name)
    return name


def is_experiment(path):
    name = os.path.basename(path)
    name, _ = os.path.splitext(name)
    if '_' in name:
        return name.count('_') == 2
    name = name.lower()
    return name.endswith("exp") or name.endswith("experiment")


def is_production(path):
    return not is_experiment(path)


def name_to_streams(name):
    """
    by https://arc.yandex-team.ru/wsvn/arc/trunk/arcadia/search/rank/sum_relev/init_rank_models.cpp?op=blame&rev=2379187&peg=2379187#l5
    """
    STREAM_SPLIT_PATTERN = '[A-Z][^A-Z]*'
    parts = name.split('_')
    subs = re.findall(STREAM_SPLIT_PATTERN, parts[0])
    stream_name = '.'.join(sub.lower() for sub in subs)

    streams = []
    if len(parts) == 1:
        streams.append('full.' + stream_name)
    elif len(parts) == 2 or len(parts) == 3 and not parts[2]:
        for part in parts[1].split('.'):
            streams.append('full.{0}:{1}:'.format(stream_name, part))
    else:
        raise ModelsError('Model {} has unknown format'.format(name))

    return streams


def production_names(models):
    return {file_name(path) for path in models if is_production(path)}


def experiment_names(models):
    return {file_name(path) for path in models if is_experiment(path)}


def streams(models):
    all_streams = set()
    for path in models:
        all_streams.update(name_to_streams(file_name(path)))
    return all_streams


def is_base_model(path):
    """
        Проверить, что по данному пути файл с моделью для базового поиска
        :param path: путь к файлу (не обязательно существующий)
    """
    stream = os.path.basename(path).split('_', 1)[0]
    # TODO(tsimkha) remove hacks for dssm after SEARCH-4210
    result = (
        "Fast" in stream or
        "mappings" in stream or
        "webmodels" in stream or
        "Hast" in stream or
        "L1" in stream or
        stream.startswith("Geo") or
        path.startswith("base/") or
        (path.endswith('.dssm') and not path.startswith('reg_chain') and not path.startswith("mmeta")) or
        path.startswith('panther/')
    )
    logging.debug("is_base: %s = %s", path, 'True' if result else 'False')
    return result


def is_middle_model(path):
    """
        Проверить, что по данному пути файл с моделью для среднего поиска
        :param path: путь к файлу (не обязательно существующий)
    """
    return not (path.startswith("base/") or path in {
        "optimized_model.dssm",
        "aggregated_ann_reg_query.dssm",
        "context_model.reformulations.dssm",
        "neuro_text_dssm_long_clicks_predictor_feb19_v3_compress.dssm",
        "neuro_text_no_title_dssm_long_clicks_predictor_mar19_v1_compress.dssm",
        "relev_sents.reformulations.dssm",
    })


def is_web_middle_model(path):
    """
        Проверить, что по данному пути файл с моделью для среднего поиска
        :param path: путь к файлу (не обязательно существующий)
    """
    stream = os.path.basename(path).split('_', 1)[0]
    return not (
        path.startswith("base/")
        or path in {
            "optimized_model.dssm",
            "aggregated_ann_reg_query.dssm",
            "context_model.reformulations.dssm",
            "neuro_text_dssm_long_clicks_predictor_feb19_v3_compress.dssm",
            "neuro_text_no_title_dssm_long_clicks_predictor_mar19_v1_compress.dssm",
            "relev_sents.reformulations.dssm",
        }
        or "Fast" in stream
        or ("L1" in stream and "Dynamic" not in stream)
    )
