# -*- coding: utf-8 -*-

import glob
import itertools
import logging
import os
import platform
import sys

from sandbox import common

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import util
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import SandboxEnvironment, TarballToolkitBase
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import which

from sandbox.projects import resource_types


YANDEX_PYPI = "https://pypi.yandex-team.ru/simple/"


class SandboxMapReduceEnvironment(SandboxEnvironment):
    """
        Архив с mapreduce утилитами (по задаче SANDBOX-759)

        Список утилит http://wiki.yandex-team.ru/mapreduce/CmdLineUtils
    """
    resource_type = 'MAP_REDUCE_UTILS_ARCHIVE'

    name = 'mapreduce_utils'

    sys_path_utils = [
        'mr_cat',
        'mr_data_manip',
        'mr_du',
        'mr_get_keys',
        'mr_head',
        'mr_ls',
        'mr_rm',
        'mr_set_ops',
        'mr_touch',
        'mr_wc',
        'mr_cp',
        'mr_diff',
        'mr_find',
        'mr_grep',
        'mr_hist',
        'mr_mv',
        'mr_sample',
        'mr_sort',
        'mr_uniq',
    ]

    @staticmethod
    def get_last_released_version(resource_type):
        """
            Получить версию последнего зарелиженного ресурса.
            Проверяется атрибут `version` у ресурса из последнего релиза для типа `resource_type`
            Если у типа ресурса нет релиза в системе или нет атрибута `version`, генерируется исключение.

            :return: версия в виде строки.
        """
        arch = util.system_info()['arch']
        stable_releases = channel.sandbox.list_releases(str(resource_type), arch=arch, limit=1)
        resources = list(itertools.chain.from_iterable(
            stable_release.resources for stable_release in stable_releases))
        last_released_resource = resources[0] if resources else None
        if last_released_resource is None:
            raise common.errors.SandboxEnvironmentError(
                "Cannot find the last release for '{}' resource type".format(resource_type)
            )
        if 'version' in last_released_resource.attributes:
            return last_released_resource.attributes['version']
        else:
            raise common.errors.SandboxEnvironmentError(
                "Cannot get a version for resource '{}'.".format(last_released_resource.id)
            )

    def prepare(self):
        if self.version is None:
            logging.info('Get default version for mapreduce_utils environment   :')
            self.version = self.get_last_released_version(self.resource_type)
        logging.info('Prepare mapreduce environment, version: {0}, plaform {1}'.format(self.version,
                                                                                       platform.platform()))
        mr_utils_tar = self.get_environment_resource()
        logging.info('Mapreduce environment resource {0} for version {1}'.format(self.resource_id, self.version))
        mr_utils_folder = self.get_environment_folder()
        if os.path.exists(mr_utils_folder):
            logging.info('Environment folder already exists. {0}'.format(mr_utils_folder))
        else:
            self.extract_tar(mr_utils_tar, mr_utils_folder)
        os.environ['PATH'] = mr_utils_folder + ':' + os.environ['PATH']
        self.check_environment()
        return self.resource_id


class SandboxYcssjsEnvironment(SandboxEnvironment):
    """
        Утилиты ycssjs для сборки репорта
    """

    resource_type = 'YCSSJS_ENVIRONMENT'

    name = 'ycssjs'

    sys_path_utils = ['ycssjs', ]

    def prepare(self):
        logging.info('Prepare xsltproc and ycssjs environment, version %s', self.version)
        env_tar = self.get_environment_resource()
        env_folder = self.get_environment_folder()
        if os.path.exists(env_folder):
            logging.info('Environment folder already exists. %s' % env_folder)
        else:
            self.extract_tar(env_tar, env_folder)
        os.environ['PATH'] = os.path.join(env_folder, 'bin') + ':' + os.environ['PATH']
        os.environ['YUI_JAR'] = os.path.join(env_folder, 'bin', 'yuicompressor.jar')
        os.environ['PERL5LIB'] = os.path.join(env_folder, 'perl') + ':' + os.environ.get('PERL5LIB', '')


class SandboxJavaEnvironment(SandboxEnvironment):
    """
        Окружение JRE (не включает в себя JDK)
    """
    resource_type = 'JAVA_ENVIRONMENT'

    name = 'java'

    sys_path_utils = ['java', ]

    def prepare(self):
        logging.info('Prepare java environment, version %s' % self.version)
        java_tar = self.get_environment_resource()
        java_folder = self.get_environment_folder()
        if os.path.exists(java_folder):
            logging.info('Environment folder already exists. %s' % java_folder)
        else:
            self.extract_tar(java_tar, java_folder)
        os.environ['PATH'] = os.path.join(java_folder, 'bin') + ':' + os.environ['PATH']
        os.environ['LD_LIBRARY_PATH'] = ":".join(filter(None, [
            os.path.join(java_folder, 'lib'), os.environ.get('LD_LIBRARY_PATH')
        ]))


class DepotToolsEnvironment(SandboxEnvironment):
    """
        Окружение для depot_tools
        https://commondatastorage.googleapis.com/chrome-infra-docs/flat/depot_tools/docs/html/depot_tools.html
    """
    resource_type = 'GIT_DEPOT_TOOLS_ENVIRONMENT'

    name = 'depot_tools'

    def prepare(self):
        logging.info('Prepare depot_tools environment, version %r', self.version)
        depot_tools_tar = self.get_environment_resource()
        env_folder = self.get_environment_folder()
        if os.path.exists(env_folder):
            logging.info('Environment folder %r already exists.', env_folder)
        else:
            self.extract_tar(depot_tools_tar, env_folder)
        depot_tools_folder = os.path.join(env_folder, 'depot_tools-%s' % self.version)
        os.environ['PATH'] = depot_tools_folder + ':' + os.environ['PATH']


class SandboxJavaJdkEnvironment(SandboxEnvironment):
    """
        Окружение JRE + JDK (Oracle JDK)
    """
    resource_type = 'JAVA_JDK_ENVIRONMENT'

    name = 'jdk'

    sys_path_utils = ['javac', ]

    def prepare(self):
        logging.info('Prepare jdk environment, version %s' % self.version)
        java_tar = self.get_environment_resource()
        java_folder = self.get_environment_folder()
        if os.path.exists(java_folder):
            logging.info('Environment folder already exists. %s' % java_folder)
        else:
            self.extract_tar(java_tar, java_folder)
        os.environ['PATH'] = os.path.join(java_folder, 'bin') + ':' + os.environ['PATH']
        os.environ['JAVA_HOME'] = os.path.join(java_folder)
        os.environ['LD_LIBRARY_PATH'] = ":".join(filter(None, [
            os.path.join(java_folder, 'lib'), os.environ.get('LD_LIBRARY_PATH')
        ]))


class SandboxAntEnvironment(SandboxEnvironment):
    """
        Окружение ANT
    """
    resource_type = 'ANT_ENVIRONMENT'

    name = 'ant'

    sys_path_utils = ['ant', ]

    def prepare(self):
        logging.info('Prepare Ant environment, version %s' % self.version)
        res_tar = self.get_environment_resource()
        env_folder = self.get_environment_folder()
        if os.path.exists(env_folder):
            logging.info('Environment folder already exists. %s' % env_folder)
        else:
            self.extract_tar(res_tar, env_folder)
        path_to_pkg = os.path.join(env_folder, 'apache-ant-%s' % self.version)
        os.environ['PATH'] = os.path.join(path_to_pkg, 'bin') + ':' + os.environ['PATH']
        os.environ['ANT_HOME'] = path_to_pkg


class SandboxMavenEnvironment(SandboxEnvironment):
    """
        Окружение Maven
    """
    resource_type = 'MAVEN_ENVIRONMENT'

    name = 'mvn'

    sys_path_utils = ['mvn', ]

    def prepare(self):
        logging.info('Prepare Maven environment, version %s' % self.version)
        res_tar = self.get_environment_resource()
        env_folder = self.get_environment_folder()
        if os.path.exists(env_folder):
            logging.info('Environment folder already exists. %s' % env_folder)
        else:
            self.extract_tar(res_tar, env_folder)
        path_to_pkg = os.path.join(env_folder, 'apache-maven-%s' % self.version)
        os.environ['PATH'] = os.path.join(path_to_pkg, 'bin') + ':' + os.environ['PATH']
        os.environ['MAVEN_OPTS'] = '-Dmaven.repo.local=' + os.path.join(SandboxEnvironment.build_cache_dir, '.m2') + \
                                   ' ' + os.environ.get('MAVEN_OPTS', '')


class SandboxGitEnvironment(SandboxEnvironment):
    """
        Окружение Git

        Если вы используете собранный из исходников git, может понадобится скрипт для
        модификации оберток bin-wrappers/*, так как в стандартных используются абсолютные пути
        от машины, где собирался git.

        https://github.yandex-team.ru/gist/unlok/d9bc1a22141faf72ebb6
    """
    resource_type = 'GIT_ENVIRONMENT'

    name = 'git'

    sys_path_utils = ['git', ]

    def prepare(self):
        logging.info('Prepare Git environment, version %s' % self.version)
        res_tar = self.get_environment_resource()
        env_folder = self.get_environment_folder()
        if os.path.exists(env_folder):
            logging.info('Environment folder already exists. %s' % env_folder)
        else:
            self.extract_tar(res_tar, env_folder)
        git_folder = os.path.join(env_folder, 'git-%s' % self.version)
        bin_folder = os.path.join(git_folder, 'bin-wrappers')
        if not os.path.exists(bin_folder):
            logging.info('Compiling Git binaries')
            log_prefix = 'install-git-%s' % self.version
            process.run_process(['./configure'], work_dir=git_folder, log_prefix=log_prefix, shell=True)
            process.run_process(['make'], work_dir=git_folder, log_prefix=log_prefix, shell=True)
        os.environ['PATH'] = bin_folder + ':' + os.environ['PATH']


class SandboxGitLfsEnvironment(SandboxEnvironment):
    """
        Окружение git-lfs
        Работает только с бинарником
    """
    resource_type = 'GIT_LFS_ENVIRONMENT'

    name = 'git-lfs'

    sys_path_utils = ['git-lfs']

    def prepare(self):
        logging.info('Prepare git-lfs environment, version %s' % self.version)
        res_tar = self.get_environment_resource()
        env_folder = self.get_environment_folder()

        if os.path.exists(env_folder):
            logging.info('Environment folder already exists. %s' % env_folder)
        else:
            self.extract_tar(res_tar, env_folder)

        name = 'git-lfs-%s' % self.version
        git_lfs_folder = os.path.join(env_folder, name)
        os.environ['PATH'] = git_lfs_folder + ':' + os.environ['PATH']
        logging.info('Running git lfs install')
        process.run_process(['git', 'lfs', 'install'], work_dir=git_lfs_folder, log_prefix=name, shell=True)


class SandboxRubyEnvironment(SandboxEnvironment):
    """
        Окружение ruby
    """
    resource_type = 'RUBY_ENVIRONMENT'

    name = 'ruby'

    sys_path_utils = ['ruby', ]

    def prepare(self):
        logging.info('Prepare ruby environment, version %s' % self.version)
        env_tar = self.get_environment_resource()
        env_folder = self.get_environment_folder()
        if os.path.exists(env_folder):
            logging.info('Environment folder already exists. %s' % env_folder)
        else:
            self.extract_tar(env_tar, env_folder)
        os.environ['PATH'] = os.path.join(env_folder, 'bin') + ':' + os.environ['PATH']
        os.environ['LD_LIBRARY_PATH'] = ":".join(filter(None, [
            os.path.join(env_folder, 'lib'), os.environ.get('LD_LIBRARY_PATH')
        ]))


class SandboxNodeNpmEnvironment(SandboxEnvironment):
    """
        Окружение с nodejs и npm
        $NODE_PATH устанавливается в папку с npm, потому нужно проверять, правильные ли версии npm-пакетов установлены
    """
    resource_type = 'NODE_AND_NPM_ENVIRONMENT'

    name = 'node_and_npm'

    sys_path_utils = ['node', 'npm', ]

    def prepare(self):
        logging.info('Prepare node and npm environment, version %s' % self.version)
        node_tar = self.get_environment_resource()
        node_folder = self.get_environment_folder()
        if os.path.exists(node_folder):
            logging.info('Environment folder already exists. %s' % node_folder)
        else:
            self.extract_tar(node_tar, node_folder)
        global_node_modules_path = os.path.join(node_folder, 'lib', 'node_modules', )
        make_folder(global_node_modules_path)
        os.environ['PATH'] = os.path.join(node_folder, 'bin') + ':' + os.environ['PATH']
        os.environ['LD_LIBRARY_PATH'] = ":".join(filter(None, [
            os.path.join(node_folder, 'lib'), os.environ.get('LD_LIBRARY_PATH')
        ]))
        os.environ['NODE_PATH'] = global_node_modules_path


class ValgrindEnvironment(SandboxEnvironment):
    """
        Valgrind environment.
    """

    resource_type = 'VALGRIND'

    name = 'valgrind'

    sys_path_utils = ['valgrind', ]

    def __init__(self, version='3.9.0'):
        super(ValgrindEnvironment, self).__init__(version)
        self.valgrind_folder = None

    def prepare(self):
        logging.info('Preparing valgrind environment ...')
        logging.info('Using valgrind-%s' % self.version)
        valgrind_tar = self.get_environment_resource()
        self.valgrind_folder = self.get_environment_folder()
        if os.path.exists(self.valgrind_folder):
            logging.info('Environment folder already exists: %s' % self.valgrind_folder)
        else:
            self.extract_tar(valgrind_tar, self.valgrind_folder)
        self.valgrind_folder = os.path.join(self.valgrind_folder, 'valgrind')
        logging.info('valgrind_folder: %s' % self.valgrind_folder)
        os.environ['VALGRIND_LIB'] = os.path.join(self.valgrind_folder, 'lib', 'valgrind')
        os.environ['PATH'] = os.path.join(self.valgrind_folder, 'bin') + ':' + os.environ['PATH']

        self.check_environment()

        valgrind_bin = which('valgrind')
        if valgrind_bin.find(self.valgrind_folder) == -1:
            raise common.errors.SandboxEnvironmentError(
                "valgrind path '{}' is incorrect. valgrind must be found inside folder '{}'".format(
                    valgrind_bin, self.valgrind_folder
                )
            )

    def get_include_dir(self):
        if self.valgrind_folder:
            return os.path.join(self.valgrind_folder, 'include')
        return None


class PipVirtualenvEnvironment(PipEnvironment):
    """
        virtualenv.
        https://pypi.python.org/pypi/virtualenv/
    """

    name = "virtualenv"

    def __init__(self):
        PipEnvironment.__init__(self, package_name="virtualenv", index_url=YANDEX_PYPI)

    def check_environment(self):
        try:
            import virtualenv  # noqa
        except ImportError:
            return False
        return True


class BaseLibraryEnvironment(SandboxEnvironment):
    """
        Окружение из заранее собранных библиотек
    """

    resource_type = None
    name = None
    library_version = None
    header_name = None

    def __init__(self, version=None):
        super(BaseLibraryEnvironment, self).__init__(version or self.library_version)

    def prepare(self):
        logging.info('Prepare %s environment with %s version', self.name, self.version)
        library_tar = self.get_environment_resource()
        library_folder = self.get_environment_folder()
        if os.path.exists(library_folder):
            logging.info('Environment folder already exists: %s', library_folder)
        else:
            self.extract_tar(library_tar, library_folder)
        library_folder = os.path.join(library_folder, self.name)
        if 'LIBRARY_PATH' in os.environ:
            os.environ['LIBRARY_PATH'] = '%s:%s' % (os.path.join(library_folder, 'lib'), os.environ['LIBRARY_PATH'])
        else:
            os.environ['LIBRARY_PATH'] = os.path.join(library_folder, 'lib')
        if 'CPATH' in os.environ:
            os.environ['CPATH'] = '%s:%s' % (os.path.join(library_folder, 'include'), os.environ['CPATH'])
        else:
            os.environ['CPATH'] = os.path.join(library_folder, 'include')
        self.check_environment()

    def check_environment(self):
        library_folder = self.get_environment_folder()
        check_file_name = os.path.join(library_folder, self.name, 'include', self.header_name)
        if not os.path.isfile(check_file_name):
            raise common.errors.SandboxEnvironmentError(
                'Cannot find "%s" header in %s directory.', check_file_name, library_folder
            )
        else:
            logging.info('Found "%s" file. Okay.', check_file_name)
        return True


class SandboxLibSnappyEnvironment(BaseLibraryEnvironment):
    """
        Заранее собранная библиотека Snappy
    """

    resource_type = 'SNAPPY_LIBRARY'
    name = 'libsnappy'
    library_version = '1.0.4'
    header_name = 'snappy.h'

    def prepare(self):
        super(SandboxLibSnappyEnvironment, self).prepare()
        library_folder = self.get_environment_folder()
        os.environ['SNAPPY_DIR'] = os.path.join(library_folder, self.name)
        os.environ['SNAPPY_HEADERS'] = os.path.join(library_folder, self.name, 'include')
        os.environ['SNAPPY_LIBRARY'] = os.path.join(library_folder, self.name, 'lib', 'libsnappy.a')


class SandboxLibFFIEnvironment(BaseLibraryEnvironment):
    """
        Заранее собранная библиотека libffi
    """

    resource_type = 'FFI_LIBRARY'
    name = 'libffi'
    library_version = '3.99999'
    header_name = 'ffi.h'

    def prepare(self):
        super(SandboxLibFFIEnvironment, self).prepare()
        library_folder = self.get_environment_folder()
        os.environ['LIBFFI_DIR'] = os.path.join(library_folder, self.name)
        os.environ['LIBFFI_HEADERS'] = os.path.join(library_folder, self.name, 'include')
        os.environ['LIBFFI_LIBRARY'] = os.path.join(library_folder, self.name, 'lib', 'libffi.a')


class SandboxLibMysqlClientEnvironment(BaseLibraryEnvironment):
    """
        Заранее собранная библиотека mysqlclient
    """

    resource_type = 'MYSQLCLIENT_LIBRARY'
    name = 'libmysqlclient'
    library_version = '6.1.6'
    header_name = 'mysql.h'

    def prepare(self):
        super(SandboxLibMysqlClientEnvironment, self).prepare()
        library_folder = self.get_environment_folder()
        os.environ['PATH'] = '%s:%s' % (
            os.path.join(self.get_environment_folder(), self.name, 'bin'), os.environ['PATH'])
        os.environ['MYSQLCLIENT_DIR'] = os.path.join(library_folder, self.name)
        os.environ['MYSQLCLIENT_HEADERS'] = os.path.join(library_folder, self.name, 'include')
        os.environ['MYSQLCLIENT_LIBRARY'] = os.path.join(library_folder, self.name, 'lib', 'libmysqlclient.a')


class SandboxLibOpensslEnvironment(BaseLibraryEnvironment):
    """
        Заранее собранная библиотека openssl
    """

    resource_type = 'OPENSSL_LIBRARY'
    name = 'libopenssl'
    library_version = '1.0.2f'
    header_name = 'openssl/ssl.h'

    def prepare(self):
        super(SandboxLibOpensslEnvironment, self).prepare()
        library_folder = self.get_environment_folder()
        openssl_dir = os.environ['OPENSSL_DIR'] = os.path.join(library_folder, self.name)
        os.environ['OPENSSL_HEADERS'] = os.path.join(library_folder, self.name, 'include')
        os.environ['OPENSSL_LIBRARY'] = os.path.join(library_folder, self.name, 'lib', 'liblibs-newssl-openssl.a')

        pkgconfig_parts = (
            'prefix={0}'.format(openssl_dir),
            'exec_prefix=${prefix}',
            'libdir=${exec_prefix}/lib',
            'includedir=${prefix}/include',
            '',
            'Name: openssl',
            'URL: http://www.openssl.org/',
            'Description: OpenSSL',
            'Version: {0}'.format(self.library_version),
            'Libs: -L${libdir} -llibs-newssl-openssl',
            'Cflags: -I${includedir}',
        )
        pkgconfig_dir = os.path.join(openssl_dir, 'lib', 'pkgconfig')
        if not os.path.exists(pkgconfig_dir):
            os.makedirs(pkgconfig_dir)
        with open(os.path.join(pkgconfig_dir, 'openssl.pc'), 'w') as fd:
            fd.write('\n'.join(pkgconfig_parts))


class SandboxPythonDevEnvironment(SandboxEnvironment):
    """
        Заголовочные файлы python
    """

    resource_type = 'PYTHON_DEV'

    name = 'python-dev'

    STABLE = '2.7.3'

    def __init__(self, version=None):
        super(SandboxPythonDevEnvironment, self).__init__(version or self.STABLE)

    def prepare(self):
        logging.info('Prepare %s environment with %s version' % (self.name, self.version))
        pythondev_tar = self.get_environment_resource()
        pythondev_folder = self.get_environment_folder()
        if os.path.exists(pythondev_folder):
            logging.info('Environment folder already exists: %s', pythondev_folder)
        else:
            self.extract_tar(pythondev_tar, pythondev_folder)
        pythondev_folder = os.path.join(pythondev_folder, self.name)
        if 'CPATH' in os.environ:
            os.environ['CPATH'] = os.path.join(pythondev_folder, 'include') + ':' + os.environ['CPATH']
        else:
            os.environ['CPATH'] = os.path.join(pythondev_folder, 'include')
        self.check_environment()

    def check_environment(self):
        pythondev_folder = self.get_environment_folder()
        check_file_name = os.path.join(pythondev_folder, self.name, 'include', 'Python.h')
        if not os.path.isfile(check_file_name):
            raise common.errors.SandboxEnvironmentError(
                'Cannot find "%s" header in %s directory.' % (check_file_name, pythondev_folder))
        else:
            logging.info('Found "%s" file. Okay.' % check_file_name)
        return True


class MongodbEnvironment(TarballToolkitBase):
    name = "mongodb"

    resource_type = "MONGODB_UTILS"

    def prepare(self):
        logging.info("Prepare mongo utils environment")
        env_dir = super(MongodbEnvironment, self).prepare()
        default_env_dir = True
        for path in os.listdir(env_dir):
            full_path = os.path.join(env_dir, path)
            if os.path.isdir(full_path) and (path == "bin" or os.path.isdir(os.path.join(full_path, "bin"))):
                if path != "bin":
                    full_path = os.path.join(full_path, "bin")
                os.environ["PATH"] = "{}:{}".format(full_path, os.environ["PATH"])  # Add each found 'bin' to PATH
                default_env_dir = False
        if default_env_dir:
            os.environ['PATH'] = "{}:{}".format(env_dir, os.environ["PATH"])
        return env_dir


class PortablePyPyEnvironment(SandboxEnvironment):
    name = 'portable_pypy'

    resource_type = 'PORTABLE_PYPY'

    sys_path_utils = []

    def prepare(self):
        pypy_folder = self.get_environment_folder()
        if os.path.exists(pypy_folder):
            logging.info('Environment folder already exists. %s' % pypy_folder)
        else:
            logging.info('Preparing PyPy, version %s', self.version)
            pypy_archive = self.get_environment_resource()
            self.extract_tar(pypy_archive, pypy_folder)
            args = ['mv']
            args.extend(glob.glob(os.path.join(pypy_folder, 'pypy-*')))
            args.append(os.path.join(pypy_folder, 'pypy'))
            process.run_process(args, shell=True)
        os.environ['PORTABLE_PYPY_DIR'] = os.path.join(pypy_folder, 'pypy')


class TankEnvironment(SandboxEnvironment):
    """
    Environment for Yandex.Tank.
    Detailed information about Yandex.Tank tool
    you can find at http://yandextank.readthedocs.org/en/latest/index.html

    Deprecated. Use common.search.tank instead.
    """

    class TankVersion(common.utils.Enum):
        STABLE = "stable"
        TESTING = "testing"
        UNSTABLE = "unstable"

    class LoadScheduleSelector(object):
        """Class represents helper for selecting Yandex.Tank RPS schedule"""

        class TankLoadScheme(common.utils.Enum):
            STEP = "step"
            LINE = "line"
            CONST = "const"

        def __init__(self, load_scheme, *args):
            """
            :param load_scheme: scheme of load. See cte.TankLoadScheme enum for available values
            :param args: appropriate args for selected `load_scheme` parameter
            """
            if load_scheme not in self.TankLoadScheme:
                raise errors.SandboxTaskUnknownError(
                    "Incorrect `load_scheme` param. Expected one of {!r}".format(
                        list(self.TankLoadScheme)
                    )
                )
            self.load_scheme = load_scheme
            self.args = map(lambda x: str(x).strip(), args)

        def __str__(self):
            """String representation to pass it to `rps_schedule` parameter of TankEnvironment"""
            return "{load_scheme}({args})".format(load_scheme=self.load_scheme, args=",".join(self.args))

    STABLE_VERSION = TankVersion.STABLE
    resource_type = str(resource_types.YANDEX_TANK)
    PLUGINS = {
        "common": {
            "plugin_rcheck": "Tank/Plugins/ResourceCheck.py",
            "plugin_shellexec": "Tank/Plugins/ShellExec.py",
            "plugin_aggreg": "Tank/Plugins/Aggregator.py",
            "plugin_autostop": "Tank/Plugins/Autostop.py",
            "plugin_monitoring": "Tank/Plugins/Monitoring.py",
            "plugin_console": "Tank/Plugins/ConsoleOnline.py",
            "plugin_web": "",
            "plugin_tips": "",
            "plugin_totalautostop": "Tank/Plugins/TotalAutostop.py",
            "plugin_loadosophia": "",
            "plugin_graphite": "",
            "plugin_rcassert": "Tank/Plugins/RCAssert.py",
            "plugin_report": "Tank/Plugins/Report.py",
        },
        "lunapark": {
            "plugin_uploader": "Tank/Plugins/DataUploader.py",
        },
        "phantom": {
            "plugin_phantom": "Tank/Plugins/Phantom.py",
        },
        "dolbilo": {
            "plugin_dolbilo": "Tank/Plugins/Dolbilo.py",
            "plugin_phantom": "",
        }
    }
    NEW_PLUGINS = {
        "common": {
            "plugin_rcheck": "",
            "plugin_shellexec": "yandextank.plugins.ShellExec",
            "plugin_aggreg": "yandextank.plugins.Aggregator",
            "plugin_autostop": "yandextank.plugins.Autostop",
            "plugin_monitoring": "yandextank.plugins.Monitoring",
            "plugin_console": "yandextank.plugins.ConsoleOnline",
            "plugin_web": "",
            "plugin_tips": "",
            "plugin_totalautostop": "yandextank.plugins.TotalAutostop",
            "plugin_loadosophia": "",
            "plugin_graphite": "",
            "plugin_rcassert": "yandextank.plugins.RCAssert",
        },
        "lunapark": {
            "plugin_uploader": "yandextank.plugins.DataUploader",
        },
        "phantom": {
            "plugin_phantom": "yandextank.plugins.Phantom",
        },
        "dolbilo": {
            "plugin_dolbilo": "yandextank.plugins.Dolbilo",
            "plugin_phantom": "",
        }
    }
    _CGROUP_FILE_PATH = "/sys/fs/cgroup/cpuset/tank_cpu_limit/tasks"
    _CGROUP_CMD = "cpuset:tank_cpu_limit"
    _BOOTSTRAP_SCRIPT_NAME = "tank.py"
    _BOOTSTRAP_NEW_SCRIPT_NAME = "ignitor.py"
    _log = lambda self, msg: logging.debug("[%s] %s", TankEnvironment.__name__, msg)

    def __init__(
            self,
            work_dir,
            target_host, target_port,
            rps_schedule,
            task_name, operator,
            version=None, additional_params=None, headers=None, uris=None, ammo_path=None, writelog="all",
            dolbilka_environment=None,
            use_new_scheme=False,
    ):
        """
        :param version: version of YandexTank resource
        :param work_dir: path to work dir. *** Do not forget to create TASK_LOGS resource with this folder
                         to save load testing results. ***
        :param target_host: hostname of target machine to fire
        :param target_port: port of target machine to fire
        :param rps_schedule: string with tank RPS schedule.
                             You can use LoadScheduleSelector helper to generate it.
        :param task_name: Open JIRA or Startrek `load testing` task name
        :param operator: user login to get notifications from Lunapark
        :param additional_params: list of strings with additional params
        :param headers: list of strings with requests headers in format `name: value`
        :param uris: requests uris. Will ignore this if `ammo-path` param is passed
        :param ammo_path: filepath to requests
        :param dolbilka_environment: object of DolbilkaEnvironment class to use dolbilka. If this object passed
                                     all phantom parameters will be ignored.
        :param use_new_scheme: Use new plugins and tank
        """
        super(TankEnvironment, self).__init__(version or self.STABLE_VERSION)
        self.target_host = target_host
        self.target_port = target_port
        self.headers = headers or []
        self.uris = uris or []
        self.ammo_path = ammo_path
        self.writelog = writelog
        self.task_name = task_name
        self.operator = operator
        self.rps_schedule = str(rps_schedule)
        self.params = additional_params or []
        self._src_path = None
        self._work_dir = work_dir
        self._dolbilka_environment = dolbilka_environment
        self.use_new_scheme = use_new_scheme
        if self.use_new_scheme:
            self.resource_type = str(resource_types.NEW_YANDEX_TANK)

    def _write_plugins(self, plugin_type, file_handler):
        plugins = self.PLUGINS
        if self.use_new_scheme:
            plugins = self.NEW_PLUGINS
        for key, value in plugins[plugin_type].iteritems():
            file_handler.write("{name}={path}\n".format(name=key, path=value))

    def _generate_load_config(self, file_name="load.ini", job_name=None):
        file_path = os.path.join(self._work_dir, file_name)
        with open(file_path, "w") as fh:
            fh.write("[tank]\n")
            self._write_plugins("common", fh)
            if self.task_name:
                self._write_plugins("lunapark", fh)
            if self._dolbilka_environment is not None:
                if self._dolbilka_environment.d_executor_path is None:
                    self._dolbilka_environment.prepare()
                self._log("Dolbilo params detected. Starting to configure dolbilo")
                self._write_plugins("dolbilo", fh)
                fh.write("[dolbilo]\nexecutor_path={}\n".format(self._dolbilka_environment.d_executor_path))
                fh.write("replace-host={}\n".format(self.target_host))
                fh.write("replace-port={}\n".format(self.target_port))
                for key, value in self._dolbilka_environment.d_args.iteritems():
                    fh.write("{}={}\n".format(key.replace("_", "-"), value))
                fh.write("[meta]\n")
            else:
                self._log("Starting to configure phantom")
                self._write_plugins("phantom", fh)
                fh.write(
                    "[phantom]\nrps_schedule={schedule}\nwritelog={writelog}\n"
                    "headers = [Host: {host}]\n  [Connection: close]\n".format(
                        schedule=self.rps_schedule,
                        host=self.target_host,
                        writelog=self.writelog
                    )
                )
                for header in self.headers:
                    fh.write("  {}\n".format(header))
                if self.uris and not self.ammo_path:
                    fh.write("uris = /\n")
                    for uri in self.uris:
                        fh.write("  {}\n".format(uri.strip()))
                fh.write(
                    "address={host}:{port}\nphantom_path={phantom_path}\nphantom_modules_path={modules_path}\n"
                    "[meta]\nignore_target_lock=1\n".format(
                        host=self.target_host,
                        port=self.target_port,
                        phantom_path=os.path.join(self._src_path, "./usr/bin/phantom"),
                        modules_path=os.path.join(self._src_path, "./usr/lib/phantom"),
                    )
                )
            fh.write(
                "task={task}\noperator={operator}\napi_address=https://lunapark.yandex-team.ru/\n".format(
                    task=self.task_name,
                    operator=self.operator
                )
            )
            if job_name is not None:
                fh.write("job_name={}\n".format(job_name))

        self._log("`load.ini` file successfully generated and saved at {}".format(file_path))
        return file_path

    def prepare(self):
        archive_path = self.get_environment_resource()
        path = os.path.join(self._work_dir, str(self.compatible_resource.path).split(".tar.gz")[0])
        common.fs.untar_archive(archive_path, self._work_dir)
        self._src_path = path
        self._log("Environment prepared successfully")

    def fire(self, job_name=None, file_suffix=""):
        if not self._src_path:
            self.prepare()

        config_path = self._generate_load_config(file_name="load.ini{}".format(file_suffix), job_name=job_name)
        artifact_path = os.path.join(self._work_dir, "logs{}".format(file_suffix))

        cmd = []
        if os.path.exists(self._CGROUP_FILE_PATH):
            self._log("Fire process will be limit according to {} CGROUP file".format(self._CGROUP_FILE_PATH))
            cmd += ["cgexec -g {cgroup_file}".format(cgroup_file=self._CGROUP_CMD)]
        else:
            self._log(
                "There is no compatible CGROUP file at {}. Can not limit fire process".format(self._CGROUP_FILE_PATH)
            )
        bootstrap_script_name = self._BOOTSTRAP_SCRIPT_NAME
        if self.use_new_scheme:
            bootstrap_script_name = self._BOOTSTRAP_NEW_SCRIPT_NAME
        cmd += [
            sys.executable,
            os.path.join(self._src_path, bootstrap_script_name),
            "-c {}".format(config_path),
            "-k {}".format(self._work_dir),
            "-o 'tank.artifacts_base_dir={}'".format(artifact_path)
        ] + self.params
        if self.ammo_path:
            self._log("Going to use specified ammo file path {}".format(self.ammo_path))
            cmd.append(self.ammo_path)
        process.run_process(
            cmd,
            log_prefix='load_test_results:{}->{}{}'.format(common.config.Registry().this.fqdn, self.target_host, file_suffix),
            shell=True
        )
        return artifact_path


class SwatZookeeperEnvironment(SandboxEnvironment):
    resource_type = 'ZOOKEEPER_ARCHIVE'

    name = 'zookeeper'

    sys_path_utils = []

    def prepare(self):
        zookeeper_folder = self.get_environment_folder()
        if os.path.exists(zookeeper_folder):
            logging.info('Environment folder already exists. %s' % zookeeper_folder)
        else:
            logging.info('Preparing Zookeeper, version %s', self.version)
            zookeeper_archive = self.get_environment_resource()
            self.extract_tar(zookeeper_archive, zookeeper_folder)
            args = ['mv']
            args.extend(glob.glob(os.path.join(zookeeper_folder, 'zookeeper-*')))
            args.append(os.path.join(zookeeper_folder, 'zookeeper'))
            process.run_process(args, shell=True)
        os.environ['ZOOKEEPER_DIR'] = os.path.join(zookeeper_folder, 'zookeeper')


class SwatMongoDbEnvironment(SandboxEnvironment):
    resource_type = 'MONGODB_EXECUTABLE'

    name = 'mongodb'

    sys_path_utils = ['mongod']

    def prepare(self):
        mongo_folder = self.get_environment_folder()
        if os.path.exists(mongo_folder):
            logging.info('Environment folder already exists. %s' % mongo_folder)
        else:
            logging.info('Preparing MongoDb, version %s', self.version)
            mongo_executable = self.get_environment_resource()
            os.makedirs(mongo_folder)
            process.run_process(['cp', mongo_executable, mongo_folder], shell=True)
        os.environ['PATH'] = mongo_folder + ':' + os.environ['PATH']


class AllureCommandLineEnvironment(SandboxEnvironment):
    """
        Allure Report Command Line Tool
    """
    resource_type = 'ALLURE_COMMANDLINE'

    name = 'allure-commandline'

    sys_path_utils = ['allure', ]

    def prepare(self):
        logging.info('Prepare allure environment, version %s' % self.version)
        allure_tar = self.get_environment_resource()
        allure_folder = self.get_environment_folder()
        if os.path.exists(allure_folder):
            logging.info('Environment folder already exists. %s' % allure_folder)
        else:
            self.extract_tar(allure_tar, allure_folder)
        os.environ['PATH'] = os.path.join(allure_folder, 'bin') + ':' + os.environ['PATH']
        os.environ['ALLURE_HOME'] = os.path.join(allure_folder)
