import logging
import time
import traceback

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import string as common_string
from sandbox.projects.common import requests_wrapper


def verify(condition, message):
    """
    Check that certain code condition is True.

    Situation where condition is False means that any further execution is useless.
    False condition probably point to error in the code logic.
    Thus, developers, responsible for the code should debug the problem manually.
    See also fail(message)
    """
    if not condition:
        logging.exception("VERIFY_FAILED: %s", message)
        raise errors.TaskError(message)


def fail(message):
    """
    Stop code execution because of error in code logic.

    Write this function in places, where the program should never be in.
    See also verify(condition, message)
    """
    logging.exception("FAIL: %s", message)
    raise errors.TaskError(message)


def ensure(condition, message):
    """
    Check that certain test condition is True.

    This function checks test invariant.
    If condition is False, consider test as failed.
    Not to be confused with verify.
    """
    if not condition:
        check_failed(message)


def check_failed(message):
    """
    Consider test as failed.

    See also ensure.
    """
    logging.error("CHECK_FAILED: %s", message)
    raise errors.TaskFailure(message)


def shifted_traceback():
    """Shift traceback right for better readability in logs, etc."""
    return common_string.shift_right(traceback.format_exc())


def log_exception(message, exc, logger=None, task=None, info=False):
    """
    Kosher method for logging errors in exception handlers.

    :param message: Error message
    :param exc: Exception object
    :param logger: Optional custom logger
    :param task: Optional task object
    :param info: Post traceback into task info (disabled by default)
    """
    prepared_traceback = shifted_traceback()
    if not logger:
        logger = logging
    try:
        url = "https://coredumps.yandex-team.ru/submit_core"
        response = requests_wrapper.post(
            url,
            json={"parsed_traces": prepared_traceback, "dump_json": {}},
            params={
                "time": int(time.time()),
                "service": "sandbox",
                "ctype": "sandbox-task",
                "task_id": task.id if task else sdk2.Task.current.id,
            },
            timeout=5,
        )
        response.raise_for_status()
        logger.debug("Core sent to %s\nResponse %s", url, response.text)
    except Exception as logging_exc:
        logger.exception("Cannot send traceback, exc %s", logging_exc)

    logger.error("%s: %s\n\n%s", message, exc, prepared_traceback)
    if info and sdk2.Task.current:
        sdk2.Task.current.set_info('EXCEPTION: {}\n{}'.format(message, prepared_traceback))
