import httplib
import select


def http_default_creator(host, port):
    res = httplib.HTTPConnection(host, port=port)
    res.auto_open = True
    return res


def https_default_creator(host, port):
    res = httplib.HTTPSConnection(host, port=port)
    res.auto_open = True
    return res


HTTP_PREFIX = "http://"
HTTPS_PREFIX = "https://"


def parse_url(url):
    scheme = HTTP_PREFIX
    host = ""
    path = ""
    if url.startswith(HTTPS_PREFIX):
        scheme = HTTPS_PREFIX
        url = url[len(HTTPS_PREFIX):]
    elif url.startswith(HTTP_PREFIX):
        url = url[len(HTTP_PREFIX):]
    else:
        raise Exception()

    host, _, path = url.partition("/")
    path = _ + path
    host, _, port = str(host).partition(":")
    if port:
        port = int(port)
    elif scheme == HTTPS_PREFIX:
        port = 443
    else:
        port = 80

    return (scheme, host, port, path)


def wait_any(responses, read_response=True, timeout=None):
    index = {}
    valid = []
    invalid = []
    for x in responses:
        invalid.append(len(index)) if x.fileno() < 0 else valid.append(x)
        index[x] = len(index)

    (r, w, x) = select.select(valid, [], [], timeout)

    if read_response:
        for resp in r:
            if not resp.is_complete():
                resp.wait()

    result = [index[_] for _ in r]
    result.extend(invalid)
    return result


class Response(object):
    def __init__(self, url, http_client, http_connection, connection_error=False):
        self.url = url
        (self._scheme, self._host, self.port, self._path) = parse_url(url)
        self.http_client = http_client
        self.http_connection = http_connection
        self.data = None
        self.connection_error = connection_error
        self.success = False

        if self.fileno() < 0:
            self.connection_error = True

    def fileno(self):
        if self.http_connection and self.http_connection.sock:
            return self.http_connection.sock.fileno()
        return -1

    def is_complete(self):
        return self.connection_error or self.success

    def is_ready(self):
        if self.is_complete():
            return True
        elif wait_any([self]):
            return True
        else:
            return False

    def wait(self):
        if self.is_complete():
            return
        try:
            response = self.http_connection.getresponse()
            self.data = response.read()
            self.success = True
            self.http_client.return_connection(self._scheme, self._host, self.http_connection)
            self.__dict__.update(response.__dict__)
        except:
            self.connection_error = True


class HttpClient(object):
    def __init__(self, http_creator=http_default_creator, https_creator=https_default_creator):
        self.http_creator = http_creator
        self.https_creator = https_creator
        self.connections_pool = {}

    def return_connection(self, scheme, host, connection):
        key = scheme + host
        connections = self.connections_pool.get(key, [])
        connections.append(connection)
        self.connections_pool[key] = connections

    def get_connection(self, scheme, host, port):
        host_pool = self.connections_pool.get(scheme + host)
        if not host_pool:
            if scheme == "http://":
                return self.http_creator(host, port)
            else:
                return self.https_creator(host, port)
        else:
            return host_pool.pop()

    def send_get(self, url):
        (scheme, host, port, path) = parse_url(url)
        con = self.get_connection(scheme, host, port)
        try:
            con.request("GET", path)
            return Response(url, self, con)
        except:
            return Response(url, self, con, True)
