import logging
import httpclient
import datetime
import hashing

# It's a fast synchronized http-queries downloader
# Known problems:
#   unclosed unused connections


class RequestBase(object):
    def process_request(self, http_client):
        pass

    def get_url(self):
        pass


class GetRequest(RequestBase):
    def __init__(self, url):
        self.url = url

    def process_request(self, http_client):
        return http_client.send_get(self.url)

    def get_url(self):
        return self.url


def get_response(http_client, request_with_id):
    (req_id, request) = request_with_id
    resp = request.process_request(http_client)
    resp.__dict__["_req_id"] = req_id
    return resp


def download_queries(requests, add_cgi="", async_size=1, retries=1, hash_validation_retries=1, post_processor=None):
    retries_table = [0] * len(requests)
    result_hashes = [(None, 0)] * len(requests)

    complete_count = 0
    problems_count = 0
    failed_count = 0
    requests_count = 0
    responses_count = 0

    last = datetime.datetime.now()

    requests = [(req_id, req) for req_id, req in enumerate(requests)]
    req_id_table = {}
    for (req_id, req) in requests:
        req_id_table[req_id] = req

    responses = []
    client = httpclient.HttpClient()
    while len(requests) + len(responses) > 0:
        while len(requests) > 0 and len(responses) < async_size:
            responses.append(get_response(client, requests.pop()))
            requests_count += 1
        complete_indexes = httpclient.wait_any(responses, read_response=False)
        requests_count -= len(responses)  # Look down: requests_count += len(responses)

        for x in complete_indexes:
            resp = responses[x]
            resp.wait()

            responses[x] = None
            resp_failed = not resp.success
            resp_fail_msg = "unk"
            req_id = resp.__dict__["_req_id"]

            if not resp_failed:
                responses_count += 1
                if resp.status == 302:
                    next_url = resp.msg.dict.get("location", None)
                    if not next_url:
                        resp_failed = True
                        resp_fail_msg = "None redirect"
                    else:
                        if next_url.startswith("/"):
                            next_url = resp._scheme + resp._host + next_url + add_cgi
                            logging.debug("Next url: {}".format(next_url))
                        req_id_table[req_id] = GetRequest(next_url)
                        responses.append(get_response(client, (req_id, req_id_table[req_id])))
                elif resp.status == 200:
                    data = resp.data
                    if not data:
                        resp_failed = True
                        resp_fail_msg = "Empty output"
                    else:
                        if post_processor:
                            (data, problem) = post_processor(data)
                            if problem:
                                resp_fail_msg = problem
                                resp_failed = True

                        if not resp_failed:
                            hash_validation_ok = hash_validation_retries <= 1
                            if not hash_validation_retries or hash_validation_retries <= 1:
                                hash_validation_ok = True
                            else:
                                output_hash = hashing.get_hash(data)
                                req_hashes = result_hashes[req_id]

                                if req_hashes[0] != output_hash:
                                    req_hashes = (output_hash, 1)
                                else:
                                    req_hashes = (output_hash, req_hashes[1] + 1)

                                if req_hashes[1] >= hash_validation_retries:
                                    hash_validation_ok = True
                                else:
                                    responses.append(get_response(client, (req_id, req_id_table[req_id])))
                                result_hashes[req_id] = req_hashes
                            if hash_validation_ok:
                                complete_count += 1
                                yield (req_id, data)
                else:
                    resp_failed = True
                    resp_fail_msg = "Result code: {}".format(resp.status)
            if resp_failed:
                logging.info("Error: {} ({} index: {})".format(resp_fail_msg, req_id, req_id_table[req_id].get_url()))

                resp_fails_count = retries_table[req_id] + 1
                retries_table[req_id] = resp_fails_count
                problems_count += 1

                if resp_fails_count > retries:
                    failed_count += 1
                    yield (req_id, None)
                else:
                    responses.append(get_response(client, (req_id, req_id_table[req_id])))

        requests_count += len(responses)  # Look up: requests_count -= len(responses)

        offset = 0
        for i in range(0, len(responses)):
            if responses[i]:
                responses[offset] = responses[i]
                offset += 1
        while offset < len(responses):
            responses.pop(-1)

        now = datetime.datetime.now()
        if (now - last).seconds >= 1:
            logging.info("complete={} problems={} fails={} reqs={} resps={}".format(
                complete_count,
                problems_count,
                failed_count,
                requests_count,
                responses_count))

            last = now
