"""
    Useful functions for files. Reduce typical boilerplate.
    No sandbox dependencies allowed
"""
import os
import fnmatch
import json
import tarfile
import io
import yaml


from sandbox.projects.common import string


def read_line_by_line(file_name):
    """
        Generator of non-empty strings without line break characters
    """
    with open(str(file_name)) as f:
        for line in f:
            line = line.strip("\n\r")
            if line:
                yield line


def read_lines(file_name):
    return list(read_line_by_line(file_name))


def read_file(file_name):
    with open(str(file_name)) as f:
        contents = f.read()
        return contents


def write_lines(file_name, lines, _mode='w'):
    """
        Write list of strings with UNIX-like line breaks.
        All other line breaks will be ignored, empty strings will be ignored.
        :return number of lines written.
    """
    counter = 0
    with open(str(file_name), _mode) as f:
        for line in lines:
            line = line.strip("\n\r")
            if line:
                f.write(line + "\n")
                counter += 1
    return counter


def append_lines(file_name, lines):
    write_lines(file_name, lines, _mode='a')


def find_files(directory, pattern):
    for root, dirs, files in os.walk(directory):
        for basename in files:
            if fnmatch.fnmatch(basename, pattern):
                filename = os.path.join(root, basename)
                yield filename


def json_dump_to_tar(archive_name, file_name, contents, compress="gz"):
    write_mode = "w:" + compress
    with tarfile.open(archive_name, write_mode) as tar:
        data = json.dumps(contents)
        tarinfo = tarfile.TarInfo(file_name)
        tarinfo.size = len(data)
        tar.addfile(tarinfo, io.BytesIO(data.encode()))


def pack_files_into_tar(archive_name, file_names, compress="gz"):
    write_mode = "w:" + compress
    with tarfile.open(archive_name, write_mode) as tar:
        for file_name in file_names:
            tar.add(file_name)


def write_file(file_name, contents):
    with open(str(file_name), 'w') as f:
        f.write(string.all_to_str(contents))


def append_file(file_name, contents):
    with open(str(file_name), 'a') as f:
        f.write(string.all_to_str(contents))


def json_load(file_name):
    with open(str(file_name)) as f:
        return json.load(f)


def json_dump(file_name, data, indent=None, sort_keys=False):
    with open(str(file_name), "w") as f:
        return json.dump(data, f, indent=indent, sort_keys=sort_keys)


def yaml_load(file_name):
    with open(str(file_name)) as f:
        return yaml.load(f)
