# -*- coding: utf-8 -*-
"""
    Place here footers used in several tasks
    Please obey PEP8 coding standards and Sandbox StyleGuide here
    https://wiki.yandex-team.ru/sandbox/codestyleguide

    New functions without documentation comments are FORBIDDEN.
"""
import six

from sandbox.projects.common.differ import coloring


def get_task_fields(client, task_id, fields):
    """
        :param client: rest.Client()
        :param task_id: id of sandbox task
        :param fields: list or string with params, divided by comma
    """
    fields_str = fields if isinstance(fields, six.string_types) else ",".join(fields)
    return client.task.read(
        id=task_id,
        fields=fields_str,
        children=True,
        hidden=True,
        limit=1,
    ).get("items", [{}])[0]


def exec_stats_footer(task):
    """
        Common footer for performance tests
    """
    if not task.is_completed():
        return None
    head = [
        {"key": "session", "title": "N session"},
        {"key": "rps", "title": "RPS"},
        {"key": "fail_rate", "title": "Fail rate"},
        {"key": "rss", "title": "Rss memory (vmtouch)"},
        {"key": "vsz", "title": "Vsz memory (vmtouch)"},
    ]
    rps_list = task.ctx.get("requests_per_sec", [0])
    max_rps = task.ctx.get("max_rps", 0)
    rps_list[rps_list.index(max_rps)] = '<b style="color:red">{}</b>'.format(max_rps)
    return [
        {
            'helperName': '',
            'content': {
                "<h4>Executor stats</h4>": {
                    "header": head,
                    "body": {
                        "session": range(task.ctx.get("dolbilka_executor_sessions", 1)),
                        "rps": rps_list,
                        "fail_rate": task.ctx.get("fail_rates", [0]),
                        "rss": task.ctx.get("memory_rss", [0]),
                        "vsz": task.ctx.get("memory_vsz", [0]),
                    }
                }
            }
        }
    ]


def memusage_footer(ids, client, custom_name=None, diff_threshold=0.5):
    """
        Common footer for memory usage tests (SEARCH-1081)
    """
    mem_types = ["rss", "vms", "uss", "pss", "anon", "shared", "vmlck"]
    content = [{"Memory type": t, "Old binary": -1, "New binary": -1, "Diff": "-"} for t in mem_types]
    mb = float(1024 * 1024)
    for n, child_id in enumerate(ids):
        if child_id:
            info = get_task_fields(client, child_id, [
                'context.memory_rss',
                'context.memory_vsz',
                'context.memory_bytes',
            ])
            try:
                mem_bytes = info.get('context.memory_bytes')
                if not mem_bytes:
                    mem_bytes = [{
                        "rss": info['context.memory_rss'][0],
                        "vms": info['context.memory_vsz'][0],
                    }]

                for i, t in enumerate(mem_types):
                    content[i]["{} binary".format(["Old", "New"][n])] = round(mem_bytes[0].get(t) / mb, 2)

            except Exception:
                continue
    for i in content:
        diff = 100 * (i["New binary"] - i["Old binary"]) / i["Old binary"] if i["Old binary"] else None
        i["Diff"] = coloring.color_diff(diff, max_diff=diff_threshold)
        i["Delta"] = round(i["New binary"] - i["Old binary"], 2)
    return {
        'helperName': '',
        'content': {custom_name or "<h3>Memory usage tests (Mb)</h3>": content},
    }
