import re
import os

from sandbox.sandboxsdk.errors import SandboxTaskFailureError


results_template = """\
<table border = "1" cellpadding = "6px" cellspacing = "0px">
    <tr>
        <th>
            Total responses
        </th>
        <th>
            GotError: YES
        </th>
        <th>
            GotError: NO
        </th>
        <th>
            TotalDocCount: 0
        </th>
        <th>
            TotalDocCount > 0
        </th>
        <th>
            Avg. Found Docs
        </th>
        </tr>
    <tr>
        <td>
            {Total}
        </td>
        <td>
            {Errors}
        </td>
        <td>
            {NoErrors}
        </td>
        <td>
            {NoDocs}
        </td>
        <td>
            {GotDocs}
        </td>
        <td>
            {AvgFound}
        </td>
</tr>
</table>"""

not_ready_template = "<h4><font color='gray'>  Stats are not ready yet </font></h4>"
error_template = "<h4><font color='red'> Unexpected state of task </font></h4>"

got_error = re.compile("GotError: (YES|NO)")
docs = re.compile("TotalDocCount: (\d+)")


class ResponsesStats(object):
    def __init__(self):
        self.Errors = None
        self.NoErrors = None
        self.NoDocs = None
        self.Total = None
        self.NotFoundRate = None
        self.AvgFound = None
        self.GotDocs = None


def get_responses_stats(resource_path):
    stats = ResponsesStats()

    if os.path.exists(resource_path):
        with open(resource_path) as responses_file:
            responses = responses_file.read()
            error_stats = got_error.findall(responses)
            stats.Errors = error_stats.count("YES")
            stats.NoErrors = error_stats.count("NO")
            doc_stats = docs.findall(responses)
            simple_found = [int(value) for n, value in enumerate(doc_stats) if not n % 3]
            stats.Total = len(simple_found)
            stats.AvgFound = float(sum(simple_found)) / stats.Total if stats.Total else 0
            stats.NoDocs = simple_found.count("0")
            stats.GotDocs = stats.Total - stats.NoDocs
            if stats.Total:
                stats.NotFoundRate = float(stats.NoDocs * 100) / stats.Total
            else:
                stats.NotFoundRate = 100
    return stats


def assert_stats_ok(stats, acceptable_not_found_rate=25, acceptable_errors_count=0):
    if stats.NotFoundRate > acceptable_not_found_rate:
        error_text = 'NotFound rate is not acceptable: %s%% > %s%%(%s Errors). See responses for extra info'
        raise SandboxTaskFailureError(error_text %
                                      (stats.NotFoundRate, acceptable_not_found_rate, stats.Errors))
    if stats.Errors > acceptable_errors_count:
        error_text = 'Errors number is not acceptable: %s > %s (%s NotFound). See responses for extra info'
        raise SandboxTaskFailureError(error_text %
                                      (stats.Errors, acceptable_errors_count, stats.NoDocs))


def get_results_table(stats):
    return results_template.format(**stats.__dict__)
