# -*- coding: utf-8 -*-

import os
import logging
import time
import shutil

from sandbox.projects import resource_types

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.parameters import ResourceSelector, SandboxBoolParameter
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.search.components import SearchComponent


def create_distributor_params():
    class DistributorParams(object):
        class EnableDistributor(SandboxBoolParameter):
            name = 'monolith_enabled_resource_id'
            description = 'Distributor enabled'
            group = "Distributor options"
            default_value = False

        class MonolithBinary(ResourceSelector):
            name = 'monolith_binary_resource_id'
            description = 'Distributor binary'
            resource_type = [resource_types.DISTRIBUTOR]
            required = False
            group = "Distributor options"

        class MonolithDump(ResourceSelector):
            name = 'monolith_dump_resource_id'
            description = 'Distributor dump'
            resource_type = [resource_types.REALSEARCH_DATA]
            required = False
            group = "Distributor options"

        class MonolithConfig(ResourceSelector):
            name = 'monolith_config_resource_id'
            description = 'Distributor config'
            resource_type = [
                resource_types.SEARCH_CONFIG,
                resource_types.MONOLITH_CONFIG,
            ]
            required = False
            group = "Distributor options"

        params = (
            EnableDistributor,
            MonolithBinary,
            MonolithConfig,
            MonolithDump,
        )

    return DistributorParams


DefaultDistributorParams = create_distributor_params()


class Monolith(SearchComponent):
    def __init__(
        self,
        binary,
        config_path,
        logs_path,
        datastore_path,
        port,
    ):
        self.binary = binary
        self.config_path = config_path
        self.logs_path = logs_path
        self.datastore_path = datastore_path
        self.base_port = port
        self.state_root = os.path.join(logs_path, "state{}".format(self.base_port))
        self.work_dir = os.path.join(logs_path, "work{}".format(self.base_port))

        logging.info("StateRoot: %s" % self.state_root)

    def get_commandline(self):
        cmd = [
            "-V", "LOG_PATH={}".format(self.logs_path),
            "-V", "DATASTORE={}".format(self.datastore_path),
            "-V", "BasePort={}".format(self.base_port),
            "-V", "StateRoot={}".format(self.state_root),
            "-V", "StateRootDir={}".format(self.state_root),
            "-V", "WorkDir={}".format(self.work_dir),
            self.config_path
        ]

        return cmd

    def start(self):
        if not os.path.exists(self.logs_path):
            os.mkdir(self.logs_path)

        if not os.path.exists(self.state_root):
            os.mkdir(self.state_root)

        self.process = process.run_process(
            [self.binary] + self.get_commandline(),
            wait=False,
            log_prefix='run_monolith',
            shell=False
        )
        time.sleep(10)
        process.check_process_return_code(self.process)

    def stop(self):
        if self.process and self.process.poll() is None:
            try:
                self.kill_softly()
            except Exception as e:
                logging.exception("Failed to stop distributor gracefully: %s", e)
                self.process.kill()


def create_distributor(
    params=DefaultDistributorParams,
    port=20000,
):
    ctx = channel.task.ctx

    binary_path = channel.task.sync_resource(ctx[params.MonolithBinary.name])
    source_datastore_path = channel.task.sync_resource(ctx[params.MonolithDump.name])
    config_path = channel.task.sync_resource(ctx[params.MonolithConfig.name])
    logs_path = channel.task.log_path("monolith_logs_{}".format(port))

    datastore_path = channel.task.abs_path("db_{}".format(port))
    shutil.copytree(source_datastore_path, datastore_path)
    paths.chmod(datastore_path, 0o755)

    logging.info("Creating distributor port %d" % port)

    return Monolith(
        binary_path,
        config_path,
        logs_path,
        datastore_path,
        port,
    )
