# -*- coding: utf-8 -*-

import os
import time
import logging
from shutil import copyfile, rmtree
from sandbox.projects.common.fusion.tools import run_rtyserver_command


class FusionRunner(object):
    """
        This class controls execution of Fusion instances for extended periods of time. Used to build shards of specified size.
    """

    class ShardBuilderParams(object):
        def __init__(self, max_time=86400):
            self.time_limit = max_time
            self.extra_doc_percent = 0.1
            self.extra_doc_percent_stage2 = 1.1
            self.max_extra_doc = 25000

    def __init__(self, fusion, use_own_dist, use_ext_dist, shard_id, max_docs, builder_params):
        self.fusion = fusion
        self.use_own_dist = use_own_dist
        self.use_ext_dist = use_ext_dist
        self.shard_id = shard_id
        self.max_docs = max_docs
        self.has_indexdir = True
        self.builder_params = builder_params

    def __enter__(self):
        return self

    def __exit__(self):
        if self.has_indexdir:
            self.clear_db()

    #
    # Controller methods
    #
    def pause_docfetcher(self):
        run_rtyserver_command("localhost", self.fusion.controller_port, "pause_docfetcher")

    def continue_docfetcher(self):
        run_rtyserver_command("localhost", self.fusion.controller_port, "continue_docfetcher")

    def force_enable_search(self):
        run_rtyserver_command("localhost", self.fusion.controller_port, "reopen_indexes")
        run_rtyserver_command("localhost", self.fusion.controller_port, "enable_search")

    #
    # Helper routines
    #
    REFRESH_IS_READY_MSG = "Refresh is ready now"

    def detach_db(self):
        db_path = self.fusion.index_dir
        self.resolve_symlinks(db_path)
        self.has_indexdir = False
        return db_path

    def clear_db(self):
        db_path = self.fusion.index_dir
        rmtree(db_path, True)
        self.has_indexdir = False

    @staticmethod
    def resolve_symlinks(path):
        for dirpath, dirnames, filenames in os.walk(path):
            for filename in filenames:
                fpath = os.path.join(dirpath, filename)
                if os.path.islink(fpath):
                    realfpath = os.path.realpath(fpath)
                    os.remove(fpath)
                    copyfile(realfpath, fpath)

    @staticmethod
    def fusion_wait(fusion, is_ready, timeout, interval):
        if timeout < 0:
            return False

        start = time.time()
        while time.time() - start < timeout:
            if fusion.process.poll() is not None:
                fusion._wait_coredump()
                fusion._process_post_mortem()
            if is_ready():
                return True
            else:
                time.sleep(interval)
        return False

    @staticmethod
    def _calc_target_doccount(cur_count, ideal_count, extra_percent, max_extra_doc):
        delta = ideal_count - cur_count
        extra_doc = int(delta * extra_percent)
        extra_doc = min(extra_doc, max_extra_doc)
        delta += extra_doc
        if delta > 5000:
            delta -= 5000  # docfetcher queue

        return cur_count + delta

    @staticmethod
    def _fusion_fetched(fusion):
        try:
            result = fusion.get_info_server()['result']
            fetched = (int(result['docs_in_final_indexes']) +
                       int(result['docs_in_disk_indexers']) +
                       int(result['docs_in_memory_indexes']))
            return fetched
        except Exception:
            pass
        return 0

    @staticmethod
    def _fusion_merged(fusion):
        try:
            num = 0
            indexes = fusion.get_info_server()['result']['indexes']
            for index in indexes.values():
                index_size = index.get("count", 0)
                if index_size > 0:
                    num += 1
            return num <= 1
        except Exception:
            pass
        return False

    #
    # Main routine
    #
    def run(self, timeout):
        logging.info("[%r] Refresh is started, now will wait until the search is filled", self.shard_id)

        fusion = self.fusion
        if self.use_own_dist:
            fusion.wait(is_ready=fusion.is_memorysearch_filled)
        elif self.use_ext_dist:
            bp = self.builder_params

            start = time.time()
            end = start + bp.max_time
            extra_doc_percent = bp.extra_doc_percent

            cur_count = 0
            ideal_count = self.max_docs
            while end > time.time():
                trg_count = self._calc_target_doccount(cur_count, ideal_count, extra_doc_percent, bp.max_extra_doc)
                if not self.fusion_wait(fusion, fusion.is_fetching_complete(trg_count), end - time.time(), 3):
                    logging.error('[%r] Failed to fetch the required amount - timed out', self.shard_id)
                    break

                self.pause_docfetcher()
                logging.info("[%r] Disabling docfetcher, waiting for merge", self.shard_id)
                self.fusion_wait(fusion, (lambda s=self, f=fusion: s._fusion_merged(f)), end - time.time(), 10)

                cur_count = self._fusion_fetched(fusion)

                if ideal_count > cur_count:
                    logging.info("[%r] Enabling docfetcher", self.shard_id)
                    self.continue_docfetcher()
                    if ideal_count - cur_count < 100000:
                        extra_doc_percent = bp.extra_doc_percent_stage2
                    continue

                logging.info("[%r] Refresh has fetched the required amount of documents, waiting for enable search", self.shard_id)
                if not self.fusion_wait(fusion, fusion.is_indexing_complete(ideal_count), 60, 10):
                    self.force_enable_search()
                    if not self.fusion_wait(fusion, fusion.is_indexing_complete(ideal_count), min(600, max(0, end - time.time())), 10):
                        logging.error("[%r] Not enough searchable documents - something is wrong, resuming fetch", self.shard_id)
                        self.continue_docfetcher()
                        continue

                logging.info("[%r] Refresh got the required amount of searchable documents: %r", self.shard_id, cur_count)
                break

            self.pause_docfetcher()
            timeout = max(0, timeout - (time.time() - start))

        logging.info(FusionRunner.REFRESH_IS_READY_MSG)

        try:
            self.wait_exit(timeout)
        finally:
            fusion._stop_impl()

    def wait_exit(self, timeout):
        # Wait until the instance is stopped; kill when timed out.
        start = time.time()
        while time.time() - start <= timeout:
            if self.fusion.process.returncode is not None:
                logging.info("[%r] Refresh was stopped with exitcode %s", self.shard_id, self.fusion.process.returncode)
                return
            else:
                logging.info("[%r] Refresh is active %s", self.shard_id, self.fusion.process.__dict__)
                time.sleep(30)

        self.fusion.kill_softly()
