# -*- coding: utf-8 -*-

import os
import shutil
import logging

from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common.fusion.distributor import create_distributor
from sandbox.projects.common.fusion.distributor import DefaultDistributorParams as DistributorParams

from sandbox.projects.common.search.components import get_fusion_search
from sandbox.projects.common.search.components import DefaultFusionParams as FusionParams

from sandbox.projects.common.search.components import FUSION_DB_PRE_CREATED, FUSION_DB_EMPTY


DISK_TEST_TYPE = "[Disk index test]"
MEMORY_TEST_TYPE = "[Memory index test]"
MIXED_TEST_TYPE = "[Mixed (disk + memory) test]"


FusionParamsDescription = """
**Fusion Params**

* **Executable** is the fusion search binary (the result of ``BUILD_BASESEARCH`` task with fusion params on)
* **Fusion package with robot tools** is a ``TEAMCITY_RESOURCE`` (with attr resource_name=fusion) containing \
bsconfig-ready shard (folder) with fusion robot tools (e.g. ftdocmerger etc)
* **Fusion Config** - a config for fusion itself (``FUSION_SEARCH_CONFIG`` resource type)
* **Basesearch Config** - is a config for a basesearch part of fusion (``SEARCH_CONFIG``)
* **Choose the way to set fusion database** - tests imply different database initialization, following options are available:
    * **Feed docs from dump**: use a dump of IndexedDocs to send them to fusion before the test
    * **Use pre-created db**: use a previously created database
    * **Use both pre_created db and dump**: combination of the two
    * (pre-created db is usually used for disk index tests, while dump can be used for memory index initialization)
* **Docs dump for fusion**: a ``REALSEARCH_DUMP`` resource containing IndexedDocs to be fed to fusion with docfeeder.\
[The option is used if database source option is set to use dump]\
* **Number of documents fed from dump**: a param passed to docfeeder to limit the number of docs sent to fusion
* **Maximum number of documents in memory index**: In combination with a revious param allows to control index creation:\
If Number of documets fed surpasses the number of docs in memory set it results to disk index creation (that could be\
saved and reused if required)
* **OxygenOptions Config**: ``[optional param]`` a path to config which enables using fusion in kw-tuples mode\

"""


class FusionTestTaskBase:
    """
        Basic fusion handling routines
    """

    def get_fusion(self, get_db, max_documents=None, default_wait=None, run=True, event_log=False):
        """
        Returns initialized fusion component ready for shooting
        """
        fusion = get_fusion_search(get_db=get_db, max_documents=max_documents, default_wait=default_wait, event_log=event_log)

        if run:
            if get_db:
                self.init_fusion_with_db(fusion)
            else:
                self.init_fusion_empty(fusion)
        return fusion

    def get_distributor(self):
        distributor = create_distributor()
        distributor.start()
        return distributor

    def cleanup_index_dir(self, fusion):
        if os.path.exists(fusion.index_dir):
            logging.info("Dropping index dir: %s" % fusion.index_dir)
            shutil.rmtree(fusion.index_dir)
        logging.info("Created new index dir: %s" % fusion.index_dir)
        paths.make_folder(fusion.index_dir)

    def get_cleanup_dir_callback(self, fusion):
        def cleanup_callback():
            return self.cleanup_index_dir(fusion)
        return cleanup_callback

    def get_memorysearch_callback(self, fusion):
        def prepare_memorysearch_callback():
            return fusion.is_memorysearch_filled()
        return prepare_memorysearch_callback

    def init_fusion_with_db(self, fusion):
        fusion.start()
        fusion.wait(is_ready=fusion.is_index_ready)
        return fusion

    def init_fusion_empty(self, fusion):
        fusion.start()
        fusion.wait(is_ready=fusion.is_server_running)
        return fusion

    def dump_metrics(self, metrics, file_path):
        with open(file_path, "w") as info:
            info.write("<build>\n")
            for key, value in metrics.items():
                info.write('\t<statisticValue key="%s" value="%s"/>\n' % (key, value))
            info.write("</build>\n")


class FusionTestTask(SandboxTask, FusionTestTaskBase):
    """
    Basic Task to simplify fusion handling routines
    """

    input_parameters = FusionParams.params + DistributorParams.params

    execution_space = 90000

    def initCtx(self):
        for key, value in self.ctx.items():
            if key.startswith("sandbox."):
                self.ctx[self.correct_context_name(key)] = value

    def correct_context_name(self, name):
        return name.replace("sandbox.", "").replace(".", "_")

    def get_doc_size_callback(self, fusion, ctx_name="doc_size"):
        def get_doc_size():
            self.ctx[ctx_name] = fusion.get_doc_size()
            return self.ctx[ctx_name]
        return get_doc_size

    def get_db_source(self):
        logging.info("DB source: %s" % self.ctx[FusionParams.DbSource.name])
        return self.ctx[FusionParams.DbSource.name]

    def use_distributor(self):
        logging.info("Using distributor: %s" % self.ctx[DistributorParams.EnableDistributor.name])
        return self.ctx[DistributorParams.EnableDistributor.name]

    def is_db_resource_required(self):
        if self.get_db_source() == FUSION_DB_PRE_CREATED:
            return True
        return False

    def get_check_function(self, fusion):
        do_not_wait = lambda: True
        db_source = self.get_db_source()
        if db_source in [FUSION_DB_EMPTY]:
            return do_not_wait
        return fusion.is_index_ready

    def get_max_docs(self):
        return self.ctx[FusionParams.MaxDocs.name]

    def get_dump(self):
        """
        Sync dump resource
        """
        dump_file = self.sync_resource(self.ctx[FusionParams.Dump.name])
        logging.info("Dump file: %s" % dump_file)
        return dump_file
