# coding: utf-8

import os
import re
import json
import shutil
import logging
import tarfile
import tempfile

from sandbox import common
import sandbox.sandboxsdk as ssdk

from sandbox.projects import resource_types
import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.constants as consts


logger = logging.getLogger(__name__)
_CTX = {}


class FuzzingException(Exception):
    pass


class ResolveError(FuzzingException):
    pass


def upload_corpus_cases(path, recursive=True, description=None):
    """
    Upload specified cases to sandbox
    :param path: path to the corpus cases
    :param recursive: search in nested directories
    :return: sandbox resource id
    """
    tempdir = tempfile.mkdtemp()
    corpus_dir = os.path.join(tempdir, 'corpus_part')
    os.mkdir(corpus_dir)
    os.chmod(corpus_dir, 0775)

    for root, dirs, files in os.walk(path):
        for filename in files:
            dst = os.path.join(corpus_dir, filename)
            if os.path.exists(dst):
                continue
            src = os.path.join(root, filename)
            os.link(src, dst)

        if not recursive:
            break

    task = ssdk.channel.channel.task

    archive_path = ssdk.paths.get_unique_file_name(task.abs_path(''), os.path.basename(corpus_dir) + '.tar')
    with tarfile.open(archive_path, 'w:') as tar:
        tar.add(corpus_dir, arcname='.')

    resource = task.create_resource(
        description=description or 'Corpus cases',
        resource_path=archive_path,
        resource_type=resource_types.FUZZ_CORPUS_DATA,
    )
    task.mark_resource_ready(resource.id)

    shutil.rmtree(tempdir)
    return resource.id


def expand_and_commit_corpus(resource_id, project_path, user_name, description=None):
    """
    Expands corpus with specified resource for the target project
    :param resource_id:
    :param project_path: path to the target project (FUZZ) relative to the arcadia root
    :param user_name: user for svn commit
    :param description: extra message appended to the commit message
    """
    resource_id = int(resource_id)
    commit_msg = "Updated corpus with new data by sandbox task #{} for target {}".format(ssdk.channel.channel.task.id, project_path)
    if description:
        commit_msg += ": {}".format(description)

    logger.info("Going to make commit with message: %s", commit_msg)

    corpus_dir = _get_corpus_dir(project_path)
    corpus_json = os.path.join(corpus_dir, 'corpus.json')

    attempts = 10
    for _ in range(attempts):
        expand_corpus(corpus_dir, resource_id, project_path)
        try:
            ssdk.svn.Arcadia.commit(corpus_json, commit_msg, user_name)
        except ssdk.errors.SandboxSvnError:
            ssdk.svn.Svn.revert(corpus_json, recursive=False)
            ssdk.svn.Svn.update(corpus_json)
            continue
        else:
            return
    raise FuzzingException("Failed to commit changes in {} attempts".format(attempts))


def _default_resolver(current, new):
    raise ResolveError("Don't know how to resolve '%s'" % new)


def commit_data_with_corpus(path, resource_id, project_path, user_name, resolver=_default_resolver, description=None):
    """
    Expands corpus with specified resource for the target project and include specified path(s) to the commit
    :param path: repository path(s)
    :param resource_id:
    :param project_path: path to the target project (FUZZ) relative to the arcadia root
    :param user_name: user for svn commit
    :param resolver: user specified function which should take 2 parameters - path to the locally changed file
        and path to the up-to-date file from repository
        resolver should return path to the file which will replace file in repository or return nothing if changes were inplace
    :param description: extra message appended to the commit message
    """
    resource_id = int(resource_id)
    paths = list(common.utils.chain(path))

    corpus_dir = _get_corpus_dir(project_path)
    expand_corpus(corpus_dir, resource_id, project_path)
    paths.append(corpus_dir)

    def resolver_wrapper(current, new):
        if new.endswith("corpus.json"):
            assert new.startswith(corpus_dir), (new, corpus_dir)
            expand_corpus(corpus_dir, resource_id, project_path)
            return new
        return resolver(current, new)

    commit_data(paths, user_name, resolver_wrapper, description)


def commit_data(path, user_name, resolver=_default_resolver, description=None, attempts=20):
    commit_msg = "Commit from sandbox task #{}".format(ssdk.channel.channel.task.id)
    if description:
        commit_msg += ": {}".format(description)

    paths = list(common.utils.chain(path))
    paths = _remove_nested(paths)

    logger.info("Going to commit changes in %s with message: %s", paths, commit_msg)

    info_map = {}
    for _ in range(attempts):
        try:
            ssdk.svn.Arcadia.commit(paths, commit_msg, user_name)
        except ssdk.errors.SubprocessErrorBase as error:
            filename = error.stderr_full_path or error.stdout_full_path
            if not filename:
                raise

            with open(filename) as afile:
                data = afile.read()
            logger.debug("Svn commit error: %s", data)

            if not info_map:
                info_map = {p: ssdk.svn.Svn.info(p) for p in paths}
                logger.debug("Info map: %s", json.dumps(info_map, indent=4, sort_keys=True))

            target = _get_outdated_file(data, info_map)
            if not target:
                raise
            assert os.path.exists(target), target

            oldfile = tempfile.NamedTemporaryFile()
            shutil.copyfile(target, oldfile.name)

            ssdk.svn.Svn.revert(target, recursive=False)
            ssdk.svn.Svn.update(target)

            logging.debug("Applying resolver for %s", target)
            filename = resolver(oldfile.name, target)
            if filename and filename != target:
                shutil.copyfile(filename, target)
        else:
            return
    raise FuzzingException("Failed to commit changes in {} attempts".format(attempts))


def expand_corpus(corpus_dir, resource_id, project_path):
    assert not project_path.startswith("/"), "Project path must be relative to the arcadia root: %s" % project_path

    corpus_json = os.path.join(corpus_dir, 'corpus.json')
    if not os.path.exists(corpus_json):
        raise FuzzingException('Failed to expand corpus - missing corpus description file: %s' % corpus_json)

    data = fu.json_load(corpus_json)

    nparts = len(data['corpus_parts'])
    if nparts > 20:
        msg = "{} project's corpus consist of {} parts. Looks like the regular pipeline doesn't work for project " \
              "and there are difficulties with corpus minimization. " \
              "See https://testenv.yandex-team.ru/?screen=jobs&database=fuzzing for more info"
        raise FuzzingException(msg.format(project_path, nparts))

    if resource_id not in data['corpus_parts']:
        data['corpus_parts'].append(resource_id)

    fu.json_dump(corpus_json, data, indent=4, sort_keys=True)


def _remove_nested(paths):
    targets = []
    for path in sorted(paths):
        if not targets or not path.startswith(targets[-1]):
            targets.append(path)
    return targets


def _get_outdated_file(data, info_map):
    vcs_path = None
    for line in data.split("\n"):
        match = re.search(r"E160028: File '(.*?)' is out of date", line)
        if match:
            vcs_path = match.group(1)
            break

    if not vcs_path:
        raise FuzzingException("Failed to determine abs filename of outdated file (%s) - see logs for more info" % vcs_path)

    target = vcs_path
    filename = ""
    while target:
        for path, entry in info_map.iteritems():
            if entry['url'].endswith(target):
                return os.path.join(*filter(None, [path, filename]))

        filename = os.path.join(*filter(None, [os.path.basename(target), filename]))
        target = os.path.dirname(target)
    raise FuzzingException("Failed to determine abs filename of outdated file (%s) - see logs for more info" % vcs_path)


def _get_corpus_dir(project_path):
    assert not project_path.startswith('/'), 'Project must be arcadia root relative (%s)' % project_path

    task = ssdk.channel.channel.task

    global _CTX
    if not _CTX.get('arc_fuzzing_dir'):
        target_dir = 'fuzzing'
        parsed_url = ssdk.svn.Arcadia.parse_url(task.ctx.get(consts.ARCADIA_URL_KEY, consts.ARCADIA_TRUNK_URL))

        url = "arcadia:{}".format(os.path.join(parsed_url.path, target_dir))
        if parsed_url.revision:
            url = "{}@{}".format(url, parsed_url.revision)

        _CTX['arc_fuzzing_dir'] = ssdk.svn.Arcadia.checkout(url, target_dir)

    return os.path.join(_CTX['arc_fuzzing_dir'], project_path)
