# -*- coding: utf-8 -*-
import json
import requests

from requests.adapters import HTTPAdapter


class GencfgApiException(Exception):
    """Ошибка при работе с gencfg API."""
    pass


def _handle_http_error(e):
    """
    Если был получен ответ от сервера, надо сформировать исключение GencfgApiException с телом ответа,
    т.к. в нём может содержаться полезная информация об ошибке.

    :type e: requests.HTTPError
    """
    message = 'Gencfg responded with code {}. Body:\n{}'.format(e.response.status_code, e.response.content)
    return GencfgApiException(message)


class GencfgApiClient(object):
    def __init__(self, api_url):
        self._api_url = api_url.rstrip('/')
        self._session = requests.Session()
        self._session.mount(self._api_url, HTTPAdapter(max_retries=3))

    def get_hosts_data(self, hosts_list):
        try:
            target_url = '{}/trunk/hosts_data'.format(self._api_url)
            if hosts_list is None:
                r = self._session.get(target_url)
            else:
                r = self._session.post(
                    target_url,
                    data=json.dumps({'hosts': hosts_list}),
                    headers={'Content-Type': 'application/json'})
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise GencfgApiException('Failed to call Gencfg API. Error: {}'.format(e))
        return r.json()

    def get_models(self):
        try:
            r = self._session.get('{}/trunk/models'.format(self._api_url))
            r.raise_for_status()
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise GencfgApiException('Failed to call Gencfg API. Error: {}'.format(e))
        return r.json()

    def get_group_instances(self, group, branch='trunk'):
        try:
            _branch = branch
            if branch == 'trunk':
                _branch = 'trunk/searcherlookup'
            target_url = '{api}/{branch}/groups/{group}/instances'.format(
                api=self._api_url, group=group, branch=_branch)
            r = self._session.get(target_url)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise GencfgApiException('Failed to call Gencfg API. Error: {}'.format(e))
        return r.json()

    def get_last_stable_tag(self, branch='trunk'):
        try:
            r = self._session.get('{api}/{branch}/tags'.format(api=self._api_url, branch=branch))
            r.raise_for_status()
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise GencfgApiException('Failed to call Gencfg API. Error: {}'.format(e))
        tags = r.json().get('displayed_tags', [])
        return tags[0] if tags else None
