import time
import logging
import datetime
import os

import requests
import dateutil.parser


def _from_utc_to_local(utc_datetime):
    return utc_datetime.replace(tzinfo=None)


def _from_local_to_utc(date_time):
    now_timestamp = time.time()
    offset = datetime.datetime.fromtimestamp(now_timestamp) - datetime.datetime.utcfromtimestamp(now_timestamp)
    return (date_time - offset).replace(tzinfo=None)


def _utc_iso_format(date_time):
    date_time = _from_local_to_utc(date_time)
    return date_time.replace(tzinfo=None).isoformat() + 'Z'


class SolomonClient(object):
    def __init__(self, project, cluster, service, host='gencfg'):
        self._project = project
        self._cluster = cluster
        self._service = service
        self._host = host
        self._get_url = 'http://api.solomon.search.yandex.net/data-api/get'
        self._push_url = "https://solomon.yandex.net/api/v2/push?project={project}&cluster={cluster}&service={service}".format(
            project=project,
            cluster=cluster,
            service=service
        )
        self._headers = {
            'Content-Type': 'application/json',
            'Authorization': 'OAuth ' + os.environ.get('SOLOMON_TOKEN')
        }
        self._who = 'robot-gencfg'

        self._sensors = []

    def add_sensor(self, sensor_name, value, date_time):
        ts = _utc_iso_format(date_time)
        self._sensors.append({
            "labels": {
                "sensor": sensor_name,
            },
            "ts": ts,
            "value": value,
        })

    def flush(self):
        if not self._sensors:
            return
        data = {
            "commonLabels": {
                "host": self._host,
            },
            "sensors": self._sensors
        }
        resp = requests.post(self._push_url, json=data, headers=self._headers)
        resp.raise_for_status()
        logging.info('flushed %i sensors', len(self._sensors))
        self._sensors = []

    def load(self, sensor, points_cnt, interval='365d'):
        params = {
            "project": self._project,
            "cluster": self._cluster,
            "service": self._service,
            "l.sensor": sensor,
            "b": interval,
            "points": points_cnt,
            'who': self._who,
            'host': self._host,
        }
        sensors = requests.get(self._get_url, params=params, timeout=30).json()['sensors']

        result = {}
        for values in sensors:
            values = values['values']
            for rec in values:
                ts, value = rec['ts'], rec['value']
                ts = _from_utc_to_local(dateutil.parser.parse(ts))
                result[ts] = value
        return result

    @property
    def sensors_collected(self):
        return self._sensors[:]
