# -*- coding: utf-8 -*-

import os
import string
import urllib2
import re
import json
import bson
import traceback
import shutil
import pymongo
import datetime
import logging
import tempfile

from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.sandboxapi import TASK_FINISHED, TASK_FAILURE, TASK_UNKNOWN, TASK_DELETED, TASK_STOPPED, TASK_STOPPING
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Svn

# auxiliarily gencfg stuff
from sandbox.projects.common.gencfg import utils as config_generator_utils
from sandbox.projects.gencfg import resource_types as sdk2_resource_types


class IGencfgTask(SandboxTask):
    def __init__(self, *args, **kwargs):
        SandboxTask.__init__(self, *args, **kwargs)

    def get_privkey_path(self):
        """
            Get ssh privkey from storage, save to file and return path to this file
        """
        keypath = self.abs_path('privkey')
        if os.path.exists(keypath):
            return keypath

        data = self.get_vault_data('GENCFG', 'gencfg_default_oauth')
        with open(keypath, 'w') as f:
            f.write(data)
        os.chmod(keypath, 0600)

        return keypath

    def get_src_path(self):
        """
            Get default location on checked out gencfg

            :return (string): absolute path of gencfg location
        """
        return self.abs_path('gencfg')

    def get_last_commit(self, path):
        """
            Function return last commit from svn repo at specified path

            :param path (string): filesystem path to svn repository
            :return (string): svn commit number
        """

        p = run_process(["svn", "info"], work_dir=path, log_prefix='get_last_commit')
        return int(re.search("Last Changed Rev: (\d+)", open(p.stdout_path).read()).group(1))

    def get_last_commit_author(self, path):
        """
            Function returns author of last commit from svn repo at specified path

            :param path (string): filesystem path to svn repository
            :return (string): svn commit number
        """

        p = run_process(["svn", "log", "-l", "1", "-v"], work_dir=path, log_prefix='get_last_commit_author')
        text = open(p.stdout_path).read()
        if text.split('|')[1].strip() != 'robot-gencfg':
            return text.split('|')[1].strip()
        return text.partition('(commited by ')[2].partition('@')[0].strip()


class IGencfgBuildTask(IGencfgTask):
    """
        Task with some common thing for everybody, who wants to build gencfg.
    """

    ramdrive = SandboxTask.RamDrive(ctm.RamDriveType.TMPFS, 10240, None)

    def __init__(self, *args, **kwargs):
        IGencfgTask.__init__(self, *args, **kwargs)

        # some processes could be run in background
        self.background_procs = []

    #  ================================ START OF RESOURCES ================================================
    def get_mail_dir(self):
        """
            Directory with what we want to send every admin on release (directory with files kimkim.mail,osol.mail,...)
        """
        return self.abs_path('mail')

    def get_startrek_dir(self):
        """
            Directory with startrek task comments (directory with files GENCFG-123,SEPE-12345,...)
        """
        return self.abs_path('startrek')

    def get_nanny_file(self):
        """
            Full diff (from tools/diffbuilder), exported to nanny on release
        """
        return self.abs_path('nanny.diff')

    def get_announce_file(self):
        """
            Get filename for things to announce
        """
        return self.abs_path('gencfg.announce')

    def get_clean_generator_path(self):
        """
            Clean checkouted gencfg (without install.sh and any builds). It is merely result of svn co <url>
        """
        return self.abs_path('generator_resource')

    def get_generator_archive_path(self):
        """
            Gencfg directory with everything built. Archived to reduce size
        """
        return self.abs_path('generator.tar.gz')

    def get_searcherlookup_path(self):
        """
            Searcherlookup as separate resource
        """
        return self.abs_path('searcherlookup.conf')

    def get_generator_configs_path(self):
        """
            Content of w-generated/all as 7z archive (use 7z, because it compress files with same content very well)
        """
        return self.abs_path('generator.configs.7z')

    def get_generator_configs_tgz_path(self):
        """
            Content of w-generated/all as 7z archive (use tar, because there are no 7z in YP)
        """
        return self.abs_path('generator.configs.tar.gz')

    def get_generator_configs_web_path(self):
        """
            Content of w-generated/web as tar.gz archive
        """
        return self.abs_path('generator.configs.web.tar.gz')

    def get_generator_configs_video_path(self):
        """
            Content of w-generated/video as tar.gz archive
        """
        return self.abs_path('generator.configs.video.tar.gz')

    def get_generator_generated_path(self):
        """
            Everything we generated (content of generated and w-generated)
        """
        return self.abs_path('generator.generated.tar.gz')

    def get_generator_cache(self):
        """
            Content of db/cache directory with all cache data precalculateed
        """
        return self.abs_path("generator_cache")

    #  ================================= END OF RESOURCES ================================================

    def get_gencfg_path(self):
        """
            Default path to gencfg
        """
        return os.path.join(self.ramdrive.path, 'gencfg')

    def _create_build_logs(self, generator_dir):
        """
            We have to create resource with build logs (usually, if gencfg build failed and task exited with status FAILURE, all gencfg data is removed.
            Therefore we should save build logs somewhere
        """
        build_logs_path = self.abs_path('build')
        remove_path(build_logs_path)
        copy_path(os.path.join(generator_dir, 'build'), build_logs_path)
        self.ctx['gencfg_build_logs_id'] = self._create_resource(
            self.descr,
            build_logs_path,
            resource_types.CONFIG_BUILD_LOGS,
            arch='any'
        ).id

    def add_background_command(self, cmd, log_prefix):
        proc = run_process(cmd, work_dir=self.get_gencfg_path(), log_prefix=log_prefix, wait=False, check=False)
        self.background_procs.append((proc, cmd))

    def clone_and_install_generator(self, gencfg_path=None, tag=None, commit=None, create_resource=True, precalc_caches=False,
                                    create_cache_resource=False, load_cache_resource=False):
        """
            Clone gencfg from specified tag or commit and run install script
        """

        import api.copier

        if gencfg_path is None:
            gencfg_path = self.get_gencfg_path()

        config_generator_utils.clone_gencfg_all(gencfg_path, 'full', tag=tag, revision=commit)

        run_process(["bash", "./install.sh"], work_dir=gencfg_path, log_prefix="install")

        if load_cache_resource:
            db_cache_resource = channel.sandbox.list_resources(resource_type='GENCFG_DB_CACHE', status='READY', limit=1)[0]
            skynet_id = db_cache_resource.skynet_id
            resource_name = db_cache_resource.file_name
            cache_dir = os.path.join(gencfg_path, 'db', 'cache')

            copier = api.copier.Copier()
            tmp_dir = tempfile.mkdtemp()
            os.chmod(tmp_dir, 0o777)
            try:
                copier.get(skynet_id, tmp_dir, network=api.copier.Network.Backbone).wait()
                shutil.move(os.path.join(tmp_dir, resource_name), cache_dir)
            finally:
                shutil.rmtree(tmp_dir, ignore_errors=True)

        if precalc_caches:
            run_process(['./utils/common/precalc_caches.py'],  work_dir=gencfg_path, log_prefix='precalc_caches')
            if create_cache_resource:
                copy_path(os.path.join(gencfg_path, "db", "cache"), self.get_generator_cache())
                self._create_resource(self.descr, self.get_generator_cache(), resource_types.GENCFG_DB_CACHE, arch='any')

    def build_generator(self, skip_cache=False, create_resources=True, run_only_checks=False, use_separate_task_for_configs=False, run_build=True):
        """
            Run gencfg build script and generated all needed resources.
        """

        os.environ['SANDBOX_TASK_ID'] = str(self.id)
        os.environ['GZIP'] = '-1'

        pigz_path = os.path.join(self.get_gencfg_path(), 'utils', 'binary', 'pigz')

        if run_build:
            # run build
            try:
                env = os.environ.copy()
                if run_only_checks:
                    cmd_line = ['bash', os.path.join(self.get_gencfg_path(), 'gen.sh'), 'run_checks']
                else:
                    cmd_line = ['bash', os.path.join(self.get_gencfg_path(), 'gen.sh')]
                if use_separate_task_for_configs:
                    env['BUILD_CONFIGS_IN_SEPARATE_TASK'] = '1'
                run_process(cmd_line, work_dir=self.get_gencfg_path(), log_prefix='generator', environment=env)
            except Exception:
                self._create_build_logs(self.get_gencfg_path())
                raise
            self._create_build_logs(self.get_gencfg_path())

        if create_resources:
            # create CONFIG_ARCHIVE resource
            self.ctx['gencfg_archive_resource_id'] = self._create_resource(
                self.descr,
                self.get_generator_archive_path(),
                resource_types.CONFIG_ARCHIVE, arch='any'
            ).id
            run_process(
                [
                    'tar',
                    '-c',
                    '--use-compress-program=%s' % pigz_path,
                    '-f',
                    self.get_generator_archive_path(),
                    os.path.relpath(os.path.abspath(self.get_gencfg_path()), self.abs_path())
                ],
                work_dir=self.abs_path()
            )

            # create CONFIG_SEARCHERLOOKUP resource
            self.ctx['gencfg_searcherlookup_resource_id'] = self._create_resource(
                self.descr,
                self.get_searcherlookup_path(),
                resource_types.CONFIG_SEARCHERLOOKUP,
                arch='any'
            ).id
            shutil.copyfile(
                os.path.join(self.get_gencfg_path(), 'w-generated', 'searcherlookup.conf'),
                self.get_searcherlookup_path()
            )

            # create CONFIG_GENERATOR_CONFIGS resource
            self.ctx['gencfg_generator_configs_resource_id'] = self._create_resource(
                self.descr,
                self.get_generator_configs_path(),
                resource_types.CONFIG_GENERATOR_CONFIGS,
                arch='any'
            ).id
            run_process(['7zr', 'a', '-mx=3', self.get_generator_configs_path(), 'all', ],
                        work_dir=os.path.join(self.get_gencfg_path(), 'w-generated'),
                        log_prefix='7z')

            # create CONFIG_GENERATOR_CONFIGS_TGZ resource
            self.ctx['gencfg_generator_configs_tgz_resource_id'] = self._create_resource(
                self.descr,
                self.get_generator_configs_tgz_path(),
                sdk2_resource_types.CONFIG_GENERATOR_CONFIGS_TGZ,
                arch='any'
            ).id
            run_process(['tar', 'zcf', self.get_generator_configs_tgz_path(), 'all', ],
                        work_dir=os.path.join(self.get_gencfg_path(), 'w-generated'),
                        log_prefix='tar')

            # create CONFIG_GENERATOR_CONFIGS_WEB resource
            self.ctx['gencfg_generator_configs_web_resource_id'] = self._create_resource(
                self.descr,
                self.get_generator_configs_web_path(),
                sdk2_resource_types.CONFIG_GENERATOR_CONFIGS_WEB,
                arch='any'
            ).id
            run_process(['tar', 'zcf', self.get_generator_configs_web_path(), 'web', ],
                        work_dir=os.path.join(self.get_gencfg_path(), 'w-generated'),
                        log_prefix='tar')

            # create CONFIG_GENERATOR_CONFIGS_VIDEO resource
            self.ctx['gencfg_generator_configs_video_resource_id'] = self._create_resource(
                self.descr,
                self.get_generator_configs_video_path(),
                sdk2_resource_types.CONFIG_GENERATOR_CONFIGS_VIDEO,
                arch='any'
            ).id
            run_process(['tar', 'zcf', self.get_generator_configs_video_path(), 'video', ],
                        work_dir=os.path.join(self.get_gencfg_path(), 'w-generated'),
                        log_prefix='tar')

            # create CONFIG_GENERATOR_GENERATED resource
            self.ctx['gencfg_generator_generated_resource_id'] = self._create_resource(
                self.descr,
                self.get_generator_generated_path(),
                resource_types.CONFIG_GENERATOR_GENERATED, arch='any'
            ).id
            run_process(["bash", "./scripts/gen-configs-basesearch.sh", "clean"], work_dir=self.get_gencfg_path(),
                        log_prefix='clean_basesearch')
            run_process(["bash", "./scripts/gen-configs-web.sh", "clean"], work_dir=self.get_gencfg_path(),
                        log_prefix='clean_intsearch')
            run_process(['tar', '-c', '--use-compress-program=%s' % pigz_path, '-f', self.get_generator_generated_path(), 'w-generated', 'generated', ],
                        work_dir=os.path.join(self.get_gencfg_path()), log_prefix='tar')

            # wait background procs
            logging.info("Waiting for background processes")
            for proc, cmd in self.background_procs:
                proc.wait()
                logging.info("Command <%s> exited with status <%s>" % (cmd, proc.returncode))
                if proc.returncode != 0:
                    raise Exception("Command <%s> exited with status <%s>" % (cmd, proc.returncode))

    def generate_diff_create_cmd(self, generator_path, differs, tag_pattern, commit_filter='lambda x: True'):
        """
            Function runs diffbuilder and creates all necessary resources

            :param generator_path: path to gencfg main repository (balancer and db repository must be also cloned)
            :param differs: list of differs for diffbuilder
            :param tag_pattern (str): pattern for intresting tags. We need to find previous tag from current state, thus, we should not tag name pattern
            :return: nothing
        """

        remove_path(self.get_mail_dir())
        remove_path(self.get_startrek_dir())
        remove_path(self.get_nanny_file())
        remove_path(self.get_announce_file())

        make_folder(self.get_mail_dir())
        make_folder(self.get_startrek_dir())

        return ['/skynet/python/bin/python', os.path.join(generator_path, 'tools', 'diffbuilder', 'main.py'),
                '--commit-filter', commit_filter, '--tag-pattern', tag_pattern, '-p', '-m', self.get_mail_dir(),
                '-s', self.get_startrek_dir(), '--nanny-filename', self.get_nanny_file(), '--gencfg-announce-filename', self.get_announce_file(), '-d', ','.join(differs)]

    def generate_diff_process_result(self):
        # create CONFIG_MAIL_DIR resource
        if len(os.listdir(self.get_mail_dir())):
            resource = self._create_resource(self.descr, self.get_mail_dir(), resource_types.CONFIG_MAIL_DIR,
                                             arch='any')
            self.ctx['maildir_id'] = resource.id
        self.ctx['generated_changelog'] = open(os.path.join(self.get_mail_dir(), 'ALL.mail')).read().decode('utf-8')
        self.ctx['generated_changelog'] = self.ctx['generated_changelog'][:10000]

        # create CONFIG_STARTREK_DIR resource
        if len(os.listdir(self.get_startrek_dir())):
            resource = self._create_resource(self.descr, self.get_startrek_dir(),
                                             resource_types.CONFIG_STARTREK_DIR, arch='any')
            self.ctx['startrek_id'] = resource.id

        # create CONFIG_NANNY_DIR resource
        resource = self.create_resource(self.descr, self.get_nanny_file(),
                                        resource_types.CONFIG_NANNY_DIR, arch='any',
                                        attributes=dict(ttl='inf'))
        self.ctx['nannyfile_id'] = resource.id

        # create GENCFG_ANNOUNCE_FILE resource
        resource = self._create_resource(self.descr, self.get_announce_file(),
                                         resource_types.GENCFG_ANNOUNCE_FILE, arch='any')
        self.ctx['announce_id'] = resource.id

    def on_release_get_resource_files(self, resource_type):
        """
            Download resource of specified type and put in somewhere in filesystem
        """

        resources = self.list_resources(resource_type=resource_type)

        if len(resources) != 1:
            raise Exception('Task should have exactly one resource of type {0:s}'.format(resource_type.name))

        resource = resources[0]

        import api.copier
        mycopier = api.copier.Copier()
        skynet_resource = mycopier.handle(resource.skynet_id)
        filenames = map(lambda x: x['name'].partition('/')[2], filter(lambda x: x['name'].find('/') >= 0,
                                                                      skynet_resource.list(network=api.copier.Network.Backbone).wait().files()))

        return resource, filenames

    def report_mail_on_release(self, release_status, release_subject):
        """
            In this function we send mail to people, whose groups have been changed in current tag. What to send, already calculated and
            put in CONFIG_MAIL_DIR resource as separate file (one per admin)

            :param relese_status(str): release status: stable/prestable/...
            :param relese_subject(str): short release description
            :return None:
        """

        if 'maildir_id' not in self.ctx:  # no need to send mail
            return

        mail_prefix = (
            "Добрый день.\n\n"
            "Вы получили это письмо из-за изменений в некоторых GenCfg-группах. "
            "Более подробно про релиз можно посмотреть тут {} или спросить у одного из следующих людей: {}.\n\n"
            "Ниже -- полный diff групп, за которыми вы наблюдаете и которые поменялись.\n\n".format(
                common.utils.get_task_link(self.id), 'sereglond,shotinleg'
            )
        )

        mail_log = open(self.log_path('startrek.txt'), 'w')
        mail_resource, mail_filenames = self.on_release_get_resource_files(resource_types.CONFIG_MAIL_DIR)
        for filename in mail_filenames:
            httplink = '%s/%s' % (mail_resource.proxy_url, filename)
            try:
                mail_body = mail_prefix + config_generator_utils.retry_urlopen(4, httplink, timeout=10)
            except Exception:
                mail_log.write("Got exception while fetching data from <%s>\n%s" % (httplink, traceback.format_exc()))
            mail_body = mail_body[:1000000]

            # remove all users ending with non yandex-team.ru domain, as send_email cannot work with them
            mail_user, _ = os.path.splitext(filename)
            if mail_user.find('@') != -1:
                if mail_user.split('@', 1)[1] != 'yandex-team.ru':
                    mail_user = None
                else:
                    mail_user = mail_user.split('@', 1)[0]
            if not mail_user:
                continue

            if release_status:
                if mail_user == 'ALL':
                    mail_subj = '[{0:s}] {1:s} (full diff)'.format(release_status, release_subject)
                else:
                    mail_subj = '[{0:s}] {1:s} (personal to {2:s})'.format(release_status, release_subject, mail_user)
            else:
                mail_subj = release_subject

            if mail_user == 'ALL':
                channel.sandbox.send_email('osol', '', mail_subj, mail_body)
            else:
                channel.sandbox.send_email(mail_user, '', mail_subj, mail_body)

    def report_startrek_on_release(self, tag_name):
        """
            Added release message to all startrek tasks, mentioned in current tag commits

            :param tag_name(str): tag name, e.g. stable-86-r1
            :return None:
        """

        if 'startrek_id' not in self.ctx:
            return

        startrek_log = open(self.log_path('startrek.txt'), 'w')
        startrek_resource, startrek_filenames = self.on_release_get_resource_files(
            resource_types.CONFIG_STARTREK_DIR)
        for filename in startrek_filenames:
            httplink = '%s/%s' % (startrek_resource.proxy_url, filename)
            body = config_generator_utils.retry_urlopen(4, httplink, timeout=10)
            taskid = filename

            text = "Released gencfg tag <{0:s}> in task {1:s} :\n\n{2:s}".format(
                tag_name, self.http_url(), body
            )
            text = json.dumps({'text': text})

            url = "https://st-api.yandex-team.ru/v2/issues/%s/comments" % taskid
            headers = {
                "Authorization": "OAuth {}".format(self.get_vault_data('GENCFG', 'gencfg_default_oauth')),
                "Content-Type": "application/json",
            }

            req = urllib2.Request(url, text, headers)

            try:
                config_generator_utils.retry_urlopen(4, req)
            except Exception, e:
                startrek_log.write("Got exception while adding comment to %s: (%s, %s)\n%s" % (filename, type(e), e, traceback.format_exc()))
        startrek_log.close()

    def mark_tag_released(self, tag, commit, full=False, diff_to=None):
        commit = normalize_commit(commit)
        self.set_test_status(commit, success=True)
        logging.info('tag: {}, commit: {}, full: {}, diff_to: {}'.format(tag, commit, full, diff_to))

        data = {'commit': commit}
        data.update({'fullstate': 1} if full else {'diff_to': diff_to})

        get_topology_mongo()['tags'].update({'tag': tag}, {'$set': data}, upsert=True)

    def mark_searcher_lookup(self, commit, full=False, diff_to=None):
        commit = normalize_commit(commit)
        logging.info('commit: {}, full: {}, diff_to: {}'.format(commit, full, diff_to))
        if full:
            get_topology_mongo()['tags'].update({'commit': commit}, {'$set': {'fullstate': 1}})
        elif diff_to:
            get_topology_mongo()['tags'].update({'commit': commit}, {'$set': {'diff_to': diff_to}})
        else:
            raise Exception('not full and not diff_to')

    def set_test_status(self, commit, success, author=None):
        commit = normalize_commit(commit)
        doc = {'test_passed': success}
        if author:
            doc['author'] = author
        doc['task_id'] = self.ctx.get('__GSID', None)
        get_topology_mongo()['commits'].update({'commit': commit}, {'$set': doc}, upsert=True)

    def list_major_tags(self):
        tags = list(get_topology_mongo()['tags'].find({}, sort=[('commit', 1)]))
        major_tags = filter(lambda tag: re.match('^stable-\d+-r1$', tag['tag']), tags)
        return major_tags

    @staticmethod
    def _get_oldest_gencfg_trunk_commit(commit, leave_last_n=100, max_time_delta=datetime.timedelta(hours=7)):
        count = 0
        old_record = bson.ObjectId.from_datetime(datetime.datetime.utcnow() - max_time_delta)
        logging.debug('old record: %s', old_record)
        for record in get_topology_mongo()['commits'].find(
            {
                'test_passed': True,
                'commit': {'$lt': commit}
            },
            sort=[('commit', -1)]
        ):
            count += 1
            if record['_id'] < old_record and count >= leave_last_n:
                return int(record['commit'])

    def clean_gencfg_trunk(self, commit):
        commit = normalize_commit(commit)
        oldest_commit = self._get_oldest_gencfg_trunk_commit(commit)
        logging.info('clean gencfg_trunk: commit < %i', oldest_commit)

        for attempt in range(3):
            try:
                get_topology_mongo()['gencfg_trunk'].remove({'commit': {'$lt': oldest_commit}})
                return
            except:
                pass
        else:
            raise Exception('Failed to remove commits older than <{}> from mong'.format(oldest_commit))

    def remove_old_tags(self, keep_last_n_major):
        plain_collections = ['tags', 'instances']
        plain_collections_mongo_a = ['groups']
        diff_like_collections = ['search_lookup', 'search_map']
        tag = self.list_major_tags()[-keep_last_n_major]
        logging.info(str(tag))
        commit = tag['commit'] if 'fullstate' in tag else tag['diff_to']
        for coll in plain_collections:
            logging.info('remove commits lt {} from {}'.format(commit, coll))
            get_topology_mongo()[coll].remove({'commit': {'$lt': str(commit)}})
        for coll in plain_collections_mongo_a:
            logging.info('remove commits lt {} from {}'.format(commit, coll))
            get_topology_mongo_a()[coll].remove({'commit': {'$lt': str(commit)}})
        for coll in diff_like_collections:
            logging.info('remove commits lt {} from {}'.format(commit, coll))
            get_topology_mongo()[coll].remove({'commit': {'$lt': int(commit)}})


class IGencfgReleaseTask(IGencfgTask):
    def check_running_tasks(self):
        """
            Function waits while all task of same type with lesser ids finished. We have to wait because release tasks can not be run in
            parallel.

            :return: nothing
        """

        finished_statuses = [TASK_FINISHED, TASK_FAILURE, TASK_UNKNOWN, TASK_DELETED, TASK_STOPPED, TASK_STOPPING, 'DRAFT']
        tasks = channel.sandbox.list_tasks(task_type=self.type)
        tasks = filter(lambda x: (x.id < self.id) and (x.status not in finished_statuses), tasks)

        if len(tasks) > 0:
            self.wait_all_tasks_stop_executing(tasks)

    def need_new_tag(self, src_path, db_path):
        """
            Function check if we need to create new tag by checking if last commit is tagged. Fill task context with info on why tag can not be created.

            :param path (string): filesystem paths to svn repository
            :return (bool): True if we need new tag, False otherwise
        """
        # check if last commit is tagged
        url = "svn+ssh://arcadia.yandex.ru/arc/tags/gencfg"
        tag_last_revision = int(Svn.info(url)["commit_revision"])
        src_last_revision = int(Svn.info(src_path)["commit_revision"])
        db_last_revision = int(Svn.info(db_path)["commit_revision"])

        if src_last_revision <= tag_last_revision and db_last_revision <= tag_last_revision:
            self.set_info('Can not release commit {}: commit {} already released'.format(
                src_last_revision if src_last_revision > db_last_revision else db_last_revision,
                tag_last_revision
            ))
            return False

        return True

    def generate_create_tag_bash_command(self, prj_type):
        """
            Function generate bash command to create new tag.

            :param script_name (string): script name (full path to script, which creates new tag)
            :return: list of bash arguments to create new tag (for utility utils/common/create_new_tag.py)
        """

        script_name = os.path.join(self.get_src_path(), 'utils', 'common', 'create_new_tag.py')
        args = ' '.join([script_name, '-s', 'yes', '-t', 'no', '-p', prj_type, '--notify-user', self.author])
        return args

    def get_full_gencfg_diff(self, child_task):
        """
            Functions extracts full diff, generated by child task (full diff is located in specific file in child task)

            :param child_task (int): id of sandbox child task
            :return (str): full diff generated by child task
        """

        resources = channel.sandbox.list_resources(resource_type=resource_types.CONFIG_MAIL_DIR, task_id=child_task)
        if len(resources) != 1:
            raise Exception("Found %d resources of type CONFIG_MAIL_DIR (should be exactly one)")
        resource = resources[0]

        links = channel.sandbox.get_resource_http_links(resource.id)
        if len(links) == 0:
            raise Exception("Found 0 http links to resource %d" % resource.id)
        httplink = "%s/ALL.mail" % (links[0])

        return config_generator_utils.retry_urlopen(4, httplink, timeout=10)

    def release_built_tag(self, stdout_path):
        """
            Function creates release of build task, which was previously started by this one.

            :param stdout_path (string): path to full stdout of script, which created build task
            :retunrn: nothing
        """

        tagname = string.strip(re.search('Creating new tag: (.*)', open(stdout_path).read()).group(1))
        tagname = 'tag@%s' % tagname

        child_task_id = int(re.search('Waiting sandbox task: (\d+)', open(stdout_path).read()).group(1))

        diff = self.get_full_gencfg_diff(child_task_id)
        changelog = """Created new gencfg tag %s.

Changes:
    - see autodiff.

%s""" % (tagname, diff)

        changelog = changelog.decode('utf8', 'replace').encode('utf8')  # to convert non-utf symbols to something correct
        changelog = changelog[:500000]

        release_id = self.create_release(
            child_task_id,
            status='stable',
            subject='Building %s' % tagname,
            comments=changelog,
            addresses_to=['%s@yandex-team.ru' % self.author],
        )

        # wait for finish
        self.ctx['releasing'] = True
        self.wait_tasks(
            tasks=[child_task_id],
            statuses=list(self.Status.Group.RELEASE),
            wait_all=True
        )

        self.ctx['result'] = "Successfully created tag %s (task %s, release %s)" % (tagname, child_task_id, release_id)


def normalize_commit(commit):
    commit = str(commit)
    if commit.startswith('r'):
        commit = commit[1:]

    return commit


def get_topology_mongo():
    # not supposed to be changed from sandbox
    ALL_HEARTBEAT_C_MONGODB = ','.join([
        'myt0-4012.search.yandex.net:27017',
        'myt0-4019.search.yandex.net:27017',
        'sas1-6063.search.yandex.net:27017',
        'sas1-6136.search.yandex.net:27017',
        'vla1-3984.search.yandex.net:27017',
    ])

    return pymongo.MongoReplicaSetClient(
        ALL_HEARTBEAT_C_MONGODB,
        connectTimeoutMS=15000,
        replicaSet='heartbeat_mongodb_c',
        w='majority',
        wtimeout=15000,
    )['topology_commits']


def get_topology_mongo_a():
    # not supposed to be changed from sandbox
    ALL_HEARTBEAT_A_MONGODB = ','.join([
        'myt0-4007.search.yandex.net:27017',
        'myt0-4010.search.yandex.net:27017',
        'sas1-5874.search.yandex.net:27017',
        'sas1-5919.search.yandex.net:27017',
        'vla1-6003.search.yandex.net:27017',
    ])

    return pymongo.MongoReplicaSetClient(
        ALL_HEARTBEAT_A_MONGODB,
        connectTimeoutMS=15000,
        replicaSet='heartbeat_mongodb_a',
        w='majority',
        wtimeout=15000,
    )['topology_commits']
