import os
import ssl
import time
import urllib2

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.sandboxapi import Sandbox
import re
import logging


GENCFG_TRUNK_DATA_PATH = "svn+ssh://arcadia.yandex.ru/arc/trunk/data/gencfg_db"
GENCFG_TRUNK_SRC_PATH = "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/gencfg"
GENCFG_BRANCH_PREFIX = "svn+ssh://arcadia.yandex.ru/arc/branches/gencfg"
GENCFG_TAG_PREFIX = "svn+ssh://arcadia.yandex.ru/arc/tags/gencfg"
GENCFG_BALANCER_POSTFIX = "custom_generators/balancer_gencfg"


class ERepoType(object):
    FULL = 'full'  # full gencfg (src+db)
    DB = 'db'  # gencfg db
    BALANCER = 'balancer'  # gencfg balancer
    ALL = (FULL, DB, BALANCER)


def run_process_with_retry(attempts, *args, **kwargs):
    timeout_table = [1, 5, 10, 20]
    for attempt in xrange(attempts):
        try:
            return run_process(*args, **kwargs)
        except Exception:
            if attempt == attempts - 1:
                raise

        time.sleep(timeout_table[min(attempt, len(timeout_table) - 1)])


def clone_gencfg_repo(url, dest_dir, revision=None):
    """Clone repo to specified directory"""

    Arcadia.checkout(url, dest_dir, revision=revision)


def clone_gencfg_all(dest, repo_type, revision=None, tag=None):  # FIXME: copy-paste from gencfg
    if tag in (None, 'trunk'):  # checking out trunk
        if repo_type == ERepoType.FULL:
            clone_gencfg_repo(GENCFG_TRUNK_SRC_PATH, dest, revision=revision)
            clone_gencfg_repo(GENCFG_TRUNK_DATA_PATH, os.path.join(dest, 'db'), revision=revision)
        elif repo_type == ERepoType.DB:
            clone_gencfg_repo(GENCFG_TRUNK_DATA_PATH, dest, revision=revision)
        elif repo_type == ERepoType.BALANCER:
            url = '{}/{}'.format(GENCFG_TRUNK_SRC_PATH, GENCFG_BALANCER_POSTFIX)
            clone_gencfg_repo(url, dest, revision=revision)
        else:
            raise Exception('Unknown repo_type <{}>'.format(repo_type))
    else:  # checking out tag
        if repo_type == ERepoType.FULL:
            url = '{}/{}'.format(GENCFG_TAG_PREFIX, tag)
            clone_gencfg_repo(url, dest, revision=revision)
        elif repo_type == ERepoType.DB:
            url = '{}/{}/db'.format(GENCFG_TAG_PREFIX, tag)
            clone_gencfg_repo(url, dest, revision=revision)
        elif repo_type == ERepoType.BALANCER:
            url = '{}/{}/{}'.format(GENCFG_TAG_PREFIX, tag, GENCFG_BALANCER_POSTFIX)
            clone_gencfg_repo(url, dest, revision=revision)
        else:
            raise Exception('Unknown repo_type <{}>'.format(repo_type))


def pull_gencfg_all(path):
    run_process_with_retry(3, ['./pull.sh'], work_dir=path, log_prefix='pull_gencfg_db')


def commit_gencfg_db(path, commit_message):
    run_process_with_retry(3, ['svn', 'commit', '-m', commit_message], work_dir=os.path.join(path, 'db'), log_prefix='commit_gencfg_db')


REQS = 'requirements_list'


class Reqs:
    def __init__(self, dct=None):
        self.dct = dct or {}

    @classmethod
    def parse(cls, text):
        self = cls()
        for line in text.split('\n'):
            line = line.strip()
            t = re.match('[^ =<>]+', line)
            if t:
                self.dct[line[:t.end()]] = line[t.end():].strip()
            else:
                self.dct[line] = ''
        return self

    def __eq__(self, other):
        for key in self.dct:
            if key not in other.dct:
                return False
            newer = self.dct[key]
            older = other.dct[key]
            if newer != '' and newer != older:
                return False
        return True

    @staticmethod
    def read_reqs(path):
        with open(path) as f:
            res = f.read()
        return res

    def find_existing(self):
        s = Sandbox()
        tasks_ = sorted(s.list_tasks(task_type='CREATE_VENV', status='FINISHED'), key=lambda x: x.id, reverse=True)
        logging.info('____total: {}'.format(len(tasks_)))
        for one in tasks_:
            logging.info('____{}'.format(one.ctx))
            if REQS in one.ctx and 'RESOURCES' in one.ctx:
                logging.info(self == Reqs.parse(one.ctx[REQS]))
                if self == Reqs.parse(one.ctx[REQS]):
                    resource = s.get_resource(one.ctx['RESOURCES']['resources'][0])
                    if resource and resource.is_ready():
                        logging.info('____res ready')
                        return one.ctx['RESOURCES']['resources'][0]
        return None


def get_venv_id(requirements_url, task):
    Arcadia.export(requirements_url, 'requirements.txt')
    newer = Reqs.parse(Reqs.read_reqs('requirements.txt'))
    existing = newer.find_existing()
    if existing:
        return existing
    s = Sandbox()
    task_id = task.create_subtask(
        'CREATE_VENV', 'auto building venv for gencfg task',
        execution_space=1024,
        priority=('USER', 'HIGH')
    )
    s.wait_task(task_id, timeout=1000)
    task_ = s.get_task(task_id)
    try:
        res = task_.ctx['RESOURCES']['resources'][0]
    except:
        logging.info('ctx: {}'.format(task_.ctx))
        res = task_.ctx['resources'][0]
    return res


def retry_urlopen(retries, *args, **kwargs):
    SLEEP_TIMINGS = [1, 5, 10, 50, 200]
    for i in range(retries):
        try:
            return urllib2.urlopen(*args, **kwargs).read()
        except (urllib2.HTTPError, ssl.SSLError, urllib2.URLError) as e:
            if i < retries - 1:
                time.sleep(SLEEP_TIMINGS[min(i, len(SLEEP_TIMINGS) - 1)])
            else:
                url = args[0]
                if isinstance(url, urllib2.Request):
                    url = url.get_full_url()
                raise Exception("Got exception <%s> while processing url <%s> (%s)" % (e.__class__, url, str(e)))
        except Exception:
            if i < retries - 1:
                time.sleep(SLEEP_TIMINGS[min(i, len(SLEEP_TIMINGS) - 1)])
            else:
                raise
