# -*- coding: utf-8 -*-
import logging
import json
import os

from sandbox import sdk2

from sandbox.projects import resource_types
from collections import OrderedDict
from sandbox.common.types.misc import NotExists
import sandbox.common.types.task as ctt
from sandbox import common

RESULT_MARK = 'mark'
LAST_MODIFIED = 'last_modified'
DEFAULT_CORES = 1
DEFAULT_RAM = 8192
DEFAULT_DISK = 30 * 1024


class SearchFor(object):
    ANY_RESOURCE = 'any_resource'
    STABLE = 'stable'
    TESTING = 'testing'
    TESTING_OR_STABLE = 'testing_or_stable'
    FIXED = 'fixed'
    NONE = 'none'


class YtTablePath(sdk2.parameters.String):
    pass


def _rs_name(name):
    return 'release_status_' + name


def _attrs_name(name):
    return 'attrs_' + name


def _ignore_update(name):
    return 'ignore_update_' + name


def _fetch_released_tasks(resource_type, release_status):
    limit = 20
    offset = 0
    while True:
        releases = sdk2.task.Task.server.release.read(
            resource_type=resource_type,
            type=release_status,
            limit=limit,
            offset=offset,
            order='-id',
            include_broken=False,
        ).get('items')
        if not releases:
            break
        for task in releases:
            yield task['task_id']
        offset += limit


def _fetch_task_resource_id(resource_type, task_id, attrs):
    resources = sdk2.task.Task.server.resource.read(
        type=resource_type,
        task_id=task_id,
        limit=1,
        attrs=attrs,
    ).get('items')
    if resources:
        return resources[0]['id']


class Indexing(sdk2.Task):
    """
        Update everything with indexing
    """

    def setup_indexing_params(self,
                              indexing_task,
                              index_type_for_timestamp,
                              release_subject="ADDRS new resource",
                              release_comments="automatic update",
                              indexing_execution_space=None,
                              force_rebuild=False,
                              force_parent_requirements=False,
                              ):
        self.indexing_task = indexing_task
        self.time_reference_type = index_type_for_timestamp
        self.release_subject = release_subject
        self.release_comments = release_comments
        self.indexing_execution_space = indexing_execution_space
        self.force_rebuild = force_rebuild
        self.force_parent_requirements = force_parent_requirements

    def setup_params(self, resource_params, yt_path_params, other_params):
        self.resource_params = resource_params
        self.yt_path_params = yt_path_params
        self.other_params = other_params

        logging.info('Resource params: %d', len(resource_params))
        logging.info(resource_params)
        logging.info('YT path params: %d', len(yt_path_params))
        logging.info(yt_path_params)
        logging.info('Other params: %d', len(other_params))
        logging.info(other_params)

    def _get_resource(self, name, resource_type):
        search_for = getattr(self.Parameters, _rs_name(name))
        attrs = getattr(self.Parameters, _attrs_name(name))
        if not attrs:
            attrs = {}
        else:
            attrs = json.loads(attrs)

        if search_for == SearchFor.NONE:
            return None

        if search_for == SearchFor.ANY_RESOURCE:
            return self._find_any_resource(resource_type, attrs)

        if search_for == SearchFor.FIXED:
            return getattr(self.Parameters, name)

        if search_for == SearchFor.TESTING:
            return self._find_release(resource_type, ctt.ReleaseStatus.TESTING, attrs)

        if search_for == SearchFor.STABLE:
            return self._find_release(resource_type, ctt.ReleaseStatus.STABLE, attrs)

        if search_for == SearchFor.TESTING_OR_STABLE:
            testing_resource = self._find_release(resource_type, ctt.ReleaseStatus.TESTING, attrs)
            stable_resource = self._find_release(resource_type, ctt.ReleaseStatus.STABLE, attrs)

            if testing_resource is not None and stable_resource is not None:
                if testing_resource.created > stable_resource.created:
                    return testing_resource
                else:
                    return stable_resource

            return stable_resource or testing_resource

        raise common.errors.TaskError('Unsupported search_for parameter: "{}"'.format(search_for))

    def _find_release(self, resource_type, status, attrs={}):
        for task_id in _fetch_released_tasks(resource_type, status):
            resource_id = _fetch_task_resource_id(resource_type, task_id, attrs)
            if resource_id is not None:
                return sdk2.Resource[resource_id]

    def _find_any_resource(self, resource_type, attrs={}):
        resource_id = _fetch_task_resource_id(resource_type, None, attrs)
        if resource_id is not None:
            return sdk2.Resource[resource_id]

    def _check_need_update(self):
        logging.info('Start updating sources')

        index_resource = self._find_release(self.time_reference_type, self.Parameters.release_type) if self.time_reference_type else None
        found_smth_new = index_resource is None

        for source in self.resource_params:
            logging.info('Checking %s...', source['name'])

            resource = self._get_resource(source['name'], source['param'].resource_type)
            updated = False
            source['resource'] = None
            if resource:
                if index_resource:
                    updated = index_resource.created < resource.created
                source['resource'] = resource
            else:
                logging.warn('No resource found: %s', source)
                continue

            affect_rebuild = (updated and not getattr(self.Parameters, _ignore_update(source['name'])))
            found_smth_new |= affect_rebuild

            logging.info('Checked: id=%s new=%r affect to rebuild=%r', resource.id, updated, affect_rebuild)
            if self.Context.rebuild_reason is NotExists and affect_rebuild:
                self.Context.rebuild_reason = source['name']
                self.Context.save()

        if self.force_rebuild:
            self.Context.rebuild_reason = 'force_rebuild'
            self.Context.save()
            found_smth_new = True

        logging.info('Sources, need reindex: %r', found_smth_new)
        return found_smth_new

    def _setup_yt_client(self, yt_client):
        return False

    def _translate_parameters(self, subtask):
        for source in self.resource_params:
            value = source['resource']
            setattr(subtask.Parameters, source['name'], value)

        if len(self.yt_path_params):
            import yt.wrapper as yt_client
            fix_paths = self._setup_yt_client(yt_client)
            for source in self.yt_path_params:
                value = getattr(self.Parameters, source['name'])
                if fix_paths and value and len(value):
                    try:
                        real_path = yt_client.get(os.path.join(value, '@path'))
                        logging.info('%s real path is %s' % (value, real_path))
                        value = real_path
                    except yt_client.YtHttpResponseError as err:
                        logging.info('Error geting %s real path. Details:' % value)
                        logging.info(err)
                setattr(subtask.Parameters, source['name'], value)

        for source in self.other_params:
            value = getattr(self.Parameters, source['name'])
            setattr(subtask.Parameters, source['name'], value)
        if self.Requirements.tasks_resource:
            subtask.Requirements.tasks_resource = self.Requirements.tasks_resource
        if self.force_parent_requirements:
            # All work is done by subtask, so we can use defaults in update task
            if self.Requirements.cores:
                subtask.Requirements.cores = self.Requirements.cores
            if self.Requirements.ram:
                subtask.Requirements.ram = self.Requirements.ram
            if self.Requirements.disk_space:
                subtask.Requirements.disk_space = self.Requirements.disk_space

    def begin_indexing(self):
        logging.info('Running indexer...')
#       TODO список для выбора статуса подзадачи
        priority = ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)
        subtask = self.indexing_task(self, execution_space=self.indexing_execution_space, priority=priority, kill_timeout=self.Parameters.kill_timeout)
        self._translate_parameters(subtask)
        subtask.save()
        subtask.enqueue()
        self.Context.subtask_id = subtask.id
        self.Context.save()
        raise sdk2.WaitTask(subtask, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def check_and_begin_indexing(self):
        rebuild = self._check_need_update()
        if rebuild:
            logging.info('Rebuild due to resource ', self.Context.rebuild_reason)
            self.begin_indexing()
        else:
            logging.info('Index need not to be updated')

    def end_indexing(self):
        subtask = sdk2.Task[self.Context.subtask_id]

        if subtask.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure('Child task {} failed'.format(subtask))

        subject = self.release_subject

        mark = self.Parameters.mark_index
        if mark:
            subject += ' !!!MARK[%s]' % mark

        if self.Parameters.release_type:
            self.server.release(task_id=subtask.id,
                                type=self.Parameters.release_type,
                                subject=subject,
                                comments=self.release_comments)

    def on_execute(self):
        if self.Context.subtask_id is NotExists:
            self.check_and_begin_indexing()
        else:
            self.end_indexing()


def _extract_parameters(task_cls):
    '''
    Iterates over parameter set of a task class.
    Yields (name, parameter) pairs.
    '''
    for name in task_cls.Parameters.__custom_parameters_names__:
        parameter = getattr(task_cls.Parameters, name)
        yield name, parameter()


def _make_parameters_cls(parameters_dict):
    '''
    Creates Parameters class inherited from sdk2.Task.Parameters.
    '''
    data = {}
    names = []

    for name, parameter in parameters_dict.iteritems():
        data[name] = parameter
        names.append(name)

    data['__names__'] = names
    return type('Parameters', (sdk2.Task.Parameters,), data)


def _is_resource(parameter):
    return parameter.__name__ == 'Resource'


def _is_yt_path(parameter):
    return parameter.__name__ == 'YtTablePath'


def _is_other_param(parameter):
    return not _is_resource(parameter) and not _is_yt_path(parameter)


def _human_readable(s):
    return ' '.join(s.split('_'))


def _get_resource_class(name):
    # TODO: workaround for ./sandbox test_tasks, remove this if later
    if hasattr(resource_types, name):
        return getattr(resource_types, name)
    return sdk2.Resource[name]


def generate_base_update_task_sdk2(build_task, release_subject, index_type_for_timestamp=None, indexing_execution_space=None, force_rebuild=False):

    def create_wrap_parameters(name, parameter):
        search_for = sdk2.parameters.String('{}: search for'.format(parameter.description), default=SearchFor.NONE)
        search_for.choices = [
            (_human_readable(x), x) for x in [
                SearchFor.ANY_RESOURCE,
                SearchFor.STABLE,
                SearchFor.TESTING,
                SearchFor.TESTING_OR_STABLE,
                SearchFor.FIXED,
                SearchFor.NONE,
            ]
        ]
        attrs = sdk2.parameters.String('{}: custom resource attributes as JSON'.format(parameter.description))
        resource = sdk2.parameters.Resource('{}: fixed resource'.format(parameter.description), resource_type=_get_resource_class(parameter.resource_type))
        ignore_to_update = sdk2.parameters.Bool('{}: updates do not start rebuild'.format(parameter.description), default=False)

        wrap_parameters = OrderedDict()
        wrap_parameters[_rs_name(name)] = search_for
        wrap_parameters[_attrs_name(name)] = attrs
        wrap_parameters[_ignore_update(name)] = ignore_to_update
        wrap_parameters[name] = resource

        search_for.sub_fields = {
            SearchFor.ANY_RESOURCE: [_attrs_name(name), _ignore_update(name)],
            SearchFor.STABLE: [_attrs_name(name), _ignore_update(name)],
            SearchFor.TESTING: [_attrs_name(name), _ignore_update(name)],
            SearchFor.TESTING_OR_STABLE: [_attrs_name(name), _ignore_update(name)],
            SearchFor.FIXED: [name],
        }

        return wrap_parameters

    def get_parameters_dict():
        parameters_dict = OrderedDict()

        for name, parameter in _extract_parameters(build_task):
            if _is_resource(parameter):
                parameters_dict.update(create_wrap_parameters(name, parameter))
            else:
                parameters_dict[name] = parameter
        return parameters_dict

    class BaseUpdate(Indexing):

        class Requirements(sdk2.Requirements):
            cores = DEFAULT_CORES
            ram = DEFAULT_RAM

            class Caches(sdk2.Requirements.Caches):
                pass

        class Parameters(sdk2.Parameters):
            with sdk2.parameters.String('Release the build to') as release_type:
                release_type.values[''] = release_type.Value(value='do not release', default=True)
                for val in ctt.ReleaseStatus:
                    release_type.values[val] = release_type.Value(value=val)
            mark_index = sdk2.parameters.String('Configuration mark of built index')
            extra = _make_parameters_cls(get_parameters_dict())
            force_parent_requirements = sdk2.parameters.Bool('Child uses parent hardware requriements', default=False)

        def _select_parameters(self, parameters, condition):
            return [
                {
                    'name': name,
                    'param': param
                }
                for name, param in parameters
                if condition(param)
            ]

        def on_execute(self):
            self.setup_indexing_params(indexing_task=build_task,
                                       index_type_for_timestamp=index_type_for_timestamp,
                                       release_subject=release_subject,
                                       force_rebuild=force_rebuild,
                                       force_parent_requirements=self.Parameters.force_parent_requirements)
            resource_params = self._select_parameters(_extract_parameters(build_task), _is_resource)
            yt_path_params = self._select_parameters(_extract_parameters(build_task), _is_yt_path)
            other_params = self._select_parameters(_extract_parameters(build_task), _is_other_param)
            self.setup_params(resource_params, yt_path_params, other_params)
            Indexing.on_execute(self)

    return BaseUpdate
