# -*- coding: utf-8 -*-
from datetime import datetime
import calendar
import email.utils
import logging
import json

import requests

# import sandbox.common.rest
import sandbox.sandboxsdk.channel as sdk_channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.copy import RemoteCopyHttp
from sandbox.sandboxsdk.copy import RemoteCopyRsync
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common.apihelpers import get_last_resource_with_attrs, list_task_dependenciens

channel = sdk_channel.channel


LAST_MODIFIED = 'last_modified'
RESULT_MARK = 'mark'


class DataUpdater:

    def __init__(self, task):
        self.task = task
        self.log_dir = self.task.log_path()

    def _get_new_resource_timestamp(self, url):
        pass

    def _get_released_resource(self, resource_type, release_status, attributes={}):
        logging.info(
            'Looking for last %s resource released to %s...', resource_type, release_status)

        attrs = attributes.copy()
        attrs['released'] = release_status
        return get_last_resource_with_attrs(resource_type, attrs=attrs, all_attrs=True)

    def _get_resource(self, resparams):
        resource_type = resparams['type']
        release_status = resparams.get('release_status', 'any_resource')

        if release_status == "fixed":
            return channel.sandbox.get_resource(resparams['fix'])

        attributes = {}
        mark = resparams.get(RESULT_MARK)
        if mark:
            attributes[RESULT_MARK] = mark

        custom_attrs = resparams.get('attrs', None)
        if custom_attrs:
            attributes.update(json.loads(custom_attrs))

        if release_status == 'any_resource':
            logging.info('Looking for last %s resource...', resource_type)
            resource = get_last_resource_with_attrs(resource_type, attrs=attributes, all_attrs=True)

        elif release_status == 'testing_or_stable':
            resource_testing = self._get_released_resource(resource_type, 'testing', attributes)
            resource_stable = self._get_released_resource(resource_type, 'stable', attributes)
            logging.info('Choose fresher resource between testing and stable')
            if not resource_testing:
                resource = resource_stable
            else:
                resource = resource_stable if resource_stable and resource_testing.timestamp <= resource_stable.timestamp else resource_testing

        else:
            resource = self._get_released_resource(resource_type, release_status, attributes)

        if resource:
            logging.info('Found resource %d', resource.id)
            return resource

        logging.info('No resource found')

    def _get_dependency_resource(self, resparams, source_type):
        target_resource = self._get_resource(
            resparams)

        # Find task that builded this resource
        # and find dependency source of this build task
        # with selected resource type
        if source_type and target_resource:
            build_task = target_resource.task_id
            real_source = list_task_dependenciens(task_id=build_task, resource_type=source_type)
            if not real_source:
                logging.warn("Not founded dependency")
                return None
            if not real_source[0].is_ready():
                logging.warn("Broken resource in dependency")
                return None
            return real_source[0]

        if target_resource and not target_resource.is_ready():
            logging.warn("Broken resource")
            return None

        return target_resource

    def _get_index_timestamp(self, resparams, source_type=None):
        logging.info("_get_index_timestamp")
        logging.info(resparams)
        logging.info(source_type)

        last_index = self._get_dependency_resource(
            resparams, source_type)

        if last_index is not None:
            last_modified = last_index.attributes.get(LAST_MODIFIED)
            if last_modified is not None:
                return int(last_modified)
            else:
                last_modified = last_index.timestamp
                return int(last_modified)

    # Create new resource, if reference is older
    # return downloaded resource or None
    def update(self,
               # dict with type and url (for download, or release status for
               # checking fresh resources)
               new_resource_parameters,
               # dict with type (resource_type) and release_status(optional)
               time_reference_resource,
               force_download=False):

        index_timestamp = None
        if not force_download:
            source_type = new_resource_parameters['type'] if not new_resource_parameters.get('no_dependency', False) else None
            index_timestamp = self._get_index_timestamp(time_reference_resource, source_type=source_type)
            if not index_timestamp:
                logging.warn("No index timestamp detected!")
                logging.warn(time_reference_resource)
            else:
                logging.info('Last index timestamp is %d', index_timestamp)
        else:
            logging.info('Forced downloading by parent')

        source_timestamp = self._get_new_resource_timestamp(
            new_resource_parameters)
        logging.info('Data source timestamp is %d', source_timestamp)

        if (index_timestamp is None) or (source_timestamp > index_timestamp):
            return self._get_new_resource(new_resource_parameters, source_timestamp)
        else:
            logging.info('No new data avaliable')
        return None


class ResourceDataUpdater(DataUpdater):

    def _get_new_resource_timestamp(self, new_resource_parameters):
        return self._get_index_timestamp(new_resource_parameters)

    def _get_new_resource(self, new_resource_parameters, timestamp):
        return self._get_resource(new_resource_parameters)

    def update_or_get_last(self,
                           # dict with type and url (for download, or release
                           # status for checking fresh resources)
                           new_resource_parameters,
                           # dict with type (resource_type) and
                           # release_status(optional)):
                           time_reference_resource
                           ):
        new_resource = self.update(
            new_resource_parameters, time_reference_resource)
        if new_resource:
            return new_resource, True

        return self._get_resource(new_resource_parameters), False


class RemoteDataUpdater(DataUpdater):

    def __init__(self, task):
        DataUpdater.__init__(self, task)
        self.timeout = None

    def _get_local_name(self, url):
        for c in reversed(url.split('/')):
            if c:
                return c
        return "undefined"

    def _get_new_resource(self, new_resource_parameters, timestamp):
        resource_type = new_resource_parameters['type']
        url = new_resource_parameters['url']
        local_path = new_resource_parameters.get('local_name', self._get_local_name(url))

        logging.info('Fetching new data...')
        copier = self.Copier(url, local_path, log_dir=self.log_dir)
        if self.timeout:
            copier(timeout=self.timeout)
        else:
            copier()

        attributes = {LAST_MODIFIED: timestamp}
        mark = new_resource_parameters.get(RESULT_MARK)
        if mark:
            attributes[RESULT_MARK] = mark

        resource = self.task.create_resource(
            self.task.descr, local_path, resource_type, attributes=attributes)
        self.task.mark_resource_ready(resource)

        return resource


class HTTPDataUpdater(RemoteDataUpdater):

    def __init__(self, task):
        RemoteDataUpdater.__init__(self, task)
        self.timeout = 60 * 60

    def _parse_http_date(self, s):
        # Parses a string in the form of "Mon, 18 Jan 2016 21:37:49 GMT"
        return calendar.timegm(email.utils.parsedate(s))

    def _get_new_resource_timestamp(self, new_resource_parameters):
        url = new_resource_parameters['url']
        logging.info('HTTP HEAD request to %s...', url)
        r = requests.head(url, allow_redirects=True, timeout=5.0)
        logging.info('Response headers: %s', r.headers)

        last_modified_header = r.headers.get('Last-Modified')
        if last_modified_header is None:
            raise SandboxTaskFailureError(
                'Could not get timestamp from response headers')

        last_modified = self._parse_http_date(last_modified_header)
        return int(last_modified)

    def update_or_get_last(self,
                           # dict with type and url (for download, or release
                           # status for checking fresh resources)
                           new_resource_parameters,
                           # dict with type (resource_type) and
                           # release_status(optional)):
                           time_reference_resource
                           ):
        new_resource = self.update(
            new_resource_parameters, time_reference_resource)
        if new_resource:
            return new_resource, True

        dep_res = self._get_dependency_resource(time_reference_resource, new_resource_parameters['type'])

        if not dep_res:
            new_resource = self.update(
                new_resource_parameters, time_reference_resource, True)

            return new_resource, True

        return dep_res, False

    Copier = RemoteCopyHttp


class RSyncDataUpdater(RemoteDataUpdater):

    def _parse_date(self, s):
        # Parses a string in the form of "2016/06/15 14:44:14"
        d = datetime.strptime(s, '%Y/%m/%d %H:%M:%S')
        return calendar.timegm(d.timetuple())

    def _get_new_resource_timestamp(self, new_resource_parameters):
        url = new_resource_parameters['url']
        logging.info('Try to run rsync with %s...', url)
        answer = run_process(['rsync', url, '--list-only'], outs_to_pipe=True).communicate()
        logging.info('Response rsync: %s', answer[0])

        splt = answer[0].strip().split()
        if len(splt) < 4:
            raise SandboxTaskFailureError(
                'Could not get timestamp from rsync response')

        last_modified_string = splt[2] + ' ' + splt[3]
        logging.info('last_modified_string = ', last_modified_string)

        last_modified = self._parse_date(last_modified_string)
        return int(last_modified)

    Copier = RemoteCopyRsync
