# -*- coding: utf-8 -*-

import os

from sandbox import common

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import LastReleasedResource, ResourceSelector
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.sandboxapi import RELEASE_STABLE
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask
from sandbox.projects.common.utils import get_or_default, set_resource_attributes


class FilterExecutableParameter(LastReleasedResource):
    name = 'filter'
    description = 'Filter executable'
    resource_type = resource_types.GEOBUSINESSDATAFILTER_EXECUTABLE


class BaseFactorsParameter(ResourceSelector):
    name = None
    description = None
    resource_type = None

    required = True

    @common.utils.classproperty
    def default_value(cls):
        return next(
            iter(channel.sandbox.server.last_resources(cls.resource_type.name)),
            {'id': None},
        )['id']


class HostFactorsParameter(BaseFactorsParameter):
    name = 'host_factors'
    description = 'Host factors'
    resource_type = resource_types.MAPS_GEO_HOST_FACTORS


class WebUserFactorsParameter(BaseFactorsParameter):
    name = 'user_factors'
    description = 'Web user factors'
    resource_type = resource_types.MAPS_WEB_USER_FACTORS


class CompanyUrlsParameter(LastReleasedResource):
    name = 'company_urls'
    description = 'Company urls'
    resource_type = resource_types.MAPS_DATABASE_BUSINESS_COMPANY_URLS


class BuildMapsFilteredFactors(ReleaseToNannyTask, SandboxTask):
    """
        Filter host or user factors by company urls from backa.
    """
    type = None
    input_parameters = [FilterExecutableParameter, CompanyUrlsParameter]

    factors_type = None           # hostfactors or userfactors
    factors_parameter = None      # HostFactorsParameter or WebUserFactorsParameter
    factors_resource_type = None  # MAPS_GEO_HOST_FACTORS_FILTERED or MAPS_WEB_USER_FACTORS_FILTERED

    def on_execute(self):
        filter_executable_path = self._get_filter_executable_path()

        factors = get_or_default(self.ctx, self.factors_parameter)
        factors_dir = self.sync_resource(factors)
        filtered_factors_dir = os.path.join(self.abs_path(), 'filtered_factors')

        company_urls_path = self.sync_resource(get_or_default(self.ctx, CompanyUrlsParameter))

        run_process([
            filter_executable_path,
            '--type', self.factors_type,
            '--factors_dir', factors_dir,
            '--company_urls', company_urls_path,
            '--output_dir', filtered_factors_dir,
        ])

        self.create_resource('Filtered {}'.format(self.factors_type),
                             filtered_factors_dir,
                             self.factors_resource_type)

        # Remove infitite ttl from previously released resources
        # to avoid storing space-consuming history.
        prev_resources = channel.sandbox.list_resources(
            resource_type=self.factors_resource_type,
            attribute_name='ttl',
            attribute_value='inf',
            limit=0,
        )
        if prev_resources is None:
            prev_resources = []
        for prev_resource in prev_resources:
            set_resource_attributes(prev_resource.id, {'ttl': '14'})

        self.create_subtask(
            task_type='RELEASE_ANY',
            inherit_notifications=True,
            input_parameters={
                'release_task_id': self.id,
                'release_status': RELEASE_STABLE,
            },
            description='{} (task id: {}) autorelease'.format(self.factors_type, self.id),
        )

    def _get_filter_executable_path(self):
        filter_executable = get_or_default(self.ctx, FilterExecutableParameter)
        filter_executable_dir = self.sync_resource(filter_executable)
        filter_executable_path = os.path.join(filter_executable_dir, 'factors_filter')
        return filter_executable_path
