# -*- coding: utf-8 -*-

import itertools
import logging
import os
import json
import six

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxUrlParameter
from sandbox.sandboxsdk.parameters import SandboxParameter
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.paths import list_dir
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types
from sandbox.projects.common.apihelpers import list_task_resources
from sandbox.projects.common.geosearch.data_update import HTTPDataUpdater
from sandbox.projects.common.geosearch.data_update import ResourceDataUpdater
from sandbox.projects.common.geosearch.data_update import LAST_MODIFIED
from sandbox.projects.common.geosearch.data_update import RESULT_MARK
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask


INDEXING_TASK_ID = 'indexing_task_id'


REBUILD_REASON = 'rebuild_reason'


class ReleaseStatusParameter(SandboxStringParameter):
    description = 'Release status'
    name = 'release_status'
    choices = [(x, x) for x in ['stable', 'testing', 'unstable']]
    default_value = 'testing'
    required = True


class NotificationParameter(SandboxStringParameter):
    description = 'Notify users'
    name = 'users_to_notify'
    required = False


class IndexMarkParameters(SandboxStringParameter):
    description = 'Configuration mark of builded index'
    name = RESULT_MARK


class Indexing(SandboxTask):
    """
        Update everything with indexing
    """

    def __init__(self,
                 task_id,
                 indexing_task_type,
                 index_type_for_timestamp,
                 release_subject="ADDRS new resource",
                 release_comments="automatic update",
                 indexing_execution_space=None,
                 inherit_notifications=False,
                 force_rebuild=False
                 ):

        self.indexing_task_type = indexing_task_type
        self.time_reference_type = index_type_for_timestamp
        self.release_subject = release_subject
        self.release_comments = release_comments
        self.indexing_execution_space = indexing_execution_space
        self.inherit_notifications = inherit_notifications
        self.force_rebuild = force_rebuild
        SandboxTask.__init__(self, task_id)

    def setup_params(self, resource_params, other_params):
        self.resource_params = resource_params
        self.other_params = other_params

        logging.info('Resource params: %d', len(resource_params))
        logging.info(resource_params)
        logging.info('Other params: %d', len(other_params))
        logging.info(other_params)

    def _add_params_to_context(self, context):
        for param in self.resource_params:
            if 'id' in param:
                context[param['name']] = param['id']
            else:
                context[param['name']] = None
                logging.warn("Build without specified resource for source")
                logging.warn(param)
        for param in self.other_params:
            if param.name in self.ctx:
                context[param.name] = self.ctx[param.name]
            else:
                logging.warning('Parameter %s was skipped because it is not defined in current task context', param.name)

    def _check_need_update(self, reference_params):
        found_smth_new = False
        logging.info('Start updating sources')

        mark = self.ctx.get(IndexMarkParameters.name)
        if mark:
            reference_params[RESULT_MARK] = mark

        for source in self.resource_params:
            logging.info('Checking %s...', source['name'])

            if source['release_status'] == 'url':
                if mark:
                    source[RESULT_MARK] = mark

                updater = HTTPDataUpdater(self)
            else:
                updater = ResourceDataUpdater(self)

            resource, updated = updater.update_or_get_last(source, reference_params)

            if not resource:
                logging.warn("No resource founded")
                logging.warn(source)
                continue

            source['id'] = resource.id
            affect_rebuild = (updated and not source.get('ignore_update', False))
            found_smth_new = found_smth_new or affect_rebuild
            logging.info('Checked: id=%s new=%r affect to rebuild=%r', resource.id, updated, affect_rebuild)

            # Record first resource, to update
            if affect_rebuild and REBUILD_REASON not in self.ctx:
                self.ctx[REBUILD_REASON] = source['name']

        logging.info('Sources, need reindex: %r', found_smth_new)
        return found_smth_new

    def begin_indexing(self):
        logging.info('Running indexer...')
        ctx = {}
        self._add_params_to_context(ctx)
        ctx['kill_timeout'] = self.ctx.get('kill_timeout')

        subtask = self.create_subtask(
            task_type=self.indexing_task_type,
            description=self.descr,
            input_parameters=ctx,
            execution_space=self.indexing_execution_space,
            inherit_notifications=self.inherit_notifications
        )
        self.ctx[INDEXING_TASK_ID] = subtask.id
        self.wait_all_tasks_stop_executing(subtask)

    def check_and_begin_indexing(self):
        release_status = self.ctx[ReleaseStatusParameter.name]
        reference_params = {'type': self.time_reference_type,
                            'release_status': release_status}

        rebuild = self._check_need_update(reference_params)
        if rebuild or self.force_rebuild:
            if self.force_rebuild:
                logging.info('Forced rebuild')
            elif rebuild:
                logging.info('Rebuild due to resource ', self.ctx.get(REBUILD_REASON, "Unknown"))
            self.begin_indexing()
        else:
            logging.info('Index need not to be updated')

    def end_indexing(self):
        subtask_id = self.ctx[INDEXING_TASK_ID]
        task = channel.sandbox.get_task(subtask_id)
        if task.new_status != self.Status.SUCCESS:
            self.notify(task, failed=True)
            raise SandboxTaskFailureError('Indexing failed')

        release_status = self.ctx[ReleaseStatusParameter.name]

        subject = self.release_subject

        mark = self.ctx.get(IndexMarkParameters.name)
        if mark:
            subject += " !!!MARK[%s]" % mark

        self.create_release(subtask_id, release_status,
                            addresses_to=self.ctx.get(NotificationParameter.name, "").split(),
                            subject=subject,
                            comments=self.release_comments)

        self.notify(task, failed=False)

    def on_execute(self):
        if INDEXING_TASK_ID not in self.ctx:
            self.check_and_begin_indexing()
        else:
            # We may not get here if no indexing process has been started
            # before.
            self.end_indexing()

    def create_email(self, indexer_log, build_task, failed):
        pass

    def notify(self, build_task, failed):
        recipients = self.ctx.get(NotificationParameter.name, '').split()
        recipients = map(lambda s: s.strip(), recipients)

        if not recipients:
            return

        logging.info('Notify users')
        logging.info(recipients)

        # Get Logs from child task
        logs = list_task_resources(build_task.id, resource_type=resource_types.TASK_LOGS)
        if len(logs) == 0:
            logging.error('No logs found at build task')
            return

        logging.info("Search for build logs")
        indexer_log = None
        for log in logs:
            log_path = self.sync_resource(log.id)
            indexers = list_dir(log_path, filter="indexer", files_only=True)
            logging.info("Found:")
            logging.info(indexers)
            if len(indexers):
                indexer_log = os.path.join(log_path, indexers[0])
                break

        if not indexer_log:
            logging.error("No logs founded")
            return

        mail = self.create_email(indexer_log, build_task, failed)
        if mail is None:
            logging.error('Cant notify with empty email')
            return

        mail_subject, mail_text = mail

        logging.info('Try to send notification')
        logging.info(mail_subject)
        logging.info(mail_text)

        channel.sandbox.send_email(recipients, '', mail_subject, mail_text)


def _human_readable(s):
    return ' '.join(s.split('_'))


def _clean_attr_map(value):
    if value:
        data = json.loads(value)
        if type(data) is not dict:
            raise ValueError('dict is expected')
        for k, v in data.iteritems():
            if not (isinstance(k, six.string_types) and (v is None or isinstance(v, six.string_types))):
                raise ValueError('both key and value must be strings (value can be null)')

        # Convert back to string. dicts are shown as "[object Object]" in UI (I love JavaScript!).
        return json.dumps(data)


def generate_base_update_task(build_task, index_type_for_timestamp, release_subject,
                              indexing_execution_space=None, inherit_notifications=False,
                              force_rebuild=False):

    def rs_name(param):
        return 'release_status_' + param.name

    def url_name(param):
        return 'url_' + param.name

    def attrs_name(param):
        return 'attrs_' + param.name

    def ignore_update_name(param):
        return "ignore_update_" + param.name

    def create_parameter(param):
        if not issubclass(param, ResourceSelector):
            return [param, ]

        class InputClass(SandboxStringParameter):
            description = 'Search for (' + param.description + ')'
            name = rs_name(param)
            choices = [
                (_human_readable(x), x) for x in [
                    'any_resource', 'stable', 'testing', 'testing_or_stable', 'url', 'fixed', 'none'
                ]
            ]
            default_value = 'none'
            required = True
            group = param.group
            ui = SandboxParameter.UI('select')

        class DataSourceUrlParameter(SandboxUrlParameter):
            description = 'URL to download ' + param.description
            name = url_name(param)
            required = True
            default_value = getattr(param, 'default_url_value', None)
            group = param.group

        class AttributesParameter(SandboxStringParameter):
            description = 'Custom resource attributes for ' + param.description + ' as JSON'
            name = attrs_name(param)
            required = False
            group = param.group

            @classmethod
            def cast(cls, value):
                return _clean_attr_map(value)

        class IgnoreForUpdate(SandboxBoolParameter):
            description = 'Updates do not start rebuild ' + param.description
            name = ignore_update_name(param)
            default_value = False
            group = param.group

        InputClass.sub_fields = {
            'any_resource': [AttributesParameter.name, IgnoreForUpdate.name],
            'stable': [AttributesParameter.name, IgnoreForUpdate.name],
            'testing': [AttributesParameter.name, IgnoreForUpdate.name],
            'testing_or_stable': [AttributesParameter.name, IgnoreForUpdate.name],
            'url': [DataSourceUrlParameter.name],
            'fixed': [param.name],
        }

        return [InputClass, param, DataSourceUrlParameter, AttributesParameter, IgnoreForUpdate]

    class BaseUpdateTask(Indexing):
        input_parameters = [ReleaseStatusParameter, IndexMarkParameters, NotificationParameter] + \
            list(itertools.chain.from_iterable([create_parameter(param) for param in build_task.input_parameters]))

        def on_execute(self):
            resource_params = [
                {
                    'name': param.name,
                    'release_status': self.ctx[rs_name(param)],
                    'type': param.resource_type,
                    'url': self.ctx.get(url_name(param), None),
                    'attrs': self.ctx.get(attrs_name(param), None),
                    'fix': self.ctx.get(param.name, None),
                    'ignore_update': self.ctx.get(ignore_update_name(param), None),
                }
                for param in build_task.input_parameters
                if issubclass(param, ResourceSelector) and self.ctx[rs_name(param)] != 'none'
            ]

            other_params = [param for param in build_task.input_parameters if not issubclass(param, ResourceSelector)]
            self.setup_params(resource_params, other_params)
            Indexing.on_execute(self)

        def __init__(self, task_id=0):

            Indexing.__init__(self,
                              task_id=task_id,
                              indexing_task_type=build_task.type,
                              index_type_for_timestamp=index_type_for_timestamp,
                              release_subject=release_subject,
                              indexing_execution_space=indexing_execution_space,
                              inherit_notifications=inherit_notifications,
                              force_rebuild=force_rebuild)

    return BaseUpdateTask


def generate_single_url_updater(index_type, release_subject, default_sources):

    class DataSourceUrlParameter(SandboxUrlParameter):
        name = 'data_source_url'
        description = 'URL to download source data'
        required = False

    class BaseUpdateTask(ReleaseToNannyTask, Indexing):

        input_parameters = (ReleaseStatusParameter, DataSourceUrlParameter)

        def __init__(self, task_id=0):
            Indexing.__init__(self,
                              task_id=task_id,
                              indexing_task_type=self.type,
                              index_type_for_timestamp=index_type,
                              release_subject=release_subject
                              )

        def on_enqueue(self):
            # Set default source url
            current_url = self.ctx.get(DataSourceUrlParameter.name, None)
            if not current_url or current_url.strip() == 'null':
                self.ctx[DataSourceUrlParameter.name] = default_sources[self.ctx[ReleaseStatusParameter.name]]

            Indexing.on_enqueue(self)

        RESOURCE_ID = DataSourceUrlParameter.name + "_id"

        def create_index_from_data(self, resource_path):
            raise SandboxTaskFailureError('Unimplemented create_index_from_data')

        def create_index(self, resource_id):
            resource_path = self.sync_resource(resource_id)
            new_path = self.create_index_from_data(resource_path)
            BuildHelper.create_index_resource(self,
                                              primary_source_resource_id=resource_id,
                                              index_directory=new_path,
                                              index_type=index_type)

        def on_execute(self):
            if BaseUpdateTask.RESOURCE_ID not in self.ctx:
                resource_params = [
                    {
                        'name': BaseUpdateTask.RESOURCE_ID,
                        'release_status': 'url',
                        'type': index_type,
                        'url': self.ctx[DataSourceUrlParameter.name],
                        'ignore_update': False,
                    }
                ]
                Indexing.setup_params(self, resource_params, [ReleaseStatusParameter, ])
                Indexing.on_execute(self)
            else:
                self.create_index(self.ctx[BaseUpdateTask.RESOURCE_ID])

    return BaseUpdateTask


class BuildHelper:

    ATRIBUTES_TO_COPY = [LAST_MODIFIED, RESULT_MARK]

    @classmethod
    def create_index_resource(cls, task, primary_source_resource_id, index_directory, index_type, additional_attributes={}):

        # copy attributes from the source resource
        attributes = {}
        if primary_source_resource_id:
            for a in BuildHelper.ATRIBUTES_TO_COPY:
                value = channel.sandbox.get_resource_attribute(primary_source_resource_id, a)

                if value is not None:
                    attributes[a] = value

        attributes.update(additional_attributes)

        resource = task.create_resource(
            task.descr, index_directory, index_type, attributes=attributes
        )

        return resource
