from sandbox import sandboxsdk
from sandbox import sdk2

import os

from datetime import datetime
from datetime import timedelta


def get_current_date():
    return datetime.now().strftime('%Y-%m-%d')


def get_dates(mode, dateFrom, dateTo, dateLast, lastNDays, nDays):
    if mode == 'last N days':
        now = datetime.now()
        dateFrom = (datetime.date(now - timedelta(days=lastNDays))).strftime('%Y-%m-%d')
        dateTo = (datetime.date(now - timedelta(days=1))).strftime('%Y-%m-%d')
    elif mode == 'N days before':
        dateTo = dateLast
        dateFrom = (datetime.date(datetime.strptime(dateLast, '%Y-%m-%d') - timedelta(days=nDays-1))).strftime('%Y-%m-%d')
    return dateFrom, dateTo


def set_environment(yt_token, yt_pool):
    assert yt_token is not None
    os.environ['MR_RUNTIME'] = 'YT'
    os.environ['YT_TOKEN'] = yt_token
    if yt_pool:
        os.environ['YT_POOL'] = yt_pool


class YtParams(object):
    ''' YT parameters '''
    class YtProxy(sandboxsdk.parameters.SandboxStringParameter):
        name = 'yt_proxy'
        description = 'YT proxy'
        default_value = 'hahn.yt.yandex.net'

    class YtTokenSecretName(sandboxsdk.parameters.SandboxStringParameter):
        name = 'yt_token_secret'
        description = 'YT token vault secret'
        required = True

    class YtPool(sandboxsdk.parameters.SandboxStringParameter):
        name = 'yt_pool'
        description = 'Yt pool'
        default_value = ''

    Content = [
        YtProxy,
        YtTokenSecretName,
        YtPool,
    ]

    def set_environment(self, yt_token, yt_pool):
        assert yt_token is not None
        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token
        if yt_pool:
            os.environ['YT_POOL'] = yt_pool


class YqlParam(object):
    ''' YQL parameter '''
    class YqlTokenSecretName(sandboxsdk.parameters.SandboxStringParameter):
        name = 'yql_token_secret'
        description = 'YQL token vault secret'
        required = True

    Content = [YqlTokenSecretName]


class DatesSelectorParams2(sdk2.Parameters):
    ''' Date selection parameters '''

    dates_selection_mode = sdk2.parameters.String(
        'Dates Selection Mode: ',
        choices=[(key, key) for key in ['interval', 'last N days', 'N days before']],
        default='interval',
    )

    with dates_selection_mode.value['interval']:
        date_from = sdk2.parameters.String('Date from', required=True)
        date_to = sdk2.parameters.String('Date to', required=True)

    with dates_selection_mode.value['last N days']:
        last_n_days = sdk2.parameters.Integer('Last N days', default=1)

    with dates_selection_mode.value['N days before']:
        date_last = sdk2.parameters.String('Date to', required=True)
        n_days = sdk2.parameters.Integer('N days', default=1)


class DatesIntervalParams(object):
    ''' Date interval parameters '''
    class DateFrom(sandboxsdk.parameters.SandboxStringParameter):
        name = 'date_from'
        description = 'Date from'
        required = True

    class DateTo(sandboxsdk.parameters.SandboxStringParameter):
        name = 'date_to'
        description = 'Date to'
        required = True

    Content = [
        DateFrom,
        DateTo,
    ]


class DatesSelectorParams(DatesIntervalParams):
    ''' Date selection parameters '''
    class DatesSelectionMode(sandboxsdk.parameters.SandboxStringParameter):
        name = 'dates_selection_mode'
        description = 'Dates Selection Mode: '
        choices = [
            ('interval', 'interval'),
            ('last N days', 'last N days'),
            ('N days before', 'N days before'),
        ]
        sub_fields = {
            'interval': [
                'date_from',
                'date_to',
            ],
            'last N days': [
                'last_n_days',
                'last_n_days_offset',
            ],
            'N days before': [
                'date_to',
                'n_days',
            ],
        }
        default_value = 'interval'

    class LastNDays(sandboxsdk.parameters.SandboxIntegerParameter):
        name = 'last_n_days'
        description = 'Last N days'
        default_value = 1

    class LastNDaysOffset(sandboxsdk.parameters.SandboxIntegerParameter):
        name = 'last_n_days_offset'
        description = 'Last N days offset'
        default_value = 1

    class NDays(sandboxsdk.parameters.SandboxIntegerParameter):
        name = 'n_days'
        description = 'N days'
        default_value = 1

    Content = [
        DatesSelectionMode,
        LastNDays,
        LastNDaysOffset,
        NDays,
    ] + DatesIntervalParams.Content

    def get_dates(self, context):
        datesSelectionMode = context.get(self.DatesSelectionMode.name)
        dateFrom = context.get(DatesIntervalParams.DateFrom.name)
        dateTo = context.get(DatesIntervalParams.DateTo.name)
        if datesSelectionMode == 'last N days':
            lastNDays = context.get(self.LastNDays.name)
            lastNDaysOffset = context.get(self.LastNDaysOffset.name)
            now = datetime.now()
            dateFrom = (datetime.date(now - timedelta(days=lastNDaysOffset + lastNDays - 1))).strftime('%Y-%m-%d')
            dateTo = (datetime.date(now - timedelta(days=lastNDaysOffset))).strftime('%Y-%m-%d')
        elif datesSelectionMode == 'N days before':
            nDays = context.get(self.NDays.name)
            dateFrom = (datetime.date(datetime.strptime(dateTo, '%Y-%m-%d') - timedelta(days=nDays-1))).strftime('%Y-%m-%d')
        return dateFrom, dateTo

    def get_gap_start(self, nGapDays, dateTo):
        return (datetime.date(datetime.strptime(dateTo, '%Y-%m-%d') - timedelta(days=nGapDays-1))).strftime('%Y-%m-%d')


class CommonFolderParam(object):
    ''' Common folder parameter '''
    class CommonFolder(sandboxsdk.parameters.SandboxStringParameter):
        name = 'common_tables_folder'
        description = 'Common tables folder'
        required = True

    Content = [CommonFolder]


class JobsCountParam(object):
    ''' Jobs count parameter '''
    class JobsCount(sandboxsdk.parameters.SandboxIntegerParameter):
        name = 'jobs_count'
        description = 'Jobs Count'
        default_value = 1

    Content = [JobsCount]


class SubJobsCountParam(object):
    ''' Jobs count parameter '''
    class SubJobsCount(sandboxsdk.parameters.SandboxIntegerParameter):
        name = 'sub_jobs_count'
        description = 'Sub Jobs Count'
        default_value = 1

    Content = [SubJobsCount]


class SourcesCheckBoxParams(object):
    ''' Sources parameters '''
    class BigYandexMaps(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'big_yandex_maps'
        description = 'Source: Big Yandex Maps'
        default_value = True

    class MobileYandexMaps(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'mobile_yandex_maps'
        description = 'Source: Mobile Yandex Maps'
        default_value = True

    class MobileYandexNavigator(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'mobile_yandex_navigator'
        description = 'Source: Mobile Yandex Navigator'
        default_value = True

    class Crypta(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'crypta'
        description = 'Source: Crypta'
        default_value = True

    class Web(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'web'
        description = 'Source: Web'
        default_value = True

    class Watch(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'watch'
        description = 'Source: Watch'
        default_value = True

    class Pool(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'pool'
        description = 'Source: Pool'
        default_value = False

    class PoolTouch(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'pool_touch'
        description = 'Source: Pool Touch'
        default_value = False

    class Density(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'density'
        description = 'Source: Density'
        default_value = True

    class Photo(sandboxsdk.parameters.SandboxBoolParameter):
        name = 'photo'
        description = 'Source: Photo'
        default_value = True

    Content = [
        BigYandexMaps,
        MobileYandexMaps,
        MobileYandexNavigator,
        Crypta,
        Web,
        Watch,
        Density,
        Photo,
    ]
