# -*- coding: utf-8 -*-

import os
import shutil
import subprocess
import logging
import time
import traceback
from contextlib import closing

from sandbox.common import errors
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.paths import get_unique_file_name
from sandbox.sandboxsdk.paths import make_folder


class SandboxAptGetUpdateError(errors.TaskError):
    """
        Ошибка если так и не удалось сделать `apt-get update`
    """
    auto_restart_task = True


def make_stub_package(pkg_name, stub_pkg_name, stubs_dir):
    """ Making stub for package in specified dir """
    stub_dir = os.path.join(stubs_dir, stub_pkg_name)
    if not os.path.exists(stub_dir):
        os.makedirs(stub_dir, 0o775)
    deb_path = os.path.join(stub_dir, 'DEBIAN')
    if os.path.exists(deb_path):
        shutil.rmtree(deb_path)
    os.makedirs(deb_path, 0o775)

    with open(os.path.join(deb_path, 'control'), 'w') as f:
        if pkg_name.endswith('virtual'):
            replaces_pkg_name = stub_pkg_name
        else:
            replaces_pkg_name = pkg_name

        txt = (
            'Package: {0}\n'
            'Source: {0}\n'
            'Architecture: all\n'
            'Version: 0.0.0-0.invalid\n'
            'Maintainer: No One <null@null.null>\n'
            'Installed-Size: 0\n'
            'Section: yandex\n'
            'Priority: optional\n'
            'Description: a stub package providing {1}.\n'
            'Provides: {1}\n'
            'Replaces: {2}\n'
            'Conflicts: {2}\n'
        ).format(stub_pkg_name, pkg_name, replaces_pkg_name)

        f.write(txt)

    cmd = 'dpkg-deb -b {0}'.format(stub_dir)
    p = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)
    stdout, stderr = p.communicate()
    if p.returncode:
        raise errors.TaskFailure("Creating stub died with exit code {0} (it says {1})".format(p.returncode, stdout))


class PackageInstallTaskParams:
    class PackageList(ResourceSelector):
        name = 'package_list_resource'
        description = 'Package list to install'
        resource_type = 'MAPS_SEARCH_PACKAGES_LIST'
        group = 'Install maps packages list parameters'
        required = True

    params = (PackageList, )


class PackageInstallTask(SandboxTask):
    """
         Abstract task to install packages before execution
    """

    def initCtx(self):
        self.ctx['__hosts_chooser_os'] = 'linux_ubuntu_10.04_lucid'

    input_parameters = PackageInstallTaskParams.params
    privileged = True

    def _exec(self, cmd, log_file):
        log_file.write('SANDBOX: Executing: {0}\n'.format(cmd))
        p = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)

        stdout, stderr = p.communicate()
        log_file.write('Stdout: << __STDOUT__ \n')
        log_file.write('{0}\n'.format(stdout))
        log_file.write('__STDOUT__\n')

        if p.returncode != 0:
            log_file.write('Died with returncode {0}\n'.format(p.returncode))
            raise errors.TaskError("Command '{0}' died with exit code {1}".format(cmd, p.returncode))
        return stdout

    def _exec_ignore_error(self, cmd, log_file):
        try:
            return self._exec(cmd, log_file)
        except errors.TaskError:
            return ""

    def _try_remove_package(self, name, log_file):
        self._exec_ignore_error('sudo apt-get -o DPkg::Options::=\"--force-confnew\" -y -f remove %s' % name, log_file)

    def _unlock_cleanup(self, log_file):
        log_file.write('Possible lock problems, try to unlock')
        self._exec('sudo rm /var/lib/apt/lists/lock', log_file)
        self._exec('sudo apt-get clean', log_file)

    def _update_packages(self, log_file):
        need_cleanup = False
        iteration_count = 9
        for i in range(iteration_count, -1, -1):
            try:
                if need_cleanup:
                    need_cleanup = False
                    self._unlock_cleanup(log_file)

                self._exec('sudo apt-get update', log_file)
            except:
                time.sleep(180)
                if i == iteration_count:
                    need_cleanup = True
                if i == 0:
                    raise SandboxAptGetUpdateError('apt-get update failed:\n' + traceback.format_exc())

    def _clean_install(self, log_file):
        try:
            self._exec("sudo apt-get -o DPkg::Options::=\"--force-confnew\" -y -f install", log_file)
        except:
            raise SandboxAptGetUpdateError('apt-get -f install failed:\n' + traceback.format_exc())

    def _install_package(self, pkg_name, log_file):
        self._exec('sudo apt-get -o DPkg::Options::=\"--force-confnew\" -y install {0} --assume-yes --force-yes '.format(pkg_name), log_file)

    def _install_stub(self, stubs_dir, stub_name, log_file):
        stub_path = os.path.join(stubs_dir, 'stub-{0}.deb'.format(stub_name))
        self._exec('sudo dpkg -i -B {0}'.format(stub_path), log_file)

    def install_packages(self):
        package_list_res_id = self.ctx[PackageInstallTaskParams.PackageList.name]
        logging.info('Installing packages from resource ' + str(package_list_res_id))
        package_list_path = self.sync_resource(package_list_res_id)

        # Prepare log file
        log_filename = get_unique_file_name(self.log_path(), 'pkg_install.log')
        log_file = open(log_filename, 'w')

        logging.info('Writing all exec info into {0}'.format(log_filename))

        stubs_dir = make_folder(os.path.join(self.abs_path(), 'stubs'))

        with closing(log_file):
            f = open(package_list_path, 'r')
            pkg_need = [line.strip() for line in f.readlines()]
            f.close()

            pkg_install_list = ""
            # Install new packages
            self._update_packages(log_file)
            # Do -f install to remove some strange situations
            # self._clean_install(log_file)
            for pkg in pkg_need:
                if pkg.startswith('cmd: '):
                    # Run command
                    cmd = pkg[len('cmd: '):]
                    self._exec(cmd, log_file)
                elif pkg.startswith('stub: '):
                    # Install stub
                    stub_name = pkg[len('stub: '):]

                    # Remove package (in the case of installed)
                    self._try_remove_package(stub_name, log_file)

                    make_stub_package(stub_name, 'stub-' + stub_name,  stubs_dir)
                    self._install_stub(stubs_dir, stub_name, log_file)
                else:
                    # Install package
                    pkg_install_list += " " + pkg
                    # self._install_package(pkg, log_file)

            # Try to install all packets by one command
            if len(pkg_install_list) != 0:
                self._install_package(pkg_install_list, log_file)
        logging.info('Finished installing packages')
