import re

from sandbox.projects.common.base_search_quality.tree import htmldiff
from sandbox.projects.common.base_search_quality.tree.node import Node


class GroupNode(htmldiff.KeyNode):
    def __init__(self):
        super(GroupNode, self).__init__("Group", ["Grouping"])

    def GetKeyValue(self, node):
        category_name = node.GetPropValue("CategoryName", required=False)
        if category_name is not None:
            return "CategoryName=" + category_name

        documents = node._nodes.get("Document", None)
        if documents and len(documents) == 1:
            return "DocId=" + documents[0].GetPropValue("DocId")

        raise ValueError('Unable to determine Group id')


class DocumentNode(htmldiff.KeyNode):
    def __init__(self):
        super(DocumentNode, self).__init__("Document", ["Grouping", "Group"])

    def GetKeyValue(self, node):
        return "Document"


class SearcherPropNode(htmldiff.KeyValueMapNode):
    KEYS_FOR_VALUES_IN_NODES = ['metadata', 'GeoClassifierFactors']

    def __init__(self):
        super(SearcherPropNode, self).__init__("SearcherProp", [])

    def GetValue(self, node):
        if node.GetPropValue("Key") in SearcherPropNode.KEYS_FOR_VALUES_IN_NODES:
            return node._nodes["Value"][0]

        return node.GetPropValue("Value")


def GetGeosearchCustomNodeTypesDict():
    return htmldiff.node_types_dict([GroupNode(), DocumentNode(), SearcherPropNode()])


def CutHostAffinity(request_text, search_source):
    new_text_lines = []
    line = request_text.replace('\\n', '\n').split('\n')[0]
    if line.find('?') >= 0:
        line = line[line.find('?') + 1:]
    new_text_lines.append(line)

    # special patch for misspell requests
    if search_source == 'Misspell':
        for i, line in enumerate(new_text_lines):
            new_text_lines[i] = re.sub(r'reqid=[^&]*',
                                       r'reqid=not_involved_in_diff',
                                       line)
    return '\n'.join(new_text_lines)


def ReplaceValueByXmlNode(SearcherProp, tags_suffixes=[]):
    value = SearcherProp.GetPropValue("Value", required=False)
    if value:
        xmlnode = _make_xml_node("<data>" + value + "</data>", tags_suffixes)
        SearcherProp._nodes["Value"] = [xmlnode]
        del SearcherProp._props["Value"]


def GeosearchResponsePatcher(response):
    # maps-search-business
    for ArchiveInfo in _get_subnodes_by_path(response, ["Grouping", "Group", "Document", "ArchiveInfo"]):
        if 'Passage' not in ArchiveInfo._props:
            continue
        values = ArchiveInfo.get_prop_values('Passage')
        xmlnodes = [_make_xml_or_json_node(tmpvalue) for tmpvalue in values]
        ArchiveInfo._nodes["Passage"] = xmlnodes
        del ArchiveInfo._props["Passage"]

        for xmlnode in xmlnodes:
            for n in _get_subnodes_by_path(xmlnode, ['[XMLData]', 'CompanyMetaData', 'internal', 'docid']):
                if '_text' not in n._props:
                    continue
                n._props['_text'] = ['"not involved in diff"']

    # geometasearch
    for SearcherProp in _get_subnodes_by_path(response, ['SearcherProp']):
        if SearcherProp.GetPropValue("Key") in ['reqid', 'serpid', 'SourceTimestamp', 'DebugProps', 'geojson_time', 'big_brother_data'] or SearcherProp.GetPropValue("Key").endswith('_vertical'):
            value = SearcherProp.GetPropValue("Value", required=False)
            if value:
                SearcherProp._props["Value"] = ['"not involved in diff"']
        elif SearcherProp.GetPropValue("Key").endswith('_requested_url'):
            value = SearcherProp.GetPropValue("Value", required=False)
            if value:
                request_to = SearcherProp.GetPropValue("Key")[:-len('_requested_url')]
                SearcherProp._props["Value"] = [CutHostAffinity(value, request_to)]
        elif SearcherProp.GetPropValue("Key") == 'SearchErrors.debug':
            value = SearcherProp.GetPropValue("Value", required=False)
            if value:
                SearcherProp._props["Value"] = [CutHostAffinity(value, '')]
        elif SearcherProp.GetPropValue("Key") == 'GeoClassifierFactors':
            ReplaceValueByXmlNode(SearcherProp, ['name'])

    paths = (
        ['ymaps', 'GeoObjectCollection', 'gml__metaDataProperty', 'ResponseMetaData', 'SearchRequest', 'internal', 'debug', 'eventlog'],
        ['ymaps', 'GeoObjectCollection', 'gml__metaDataProperty', 'ResponseMetaData', 'SearchResponse', 'InternalResponseInfo', 'reqid'],
        ['ymaps', 'GeoObjectCollection', 'gml__metaDataProperty', 'ResponseMetaData', 'SearchResponse', 'InternalResponseInfo', 'serpid'],
        ['ymaps', 'GeoObjectCollection', 'gml__metaDataProperty', 'ResponseMetaData', 'SearchResponse', 'InternalResponseInfo', 'debug', 'eventlog'],
        ['ymaps', 'GeoObjectCollection', 'gml__featureMember', 'ym__GeoObjectCollection', 'gml__metaDataProperty', 'ResponseMetaData', 'SearchRequest', 'internal', 'debug', 'eventlog'],
    )
    for path in paths:
        for node in _get_subnodes_by_path(response, path):
            if '_text' not in node._props:
                continue
            node._props['_text'] = ['"not involved in diff"']

    for node in _get_subnodes_by_path(response, ['SearcherProp']):
        if node.GetPropValue("Key") in ('EventLog', 'Ig'):
            node._props['Value'] = ['"not involved in diff"']

    # Cut host affinities
    paths = (
        ['ymaps', 'GeoObjectCollection', 'gml__metaDataProperty', 'ResponseMetaData', 'SearchRequest', 'internal'],
        ['ymaps', 'GeoObjectCollection', 'gml__metaDataProperty', 'ResponseMetaData', 'SearchResponse', 'InternalResponseInfo', 'debug'],
    )
    for path in paths:
        for node in _get_subnodes_by_path(response, path):
            for props in _get_subnodes_by_path(node, ['props']):
                props._props['_text'] = ['"not involved in diff"']
            for request in _get_subnodes_by_path(node, ['request']):
                if 'to' not in request._props:
                    continue
                request_to = request._props['to'][0]
                if '_text' not in request._props:
                    request_text = ''
                else:
                    request_text = request._props['_text'][0]

                node._props['request_to_' + request_to] = [CutHostAffinity(request_text, request_to)]
            if 'request' in node._nodes:
                del node._nodes['request']


def _make_xml_node(value, tags_suffixes=[]):
    xmlnode = Node()
    nodes = [htmldiff.convert_xml_to_tree(value, False, tags_suffixes)]
    xmlnode._nodes['[XMLData]'] = nodes
    return xmlnode


def _make_xml_or_json_node(value):
    resnode = Node()
    if value.startswith('<'):
        resnode._nodes['[XMLData]'] = [htmldiff.convert_xml_to_tree(value)]
    elif value.startswith('{'):
        resnode._nodes['[JSONData]'] = [htmldiff.json_dict_to_tree(value)]
    else:
        raise ValueError('Unable to determine format: {0!r}'.format(value))
    return resnode


def _get_subnodes_by_path(node, path):
    nodes = [node]
    for p in path:
        new_nodes = []
        for n in nodes:
            if p not in n._nodes:
                continue
            new_nodes += n._nodes[p]
        nodes = new_nodes
    return nodes
