# -*- coding: utf-8 -*-

import os
# import time
import logging
import shutil
import json

from sandbox.common.share import skynet_get as sky_get
from sandbox.projects.common.search import config as sconf
from sandbox.sandboxsdk.paths import get_unique_file_name, copy_path, make_folder, list_dir, get_logs_folder
from sandbox.sandboxsdk.svn import Arcadia, ArcadiaTestData
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.parameters import SandboxSvnUrlParameter
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.errors import SandboxTaskUnknownError
from sandbox.sandboxsdk.errors import SandboxSvnError
from sandbox.projects.common.utils import create_misc_resource_and_dir
from sandbox.projects.common.search.components import StandardSearch
from sandbox.projects.common.config_processor import get_parameter_values
from sandbox.projects.common.geosearch.utils import make_softlinks_for_libs
from sandbox.projects.geobase.Geodata5BinStable import resource as gbr

import sandbox.projects.resource_types as rt


def GetGeoBasesearchParams(ctx_suffix='', iface_group=''):
    if not iface_group:
        iface_group = 'GeoBasesearch parameters'

    class Params(object):

        class Binary(ResourceSelector):
            name = 'geobasesearch_executable_resource_id%s' % ctx_suffix
            description = 'GeoBasesearch executable'
            resource_type = 'GEOBASESEARCH_EXECUTABLE'
            group = iface_group

        class Config(ResourceSelector):
            name = 'geobasesearch_config_resource_id%s' % ctx_suffix
            description = 'GeoBasesearch config'
            resource_type = ['GEOBASESEARCH_CONFIG', 'MAPS_SEARCH_CONFIG_BASE', 'MAPS_SEARCH_CONFIG_WIKI']
            group = iface_group
            required = True

        params = (Binary, Config, )

    return Params


DEFAULT_GEOBASESEARCH_START_TIMEOUT = 10 * 60


DEFAULT_BINARY_OLD = '/usr/bin/basesearch'
DEFAULT_BINARY_NEW = '/usr/bin/geobasesearch'
# GEOCODER_BINARY = '/usr/lib/yandex/maps/search-geocoder/search-geocoder'
GEOCODER_BINARY = '/usr/lib/yandex/maps/yacare-wrapper/run-yacare-service.sh'


class GeoBasesearch(StandardSearch):
    """
        Представление базового поиска для геоисточников
    """

    name = 'geobasesearch'
    http_collection = ''

    _LOAD_LOG_FILENAME = 'geobasesearch_{}_load.log'
    _SERVER_LOG_FILENAME = 'geobasesearch_%s_server.log'

    def __init__(self, work_dir, config_file, binary, database_dir, port,
                 config_class=sconf.SearchConfig, config_params=None,
                 use_profiler=False,
                 start_timeout=DEFAULT_GEOBASESEARCH_START_TIMEOUT,
                 default_binary=None
                 ):
        if not binary:
            binary = default_binary

        if not config_params:
            config_params = {}
        # config_params.update({
        #     'Server/LoadLog': os.path.join(work_dir, self._LOAD_LOG_FILENAME.format(port)),
        #     'Server/ServerLog': os.path.join(work_dir, self._SERVER_LOG_FILENAME % port),
        # })

        super(GeoBasesearch, self).__init__(work_dir, binary, port, config_class, config_file,
                                            config_params, use_profiler,
                                            start_timeout=start_timeout)

        original_database_dir = self.config.get_parameter(
            'Collection/IndexDir')
        if not original_database_dir or 'fake-index' not in original_database_dir:
            self.config.apply_local_patch({
                'Collection/IndexDir': database_dir
            })

    def _get_run_cmd(self, config_path):
        return ['-d', '-p', str(self.port), config_path]

    def get_loadlog(self):
        return os.path.join(self.work_dir, self._LOAD_LOG_FILENAME.format(self.port))

    # def _stop_impl(self):
    #     self.process.kill()
    #     time.sleep(1)  # need some time to die and release port


def get_geobasesearch_params(ctx, params):
    binary_path = None
    if ctx.get(params.Binary.name):
        binary_path = channel.task.sync_resource(ctx[params.Binary.name])

    config_path = channel.task.sync_resource(ctx[params.Config.name])

    return {
        'binary': binary_path,
        'config_file': config_path,
    }


DEFAULT_MAPS_SEARCH_BUSINESS_PORT = 17083

# New PPO (arcadia only)


class MapsSearchBusiness(GeoBasesearch):
    """
        Представление ППО на базе geobasesearch
    """

    name = 'geobasesearch_business'
    _LOAD_LOG_FILENAME = 'geobasesearch_business{}_load.log'
    _SERVER_LOG_FILENAME = 'geobasesearch_business%s_server.log'
    http_collection = ''

    def __init__(self, work_dir, geobasesearch_params,
                 business_database_dir, advert_database_path, fast_features_path=None,
                 filter_stat_path=None, social_profiles_path=None,
                 port=DEFAULT_MAPS_SEARCH_BUSINESS_PORT,
                 config_class=sconf.SearchConfig, config_params=None,
                 use_profiler=False):
        self._check_if_exists(
            business_database_dir, 'business database folder')
        self._check_if_exists(advert_database_path, 'advert database file')
        self._check_if_exists(advert_database_path, 'advert database file')

        super(MapsSearchBusiness, self).__init__(
            work_dir=work_dir,
            database_dir=business_database_dir,
            port=port,
            default_binary=DEFAULT_BINARY_NEW,
            **geobasesearch_params
        )

        self.work_dir = work_dir

        patch = {
            'Collection/IndexDir': business_database_dir,
            # 'Collection/UserParams/Adverts': advert_database_path + '/advert.mms',
            # 'Collection/UserParams/ExperimentalAdverts': advert_database_path + '/experimental',
            # 'Collection/UserParams/PageIdAdverts': advert_database_path + '/page_id_advert.mms',
            # 'Collection/UserParams/MenuAdverts': advert_database_path + '/menu_advert.mms',
            'Collection/UserParams/AdvertsPb': advert_database_path + '/advert.pb.bin',
            'Collection/UserParams/MenuAdvertsPb': advert_database_path + '/menu_advert.pb.bin',
            'Collection/UserParams/Bundle': "",
            'Collection/UserParams/WebIndexannDir': business_database_dir,
            'Collection/UserParams/VirtualShards': '1',
        }

        if fast_features_path:
            patch.update({'Collection/UserParams/FastFeatures': fast_features_path, })
        else:
            patch.update({'Collection/UserParams/FastFeatures': ''})
        if filter_stat_path:
            patch.update({'Collection/UserParams/PrecomputedFilters': filter_stat_path, })

        self.config.apply_local_patch(patch)

    def get_loadlog(self):
        return os.path.join(self.work_dir, self._LOAD_LOG_FILENAME.format(self.port))

    def get_environment(self):
        return {'LD_LIBRARY_PATH': self.work_dir}


def GetMapsSearchBusinessParams(ctx_suffix=''):
    iface_group = 'Maps Search Business parameters'
    GeoBasesearchParams = GetGeoBasesearchParams(ctx_suffix=ctx_suffix,
                                                 iface_group=iface_group)

    class Params(GeoBasesearchParams):

        class BusinessDatabase(ResourceSelector):
            name = 'maps_search_business_db_resource_id%s' % ctx_suffix
            description = 'Business database'
            resource_type = 'MAPS_DATABASE_BUSINESS'
            group = iface_group
            required = True

        class AdvertDatabase(ResourceSelector):
            name = 'maps_search_advert_db_resource_id%s' % ctx_suffix
            description = 'Advert database'
            resource_type = 'MAPS_DATABASE_ADVERT'
            group = iface_group
            required = True

        params = GeoBasesearchParams.params + \
            (BusinessDatabase, AdvertDatabase)

    return Params


def get_maps_search_business(
    ctx,
    params=GetMapsSearchBusinessParams(),
    use_profiler=False,
    port=DEFAULT_MAPS_SEARCH_BUSINESS_PORT
):
    """
        Синхронизировать ресурсы, поправить библиотечные конфиги и получить обёртку
        :return: объект MapsSearchBusinessNew
    """

    geobasesearch_params = get_geobasesearch_params(ctx, params)

    business_database_path = channel.task.sync_resource(
        ctx[params.BusinessDatabase.name])
    advert_database_path = channel.task.sync_resource(
        ctx[params.AdvertDatabase.name])

    return MapsSearchBusiness(
        work_dir=channel.task.abs_path(),
        geobasesearch_params=geobasesearch_params,

        business_database_dir=business_database_path,
        advert_database_path=advert_database_path,

        port=port,
        use_profiler=use_profiler,
    )


def get_addrs_basesearch_params(ctx_suffix=''):
    iface_group = 'Addrs Basesearch parameters'
    GeoBasesearchParams = GetGeoBasesearchParams(ctx_suffix=ctx_suffix,
                                                 iface_group=iface_group)

    class Params(GeoBasesearchParams):

        # class Database(ResourceSelector):
        #     name = 'addrs_basesearch_db_resource_id%s' % ctx_suffix
        #     description = 'Addrs Basesearch database'
        #     resource_type = ('ADDRS_BUSINESS_SHARD', 'MAPS_DATABASE_BUSINESS')
        #     group = iface_group
        #     required = True

        class Database(SandboxStringParameter):
            name = 'shard_rbtorrent'
            description = 'Addrs Basesearch shard rbtorrent'
            group = iface_group
            required = True

        class AdvertDatabase(ResourceSelector):
            name = 'maps_search_advert_db_resource_id%s' % ctx_suffix
            description = 'Advert database'
            resource_type = 'MAPS_DATABASE_ADVERT'
            group = iface_group
            required = True

        class FastFeaturesSharded(ResourceSelector):
            name = 'fast_features_sharded_resource_id%s' % ctx_suffix
            description = 'Fast features (sharded)'
            resource_type = 'MAPS_DATABASE_FAST_FEATURES_SHARD'
            group = iface_group
            required = False

        class FastFeaturesAllShards(ResourceSelector):
            name = 'fast_features_all_shards_resource_id%s' % ctx_suffix
            description = 'Fast features all shards'
            resource_type = 'MAPS_DATABASE_FAST_FEATURES_SHARDED'
            group = iface_group
            required = False

        class FilterStat(ResourceSelector):
            name = 'filter_stat_resource_id%s' % ctx_suffix
            description = 'filter_stat.bin'
            resource_type = 'BUSINESS_COMPUTED_FILTERS'
            group = iface_group
            required = True

        params = GeoBasesearchParams.params + \
            (Database, AdvertDatabase, FastFeaturesSharded, FastFeaturesAllShards, FilterStat)

    return Params


def get_fast_features_path(resource_path):
    result = None
    for root, dirs, files in os.walk(resource_path):
        for file in files:
            if '.mms' in file:
                return os.path.abspath(root)
            result = os.path.abspath(root)
    return result


def get_addrs_basesearch_business(
    ctx,
    params=get_addrs_basesearch_params(),
    use_profiler=False,
    port=DEFAULT_MAPS_SEARCH_BUSINESS_PORT
):
    """
        Синхронизировать ресурсы, поправить библиотечные конфиги и получить обёртку
        :return: объект MapsSearchBusinessNew
    """

    geobasesearch_params = get_geobasesearch_params(ctx, params)

    if str(ctx[params.Database.name]).isalnum():     # Sandbox resource
        business_database_path = channel.task.sync_resource(
            ctx[params.Database.name])
    elif str(ctx[params.Database.name]).startswith('rbtorrent'):
        shard_dir = './'
        sky_get(ctx[params.Database.name], shard_dir)
        business_database_path = os.path.join(shard_dir,
                                              'index_0000000000_0000000000')
    advert_database_path = channel.task.sync_resource(
        ctx[params.AdvertDatabase.name])

    fast_features_path = None
    if ctx[params.FastFeaturesAllShards.name]:
        fast_features_path = get_fast_features_path(channel.task.sync_resource(ctx[params.FastFeaturesAllShards.name]))
    if ctx[params.FastFeaturesSharded.name]:
        fast_features_path = channel.task.sync_resource(ctx[params.FastFeaturesSharded.name])

    filter_stat_path = channel.task.sync_resource(
        ctx[params.FilterStat.name])

    return MapsSearchBusiness(
        work_dir=channel.task.abs_path(),
        geobasesearch_params=geobasesearch_params,

        business_database_dir=business_database_path,
        advert_database_path=advert_database_path,
        fast_features_path=fast_features_path,
        filter_stat_path=filter_stat_path,

        port=port,
        use_profiler=use_profiler,
    )


DEFAULT_MAPS_SEARCH_GEOCODER_PORT = 17141
DEFAULT_MAPS_SEARCH_WIKICODER_PORT = 17145


# New geocoder
class MapsSearchGeocoder(GeoBasesearch):
    """
        Представление геокодера на базе geobasesearch
    """

    name = 'geobasesearch_geocoder'
    _LOAD_LOG_FILENAME = 'geobasesearch_geocoder{}_load.log'
    _SERVER_LOG_FILENAME = 'geobasesearch_geocoder%s_server.log'
    http_collection = 'maps'

    def __init__(self, work_dir, geobasesearch_params,
                 geocoder_config_file,
                 shlib_relevance_minimal_path,
                 index_description,
                 port=DEFAULT_MAPS_SEARCH_GEOCODER_PORT,
                 config_class=sconf.SearchConfig, config_params=None,
                 use_profiler=False):
        self.name += '_' + str(port)

        self.index_description = index_description

        super(MapsSearchGeocoder, self).__init__(
            work_dir=work_dir,
            database_dir='',
            port=port,
            default_binary=GEOCODER_BINARY,
            **geobasesearch_params
        )

        self._check_if_exists(geocoder_config_file, 'geocoder config')

        self.work_dir = work_dir
        lib_paths = [shlib_relevance_minimal_path, ]
        make_softlinks_for_libs(
            work_dir, lib_paths, '/usr/lib/libyandex-maps-search-geocoder.so')

    def _get_run_cmd(self, config_path):
        return ['/usr/lib/yandex/maps/search-geocoder/search-geocoder',
                str(self.port),
                get_logs_folder() + '/geocoder_nginx.log',
                '--log-file',
                get_logs_folder() + '/geocoder.log',
                '--index-description', self.index_description]

    def get_loadlog(self):
        return os.path.join(self.work_dir, self._LOAD_LOG_FILENAME.format(self.port))

    def get_environment(self):
        return {'LD_LIBRARY_PATH': self.work_dir}


def GetMapsSearchGeocoderParamsBase(type, ctx_suffix=''):

    if type == 'geo':
        ctx_type = 'geo'
        iface_type = 'Geo'
    elif type == 'wiki':
        ctx_type = 'wiki'
        iface_type = 'Wiki'
    else:
        raise SandboxTaskUnknownError(
            'Wrong type for GetMapsSearchGeocoderParamsBase: %s' % type)

    iface_group = 'Maps Search %scoder parameters' % (iface_type)
    GeoBasesearchParams = GetGeoBasesearchParams(
        ctx_suffix=ctx_suffix,
        iface_group=iface_group,
    )

    class Params(GeoBasesearchParams):

        class GeoConfig(ResourceSelector):
            name = 'maps_search_config_%scoder%s' % (ctx_type, ctx_suffix)
            description = '%scoder config' % (iface_type)
            resource_type = 'MAPS_SEARCH_CONFIG_GEOCODER'
            group = iface_group
            required = True

        class IndexDescription(ResourceSelector):
            name = 'maps_search_%scoder_index_description_resource_id%s' % (
                ctx_type, ctx_suffix)
            description = '%scoder index description' % (iface_type)
            resource_type = 'MAPS_DATABASE_GEOCODER_METAINFO'
            group = iface_group
            required = True

        # class Database(ResourceSelector):
        #     name = 'maps_search_%scoder_db_resource_id%s' % (
        #         ctx_type, ctx_suffix
        #     )
        #     description = '%scoder database' % (iface_type)
        #     resource_type = 'MAPS_DATABASE_GEOCODER'
        #     group = iface_group
        #     required = True

        class ShlibRelevanceMinimal(ResourceSelector):
            name = 'maps_shlib_relevance_minimal%s' % ctx_suffix
            description = 'Shlib relevance-minimal'
            resource_type = 'MAPS_SHLIB_RELEVANCE_MINIMAL'
            group = iface_group

        params = GeoBasesearchParams.params + (
            GeoConfig,
            IndexDescription,
            # Database,
            ShlibRelevanceMinimal,
        )

    return Params


def GetMapsSearchGeocoderParams(*args, **kwargs):
    return GetMapsSearchGeocoderParamsBase('geo', *args, **kwargs)


def GetMapsSearchWikicoderParams(*args, **kwargs):
    return GetMapsSearchGeocoderParamsBase('wiki', *args, **kwargs)


def patch_yacare_wrapper():
    yacare_wrapper_path = '/usr/lib/yandex/maps/yacare-wrapper/run-yacare-service.sh'
    with open(yacare_wrapper_path, 'r') as f:
        yacare_wrapper_lines = f.readlines()
    new = []
    for line in yacare_wrapper_lines:
        if 'RUN_DIR=' in line:
            line = 'RUN_DIR=./\n'
        new.append(line)
    with open(yacare_wrapper_path, 'w') as f:
        f.writelines(new)


def patch_geocoder_index_description(index_description_path):
    patched_index_description_path = './patched_index_description.json'
    desc_file = open(index_description_path)
    description = json.load(desc_file)
    russian_region = description['regions']['russia']
    del(description['regions'])
    description.update({'regions': {'russia': russian_region}})
    with open(patched_index_description_path, 'w') as output:
        json.dump(description, output)
    return patched_index_description_path


def get_geocoder_indexes_from_description(index_description_path):
    desc_file = open(index_description_path)
    description = json.load(desc_file)
    for region in description['regions'].itervalues():
        temp_path = channel.task.sync_resource(region['sandbox_resource_id'])
        copy_path(temp_path, region['path'] + '/index')
        # os.symlink(path, region['path'] + '/index')
    with open(index_description_path, 'w') as output:
        json.dump(description, output)


def get_maps_search_geocoder_base(
    ctx,
    index_description,
    params=GetMapsSearchGeocoderParams(),
    use_profiler=False,
    port=DEFAULT_MAPS_SEARCH_GEOCODER_PORT,
):
    """
        Синхронизировать ресурсы, поправить библиотечные конфиги и получить обёртку
        :return: объект MapsSearchGeocoder
    """
    geobasesearch_params = get_geobasesearch_params(ctx, params)
    geocoder_config_path = channel.task.sync_resource(
        ctx[params.GeoConfig.name])
    shlib_relevance_minimal_path = None
    if ctx.get(params.ShlibRelevanceMinimal.name):
        shlib_relevance_minimal_path = channel.task.sync_resource(
            ctx[params.ShlibRelevanceMinimal.name])

    return MapsSearchGeocoder(
        work_dir=channel.task.abs_path(),
        geobasesearch_params=geobasesearch_params,
        geocoder_config_file=geocoder_config_path,
        shlib_relevance_minimal_path=shlib_relevance_minimal_path,
        index_description=index_description,
        port=port,
        use_profiler=use_profiler,
    )


def get_maps_search_geocoder(
    ctx,
    params=GetMapsSearchGeocoderParams(),
    port=DEFAULT_MAPS_SEARCH_GEOCODER_PORT,
    **kwargs
):
    return get_maps_search_geocoder_base(
        ctx,
        params=params,
        port=port,
        **kwargs
    )


def get_maps_search_wikicoder(
    ctx,
    params=GetMapsSearchWikicoderParams(),
    port=DEFAULT_MAPS_SEARCH_WIKICODER_PORT,
    **kwargs
):
    return get_maps_search_geocoder_base(
        ctx,
        params=params,
        port=port,
        **kwargs
    )


# new wiki
DEFAULT_MAPS_SEARCH_WIKI_PORT = 11293

# Arcadia only wiki


class MapsSearchWiki(GeoBasesearch):
    """
        Представление источника Народных Карт на базе basesearch
    """

    name = 'geobasesearch_wiki'
    _LOAD_LOG_FILENAME = 'geobasesearch_wiki{}_load.log'
    _SERVER_LOG_FILENAME = 'geobasesearch_wiki%s_server.log'
    http_collection = 'wiki'

    def __init__(self, work_dir, geobasesearch_params,
                 wiki_database_dir,
                 port=DEFAULT_MAPS_SEARCH_WIKI_PORT,
                 config_class=sconf.SearchConfig, config_params=None,
                 use_profiler=False):

        self._check_if_exists(wiki_database_dir, 'wiki database folder')

        super(MapsSearchWiki, self).__init__(
            work_dir=work_dir,
            database_dir=wiki_database_dir,
            port=port,
            default_binary=DEFAULT_BINARY_NEW,
            **geobasesearch_params
        )

        self.work_dir = work_dir

        self.config.apply_local_patch({
            'Collection/IndexDir': wiki_database_dir,
            'Collection/UserParams/Bundle': "",
        })

    def get_loadlog(self):
        return os.path.join(self.work_dir, self._LOAD_LOG_FILENAME.format(self.port))

    def get_environment(self):
        return {'LD_LIBRARY_PATH': self.work_dir}


def GetMapsSearchWikiParams(ctx_suffix=''):
    iface_group = 'Maps Search Wiki parameters'
    # GeoBasesearchParams = GetGeoBasesearchParams(
    #     ctx_suffix=ctx_suffix,
    #     iface_group=iface_group
    # )
    GeoBasesearchParams = get_addrs_basesearch_params(ctx_suffix=ctx_suffix)

    class Params(GeoBasesearchParams):

        class WikiConfig(ResourceSelector):
            name = 'maps_search_wiki_config_resource_id%s' % ctx_suffix
            description = 'Wikisearch config'
            resource_type = 'MAPS_SEARCH_CONFIG_WIKI'
            group = iface_group
            required = True

        class WikiDatabase(ResourceSelector):
            name = 'maps_search_wiki_db_resource_id%s' % ctx_suffix
            description = 'Wikisearch database'
            resource_type = 'MAPS_DATABASE_POI'
            group = iface_group
            required = True

        params = GeoBasesearchParams.params + \
            (WikiDatabase,
             WikiConfig)

    return Params


def get_maps_search_wiki(
    ctx,
    params=GetMapsSearchWikiParams(),
    use_profiler=False,
    port=DEFAULT_MAPS_SEARCH_WIKI_PORT
):
    """
        Синхронизировать ресурсы, поправить библиотечные конфиги и получить обёртку
        :return: объект MapsSearchWiki
    """

    geobasesearch_params = get_geobasesearch_params(ctx, params)

    wiki_database_path = channel.task.sync_resource(
        ctx[params.WikiDatabase.name])

    return MapsSearchWiki(
        work_dir=channel.task.abs_path(),
        geobasesearch_params=geobasesearch_params,
        wiki_database_dir=wiki_database_path,
        port=port,
        use_profiler=use_profiler
    )


DEFAULT_GEOMETASEARCH_UPPER_PORT = 8031
DEFAULT_GEOMETASEARCH_MIDDLE_PORT = 8032


class GeoMetaSearchBase(StandardSearch):
    """
        Представление геометапоиска в Sandbox
    """
    name = 'geometasearch'

    _LOAD_LOG_FILENAME = 'geometasearch{}_load.log'
    _SERVER_LOG_FILENAME = 'geometasearch%s_server.log'
    _EVENT_LOG_FILENAME = 'geometasearch%s_event.log'

    memory_limit = 20000000

    def __init__(self,
                 is_upper=None,
                 work_dir=None,
                 binary=None,
                 searchsources={},
                 auxsources={},
                 config_file=None,
                 rearrange_dir=None,
                 wizard=None,
                 event_log=None,
                 load_log=None,
                 port=None,
                 use_few_threads=None,
                 disable_timeouts=None,
                 reqans_log_resource=None,
                 output_resource=None):
        if event_log is None:
            event_log_dir = create_misc_resource_and_dir(channel.task.ctx, 'eventlog_resource_id',
                                                         'event log', 'event_log')
            self.event_log_path = get_unique_file_name(
                event_log_dir, self._EVENT_LOG_FILENAME % port)
        else:
            self.event_log_path = event_log

        if load_log is None:
            load_log_path = os.path.join(
                work_dir, self._LOAD_LOG_FILENAME.format(port))
        else:
            load_log_path = load_log

        self.output_path = output_resource.path

        config_params = {
            'Server/Port': str(port),
            'Server/LoadLog': load_log_path,
            'Server/ServerLog': os.path.join(work_dir, self._SERVER_LOG_FILENAME % port),
            'Server/EventLog': self.event_log_path,
            'Collection/RearrangeDataDir': rearrange_dir,
        }

        # if wizard:
        #    config_params.update({
        #        'Collection/RemoteWizards': wizard,
        #    })

        self.reqans_log_path = None
        if is_upper:
            self.reqans_log_path = reqans_log_resource.path
            config_params.update({
                'Server/ReqAnsLog': self.reqans_log_path,
            })

        if use_few_threads:
            config_params.update({
                'Server/Threads': '2',
                'Collection/RequestThreads': '4',
                'Collection/ReAskThreads': '1',
            })

        config_class = sconf.SearchConfig
        super(GeoMetaSearchBase, self).__init__(
            work_dir, binary, port, config_class, config_file, config_params)

        if disable_timeouts:
            self._patch_disable_timeouts(searchsources)
            self._patch_disable_timeouts(auxsources)

        # self._patch_sources(
        #    searchsources, disable_timeouts=disable_timeouts, key="SearchSource")

        # self._patch_sources(
        #    auxsources, disable_timeouts=disable_timeouts, key="AuxSource")

    def start(self):
        self._truncate_reqans_log()
        super(GeoMetaSearchBase, self).start()

    def stop(self):
        out_file_name = self.process.stdout_path
        try:
            super(GeoMetaSearchBase, self).stop()
        finally:
            shutil.copyfile(out_file_name, self.output_path)

    def _get_run_cmd(self, config_path):
        return ['-d', '-p', self.port, config_path]

    def get_event_log_path(self):
        return self.event_log_path

    def _truncate_reqans_log(self):
        if not self.reqans_log_path:
            return
        with open(self.reqans_log_path, 'w') as f:
            f.truncate()

    def _patch_sources(self, searchsources, disable_timeouts=False, key="SearchSource"):
        config_params = {}
        search_source_names = get_parameter_values(
            self.config.text, 'Collection/{0}/ServerDescr'.format(key))
        search_sources_options = get_parameter_values(
            self.config.text, 'Collection/{0}/Options'.format(key))
        searchsources_patched = set()
        for pos, descr in enumerate(search_source_names):
            if descr not in searchsources:
                continue
            if descr in searchsources_patched:
                continue

            if disable_timeouts:
                cgi_prefix = (searchsources[descr] + ' ') * 10
            else:
                cgi_prefix = searchsources[descr]

            param_name = 'Collection/{1}/CgiSearchPrefix[{0}]'.format(
                pos, key)
            config_params[param_name] = cgi_prefix

            if disable_timeouts:
                new_options_list = []
                for option in search_sources_options[pos].split(','):
                    if (
                        option.startswith('TimeOut=') or
                        option.startswith('MaxAttempts=')
                    ):
                        continue
                    new_options_list.append(option)

                new_options_list.append('TimeOut=300s')
                new_options_list.append('MaxAttempts=10')

                param_name = 'Collection/{1}/Options[{0}]'.format(pos, key)
                config_params[param_name] = ','.join(new_options_list)

            searchsources_patched.add(descr)

        self.config.apply_local_patch(config_params)

    def _patch_disable_timeouts(self, searchsources):
        config_params = {
            'Collection/ScatterOptions/TimeoutTable': '300s',
            'Collection/ScatterOptions/MessengerOptions': None,
            'Collection/ScatterOptions/RemoteTimeoutOptions': None,
            'Collection/ConnectTimeout': '300s',
            'Collection/MessengerOptions': None,
            'Collection/TimeoutTable': '300s',
            'Collection/UseTimeoutTable': None,
            'Collection/WizardTimeout': '300000',
        }

        self.config.apply_local_patch(config_params)


class GeoMetaSearchMiddle(GeoMetaSearchBase):
    """
        Представление среднего геометапоиска в Sandbox
    """
    name = 'geometasearch_middle'

    def __init__(self, *args, **kwargs):
        super(GeoMetaSearchMiddle, self).__init__(
            *args, is_upper=False, **kwargs)


class GeoMetaSearchUpper(GeoMetaSearchBase):
    """
        Представление верхнего геометапоиска в Sandbox
    """
    name = 'geometasearch_upper'

    def __init__(self, *args, **kwargs):
        super(GeoMetaSearchUpper, self).__init__(
            *args, is_upper=True, **kwargs)


def GetGeoMetaSearchParams(search_type=None):
    group_descr = 'GeoMetaSearch %s parameters' % search_type

    class Params:

        class Binary(ResourceSelector):
            name = 'geometasearch_%s_executable_resource_id' % search_type
            description = 'Executable'
            resource_type = rt.GEOMETASEARCH_EXECUTABLE
            group = group_descr

        class ConfigRes(ResourceSelector):
            name = 'geometasearch_%s_config_resource_id' % search_type
            description = 'Config'
            resource_type = rt.GEOMETASEARCH_CONFIG
            group = group_descr

        class BinarySvnPath(SandboxArcadiaUrlParameter):
            name = 'geometasearch_%s_binary_svn_path' % search_type
            description = 'Svn path binary built from: '
            required = False
            group = group_descr

        class TestsDataArcadiaUrl(SandboxSvnUrlParameter):
            name = 'geometasearch_%s_arcadia_tests_data_url' % search_type
            description = 'Arcadia tests data svn url: '
            required = False
            group = group_descr

        class ConfigType(SandboxStringParameter):
            name = 'geometasearch_%s_config_type' % search_type
            description = 'Config type'
            choices = [
                ('From resource', 'from_resource'),
                ('From svn', 'from_svn'),
                ('From arcadia_tests_data', 'from_arcadia_tests_data')
            ]

            default_value = 'from_resource'
            required = True
            group = group_descr
        ConfigType.sub_fields = {
            'from_resource': [ConfigRes.name],
            'from_svn': [BinarySvnPath.name],
            'from_arcadia_tests_data': [TestsDataArcadiaUrl.name],
        }

        class DataRes(ResourceSelector):
            name = 'geometasearch_%s_data_resource_id' % search_type
            description = 'Geometasearch rearrange data'
            resource_type = rt.MIDDLESEARCH_GEO_DATA
            required = True
            group = group_descr

        class GeodataRes(ResourceSelector):
            name = 'geometasearch_%s_geodata_id' % search_type
            description = 'Geodata resource'
            resource_type = gbr.GEODATA5BIN_STABLE
            required = True
            group = group_descr

        class UseFewThreads(SandboxBoolParameter):
            name = 'geometasearch_%s_use_few_threads' % search_type
            description = 'Use small number of threads'
            group = group_descr
            default_value = False

        class DisableTimeouts(SandboxBoolParameter):
            name = 'geometasearch_%s_disable_timeouts' % search_type
            description = 'Disable timeouts'
            group = group_descr
            default_value = False

        params = (
            Binary,
            ConfigType,
            ConfigRes,
            BinarySvnPath,
            TestsDataArcadiaUrl,
            DataRes,
            GeodataRes,
            UseFewThreads,
            DisableTimeouts,
        )

    return Params


GeoMetaSearchMiddleParams = GetGeoMetaSearchParams(search_type='middle')
GeoMetaSearchUpperParams = GetGeoMetaSearchParams(search_type='upper')

UPPER_CONFIG_ARCADIA_PATH = 'search/scripts/gencfg/addrs/generated/sandboxhost--8031.cfg'
MIDDLE_CONFIG_ARCADIA_PATH = 'search/scripts/gencfg/addrs/generated/sandboxhost--8032.cfg'

UPPER_CONFIG_ARCADIA_TESTS_DATA_PATH = 'geosearch/sandbox_test_data/geometasearch/sandboxhost--8031.cfg'
MIDDLE_CONFIG_ARCADIA_TESTS_DATA_PATH = 'geosearch/sandbox_test_data/geometasearch/sandboxhost--8032.cfg'


def get_geometasearch_params_from_ctx(task=None, search_type=None, params=None):
    binary_res_id = task.ctx.get(params.Binary.name)
    binary_path = channel.task.sync_resource(binary_res_id)

    config_path = None
    config_type = task.ctx.get(params.ConfigType.name)
    if config_type == 'from_resource':
        config_res_id = task.ctx.get(params.ConfigRes.name)
        config_path = channel.task.sync_resource(config_res_id)
    elif config_type == 'from_svn':
        svn_path = task.ctx.get(params.BinarySvnPath.name)
        svn_subpath = {'upper': UPPER_CONFIG_ARCADIA_PATH,
                       'middle': MIDDLE_CONFIG_ARCADIA_PATH}.get(search_type)
        new_path = os.path.join(Arcadia.parse_url(svn_path).path, svn_subpath)
        svn_url = Arcadia.replace(svn_path, path=new_path)
        logging.info('Geometasearch %s config svn path: %s' %
                     (search_type, svn_url))
        config_path = task.abs_path(os.path.basename(svn_subpath))
        try:
            Arcadia.info(svn_url)
        except SandboxSvnError as e:
            raise SandboxTaskFailureError('%s' % e)
        Arcadia.export(svn_url, config_path)

        logging.info('Exported %s config to %s' % (search_type, config_path))
    elif config_type == 'from_arcadia_tests_data':
        tests_data_url = task.ctx.get(params.TestsDataArcadiaUrl.name)
        svn_subpath = {'upper': UPPER_CONFIG_ARCADIA_TESTS_DATA_PATH,
                       'middle': MIDDLE_CONFIG_ARCADIA_TESTS_DATA_PATH}.get(search_type)
        new_path = os.path.join(Arcadia.parse_url(tests_data_url).path, svn_subpath)
        svn_url = Arcadia.replace(tests_data_url, path=new_path)
        logging.info('Geometasearch %s config svn path: %s' %
                     (search_type, svn_url))
        config_path = ArcadiaTestData.get_arcadia_test_data(
            channel.task, svn_url)

    if not config_path:
        raise SandboxTaskFailureError(
            'Cannot get config from params group "%s"' % params.ConfigType.group)

    # Prepare rearrange data dir
    data_res_id = task.ctx.get(params.DataRes.name)
    rearrange_dir = channel.task.sync_resource(data_res_id)

    geodata_res_id = task.ctx.get(params.GeodataRes.name)
    if geodata_res_id:
        geodata = channel.task.sync_resource(geodata_res_id)
        new_data = 'fake_geometadata_%s' % search_type
        new_data_folder = make_folder(os.path.join(new_data, 'geosearch'))
        for filename in list_dir(os.path.join(rearrange_dir, 'geosearch'), abs_path=True):
            if "geodata" not in filename:
                copy_path(filename, new_data_folder)
        copy_path(geodata, new_data_folder)

        rearrange_dir = new_data

    use_few_threads = task.ctx.get(params.UseFewThreads.name, False)
    disable_timeouts = task.ctx.get(params.DisableTimeouts.name, False)

    return {
        'binary': binary_path,
        'config_file': config_path,
        'rearrange_dir': rearrange_dir,
        'use_few_threads': use_few_threads,
        'disable_timeouts': disable_timeouts,
    }


def get_geometasearch_middle(task,
                             params=GeoMetaSearchMiddleParams,
                             searchsources=None,
                             wizard=None,
                             port=DEFAULT_GEOMETASEARCH_MIDDLE_PORT,
                             event_log=None,
                             load_log=None,
                             output_resource=None):
    """
        Синхронизировать ресурсы и получить обёртку промежуточного поиска
    """
    common_params = get_geometasearch_params_from_ctx(
        task=task, search_type='middle', params=params)
    return GeoMetaSearchMiddle(
        work_dir=channel.task.abs_path(),
        searchsources=searchsources,
        port=port,
        event_log=event_log,
        load_log=load_log,
        wizard=wizard,
        output_resource=output_resource,
        **common_params)


def get_geometasearch_upper(task,
                            params=GeoMetaSearchUpperParams,
                            searchsources=None,
                            port=DEFAULT_GEOMETASEARCH_UPPER_PORT,
                            event_log=None,
                            load_log=None,
                            output_resource=None,
                            reqans_log_resource=None):
    """
        Синхронизировать ресурсы и получить обёртку промежуточного поиска
    """
    common_params = get_geometasearch_params_from_ctx(
        task=task, search_type='upper', params=params)
    return GeoMetaSearchUpper(
        work_dir=channel.task.abs_path(),
        searchsources=searchsources,
        port=port,
        event_log=event_log,
        load_log=load_log,
        reqans_log_resource=reqans_log_resource,
        output_resource=output_resource,
        **common_params
    )
