# -*- coding: utf-8 -*-

import os
import time
import logging
import socket

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.network import is_port_free
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import LastReleasedResource

from sandbox.projects.common.utils import create_misc_resource_and_dir
from sandbox.projects.common.search import config as sconf


SPELLCHECKER_DEFAULT_START_TIMEOUT = 60 * 10
SPELLCHECKER_DEFAULT_PORT = 19036
SPELLCHECKER_DEFAULT_THREADS = 'ncpu'


class SpellcheckerComponent(object):
    def __init__(
        self,
        binary_path=None,
        data_path=None,
        server_config_path=None,
        misspell_config_path=None,
        configs_dir=None,
        log_dir=None,
        start_timeout=SPELLCHECKER_DEFAULT_START_TIMEOUT,
        port=SPELLCHECKER_DEFAULT_PORT,
        threads=SPELLCHECKER_DEFAULT_THREADS
    ):
        self.server_config = sconf.SearchConfig.get_config_from_file(server_config_path)
        self.misspell_config = sconf.IniConfig.get_config_from_file(misspell_config_path)
        autocorrection_config_path = os.path.join(data_path, 'autocorrection', 'autocorrection.ini')
        self.autocorrection_config = sconf.IniConfig.get_config_from_file(autocorrection_config_path)

        self.binary_path = binary_path
        self.data_path = data_path
        self.configs_dir = configs_dir
        self.log_dir = log_dir

        self.port = port
        self.start_timeout = start_timeout
        self.threads = threads

    def start(self):
        if not is_port_free(self.port):
            raise SandboxTaskFailureError('Port for spellchecker %s is not free.' % self.port)

        self.autocorrection_config.apply_local_patch({
            'general/data-root': os.path.join(self.data_path, 'autocorrection')
        })
        patched_autorrection_config_path = os.path.join(self.configs_dir, 'spellchecker_autocorrection.ini')
        self.autocorrection_config.save_to_file(patched_autorrection_config_path)

        self.misspell_config.apply_local_patch({
            'misspell/data-root': self.data_path,
            'misspell/autocorrection': patched_autorrection_config_path
        })
        patched_misspell_config_path = os.path.join(self.configs_dir, 'spellchecker_misspell.ini')
        self.misspell_config.save_to_file(patched_misspell_config_path)

        self.server_config.apply_local_patch({
            'Server/Port': str(self.port),
            'Server/Threads': str(self.threads),
            'Server/LogDir': self.log_dir,
            'App/InitParam/Config': patched_misspell_config_path,
            'App/InitParam/LogDir': self.log_dir,
        })
        patched_server_config_path = os.path.join(self.configs_dir, 'spellchecker_server.cfg')
        self.server_config.save_to_file(patched_server_config_path)

        # Patch, allow spellchecker to execute
        # run_process('chmod +x %s' % self.binary_path, wait=True, check=True)

        run_cmd = [self.binary_path,
                   '-p', str(self.port),
                   '-c', patched_server_config_path,
                   '-n', str(self.threads)]
        self.process = run_process(run_cmd, wait=False, log_prefix='spellchecker')
        return self.process

    def is_running(self):
        if not self.process:
            return False
        if is_port_free(self.port):
            return False
        return True

    def wait(self):
        start = time.time()
        logging.info('Waiting spellchecker to start for %s seconds' % str(self.start_timeout))
        while (time.time() - start < self.start_timeout):
            if self.is_running():
                break
            time.sleep(10)
        else:
            raise SandboxTaskFailureError('Spellchecker not started')
        logging.info('Spellchecker started on port %s' % self.port)

    def stop(self):
        if self.process is None:
            raise SandboxTaskFailureError("search component is not started")
        if not self.is_running():
            raise SandboxTaskFailureError("Spellchecker died")

        try:
            sock = socket.create_connection(('localhost', int(self.port)))
            sock.send('CTL STOP HTTP/1.1\r\n')
            sock.close()
        except Exception as e:
            logging.error("Error in %s termination: %s" % (self.process, e))
            self.process.kill()
            logging.info("%s - killed" % self.process)
            time.sleep(1)  # need some time to release port
            return

        for i in range(60):
            if self.process.poll() is not None:
                break
            logging.info("waiting 1sec for %s" % self.process)
            time.sleep(1)

        if self.process.poll() is not None:
            logging.info("%s - terminated correctly" % self.process)
        else:
            self.process.kill()
            logging.info("%s - killed by timeout" % self.process)
            time.sleep(1)  # need some time to release port


def GetSpellcheckerParams():
    group_name = 'Spellchecker parameters'

    class Params:
        class Binary(LastReleasedResource):
            name = 'spellchecker_binary_resource_id'
            description = 'Spellchecker binary'
            resource_type = 'SPELLCHECKER_EXECUTABLE'
            group = group_name

        class Data(LastReleasedResource):
            name = 'spellchecker_data_resource_id'
            description = 'Spellchecker data'
            resource_type = 'SPELLCHECKER_DATA'
            group = group_name

        class ServerConfig(LastReleasedResource):
            name = 'spellchecker_server_config_resource_id'
            description = 'Spellchecker server config (server.cfg)'
            resource_type = 'SPELLCHECKER_SERVER_CONFIG'
            group = group_name

        class MisspellConfig(LastReleasedResource):
            name = 'spellchecker_misspell_config_resource_id'
            description = 'Spellchecker misspell config (misspell.ini)'
            resource_type = 'SPELLCHECKER_MISSPELL_CONFIG'
            group = group_name

        params = (Binary, Data, ServerConfig, MisspellConfig, )

    return Params


def get_spellchecker_component(ctx=None,
                               params=GetSpellcheckerParams(),
                               configs_dir=None,
                               **kwargs):
    binary_path = channel.task.sync_resource(ctx[params.Binary.name])
    data_path = channel.task.sync_resource(ctx[params.Data.name])
    server_config_path = channel.task.sync_resource(ctx[params.ServerConfig.name])
    misspell_config_path = channel.task.sync_resource(ctx[params.MisspellConfig.name])

    # Patch for new resource
    data_path = os.path.join(data_path, 'data')

    logging.debug(binary_path)
    logging.debug(data_path)
    logging.debug(server_config_path)
    logging.debug(misspell_config_path)

    if 'log_dir' not in kwargs:
        log_dir = create_misc_resource_and_dir(channel.task.ctx, 'spellchecker_logs_resource_id',
                                               'spellchecker logs', 'spellchecker_logs')

    return SpellcheckerComponent(
        binary_path=binary_path,
        data_path=data_path,
        server_config_path=server_config_path,
        misspell_config_path=misspell_config_path,
        configs_dir=configs_dir,
        log_dir=log_dir,
        **kwargs
    )
