import os
import logging
import re
import subprocess

from sandbox.sandboxsdk.paths import make_folder

from sandbox.sandboxsdk.errors import SandboxTaskUnknownError
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.process import run_process


def make_softlinks_for_libs(work_dir, lib_paths, target_lib):
    lib_name_to_path = {}
    for lib_path in lib_paths:
        if not lib_path:
            continue
        basename = os.path.basename(lib_path)
        lib_name_to_path[basename] = lib_path

    if not len(lib_name_to_path):
        return

    lib_name_to_linkname = {}
    ldd_process = run_process(['ldd', target_lib], outs_to_pipe=True)
    for l in ldd_process.stdout.readlines():
        lib_linkname = l[:l.find('=>')].strip()
        for lib_name in lib_name_to_path:
            if lib_linkname.startswith(lib_name):
                lib_name_to_linkname[lib_name] = lib_linkname
    ldd_process.stdout.close()
    ldd_process.stderr.close()

    for lib_name, lib_path in lib_name_to_path.iteritems():
        if lib_name not in lib_name_to_linkname:
            raise SandboxTaskUnknownError('%s does not link with %s' % (target_lib, lib_name))
        lib_linkname = lib_name_to_linkname[lib_name]
        run_process(['ln', '-s', '-f', lib_path, os.path.join(work_dir, lib_linkname)])

    ldd_process = run_process(['ldd', target_lib],
                              outs_to_pipe=True,
                              environment={'LD_LIBRARY_PATH': work_dir})
    logging.debug('Final ldd output for %s:\n%s' % (target_lib, ''.join(ldd_process.stdout.readlines())))
    ldd_process.stdout.close()
    ldd_process.stderr.close()


def bash_run(cmd, env=None, allowed_codes=[0]):
    if env is not None:
        cur_env = os.environ.copy()
        cur_env.update(env)
        env = cur_env

    p = subprocess.Popen(
        cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, close_fds=True, env=env)
    stdout, stderr = p.communicate()
    returncode = p.returncode
    if returncode not in allowed_codes:
        raise SandboxTaskFailureError(
            "Command '{0}' died with exit code {1}".format(cmd, returncode))


def pack_dir(dirname, to_name, arch_type):
    if arch_type == 'tar.gz':
        logging.info('Packing %s into %s' % (dirname, to_name))
        bash_run(
            'tar czvf "{0}" -C "{1}" .'.format(to_name, dirname))
        return True
    return False


def unpack_file(filename, to_dir, to_name=None, arch_type=None):
    if arch_type == 'tar' or re.match(r"""^.+\.tar$""", filename):
        logging.info('Unpacking %s into %s' % (filename, to_dir))
        bash_run(
            'cat {0} | tar xvf - -C "{1}"'.format(filename, to_dir))
        return True

    elif arch_type in ('tar.gz', 'tgz') or re.match(r"""^.+(\.tar\.gz|\.tgz)$""", filename):
        logging.info('Unpacking %s into %s' % (filename, to_dir))
        bash_run(
            'gzip -cd "{0}" | tar xvf - -C "{1}"'.format(filename, to_dir), allowed_codes=[0, 2])
        return True

    else:
        if not to_name:
            m = re.match(r"""^(.+)\.gz$""", filename)
            if m:
                to_name = m.group(1)
            elif arch_type == 'gz':
                to_name = filename

        if to_name is not None:
            logging.info('Unpacking %s into %s' % (filename, to_dir))
            bash_run(
                'gzip -cd "{0}" > {1}'.format(filename, os.path.join(to_dir, to_name)), allowed_codes=[0, 2])
            return True

    return False


def unpack_files(local_dir):
    have_new_files = True
    processed = []
    while have_new_files:
        have_new_files = False
        for filename in os.listdir(local_dir):
            fullname = os.path.join(local_dir, filename)
            if fullname in processed:
                continue
            processed.append(fullname)
            if unpack_file(fullname, local_dir):
                have_new_files = True
                try:
                    logging.info("Removing %s", fullname)
                    os.remove(fullname)
                except OSError as ex:
                    logging.info(
                        'Error while remove {0}: {1}'.format(fullname, ex))


def prepare_geo_user_factors(task, resource_id, factor_prefix):
    resource = task.sync_resource(resource_id)
    if os.path.isdir(resource):
        # just directory with files, no need to unpack
        return resource

    if os.path.isfile(resource) and resource.endswith('.tar.gz'):
        output_dir = make_folder('geo_user_factors')
        run_process([
            'tar', '--extract',
            '--file', resource,
            '--directory', output_dir,
            '--verbose',
            '--wildcards',
            './{0}_*'.format(factor_prefix)
        ], log_prefix='unpack_geo_user_factors')
        return output_dir

    raise SandboxTaskFailureError('unknown format of user factors')
