import os
import errno
import base64
import collections
import shutil
import six
from sandbox import sdk2

VaultKey = collections.namedtuple('VaultKey', 'key_owner key_name')


class GpgKey(object):
    """
    Context manager for using gpg keys pair.

    Usage:
        with gnupg.GpgKey('task_object', '<vault_items_owner>', '<vault_secret_key_name>', '<vault_public_key_name>'):
            ...
    """
    gnupg_dir = os.path.expanduser('~/.gnupg')
    TEMPGNUPG_DIR = os.path.expanduser('~/.local/tempgnupg')

    def __init__(self, task, key_owner, secret_key_name, public_key_name):
        """
        :param task: task object that requires key
        :param key_owner: owner of vault item's that contains secret and public keys
        :param secret_key_name: vault item that contains secret gpg key in base64 representation
        :param public_key_name: vault item that contains public gpg key in base64 representation
        """
        self._task = task
        self._secret_key = VaultKey(key_owner=key_owner, key_name=secret_key_name)
        self._public_key = VaultKey(key_owner=key_owner, key_name=public_key_name)
        self._secring_filename = os.path.join(self.gnupg_dir, 'secring.gpg')
        self._pubring_filename = os.path.join(self.gnupg_dir, 'pubring.gpg')

    def __enter__(self):
        if os.path.exists(self.gnupg_dir):
            shutil.move(self.gnupg_dir, self.TEMPGNUPG_DIR)
        os.mkdir(self.gnupg_dir)
        open(self._secring_filename, 'wb').write(
            base64.b64decode(
                self.get_vault_data(self._secret_key)
            )
        )
        open(self._pubring_filename, 'wb').write(
            base64.b64decode(
                self.get_vault_data(self._public_key)
            )
        )
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        os.unlink(self._secring_filename)
        os.unlink(self._pubring_filename)

        def onerror(_f, _path, exc):
            t, v, tb = exc
            if isinstance(v, OSError) and v.errno == errno.ENOENT:
                return
            six.reraise(t, v, tb)

        shutil.rmtree(self.gnupg_dir, onerror=onerror)

        if os.path.exists(self.TEMPGNUPG_DIR):
            shutil.move(self.TEMPGNUPG_DIR, self.gnupg_dir)

    def get_vault_data(self, key):
        """
        :type key: VaultKey
        """
        data = self._task.get_vault_data(*key)
        return data


class GpgKey2(GpgKey):
    """
    Context manager for using gpg keys pair. Created to use with SDK2.

    Usage:
        with gnupg.GpgKey('task_object', '<vault_items_owner>', '<vault_secret_key_name>', '<vault_public_key_name>'):
            ...
    """

    def __init__(self, key_owner, secret_key_name, public_key_name):
        """
        :param key_owner: owner of vault item's that contains secret and public keys
        :param secret_key_name: vault item that contains secret gpg key in base64 representation
        :param public_key_name: vault item that contains public gpg key in base64 representation
        """
        self._secret_key = VaultKey(
            key_owner=key_owner,
            key_name=secret_key_name,
        )
        self._public_key = VaultKey(
            key_owner=key_owner,
            key_name=public_key_name,
        )
        self._secring_filename = os.path.join(self.gnupg_dir, 'secring.gpg')
        self._pubring_filename = os.path.join(self.gnupg_dir, 'pubring.gpg')

    def get_vault_data(self, key):
        """
        :type key: VaultKey
        """
        data = sdk2.Vault.data(*key)
        return data
