# -*- coding: utf-8 -*-

import os
import collections
import itertools

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common.http_responses import request as httpreq


FIRST_RESPONSES_KEY = 'responses1_resource_id'
SECOND_RESPONSES_KEY = 'responses2_resource_id'
IGNORED_HEADERS_KEY = 'ignored_headers'
IGNORED_HEADERS_SEPARATOR = ','
DIFF_KEY = 'has_diff'
COMPARE_RESULT = 'compare_result'  # for compatibility with compare_basesearch_responses


def create_parameters(responses_resource_type, responses_resource_params=True):
    class FirstResponsesResourceParameter(sp.ResourceSelector):
        name = FIRST_RESPONSES_KEY
        description = 'First responses'
        resource_type = responses_resource_type

    class SecondResponsesResourceParameter(sp.ResourceSelector):
        name = SECOND_RESPONSES_KEY
        description = 'Second responses'
        resource_type = responses_resource_type

    class IgnoredHeadersParameter(sp.SandboxStringParameter):
        name = IGNORED_HEADERS_KEY
        description = 'HTTP headers to ignore (comma-separated)'
        default_value = ''

    params = (
        IgnoredHeadersParameter,
    )

    if responses_resource_params:
        params = (
            FirstResponsesResourceParameter,
            SecondResponsesResourceParameter,
        ) + params
    return params


class Response:
    def __init__(self, request, status, headers, data):
        self.request = request
        self.status = status
        self.headers = headers
        self.data = data


class BaseCompareResponsesTask(SandboxTask):

    def get_responses_resource_type(self):
        raise NotImplementedError()

    def get_diff_resource_type(self):
        raise NotImplementedError()

    def iterate_responses(self, responses_resource_id, number):
        responses_resource_type = self.get_responses_resource_type()
        responses_resource_path = self.sync_resource(responses_resource_id)
        extract_dir = self.abs_path('responses_data%d' % number)
        paths.make_folder(extract_dir, True)
        archive_path = os.path.join(responses_resource_path,
                                    responses_resource_type.archive_file)
        process.run_process(['tar', '-xf', archive_path, '-C', extract_dir], wait=True, check=True)
        with open(os.path.join(responses_resource_path, responses_resource_type.meta_file), 'r') as stats_file:
            request = None
            status = None
            headers = {}

            def result():
                data_file_name = os.path.join(
                    extract_dir,
                    responses_resource_type.data_dir,
                    httpreq.filename_for_request(request),
                )
                data = fu.read_file(data_file_name)
                return Response(request, status, headers, data)

            for line in stats_file:
                line = line.strip()
                if line.startswith('--'):
                    if request is not None:
                        yield result()
                        request = None
                        status = None
                        headers = {}
                    fields = line.split()
                    if len(fields) != 3:
                        eh.check_failed("Could not parse request line '{}' in stats".format(line))
                    request = fields[1]
                    status = int(fields[2])
                else:
                    fields = line.split(": ")
                    if len(fields) != 2:
                        eh.check_failed("Could not parse header line '{}' in stats".format(line))
                    headers[fields[0]] = fields[1]
            if request is not None:
                yield result()

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self._create_resource(self.descr, 'diff', self.get_diff_resource_type())
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        diff_dir = channel.sandbox.get_resource(self.ctx['out_resource_id']).path
        diff_resource_type = self.get_diff_resource_type()
        meta_file_path = os.path.join(diff_dir, diff_resource_type.meta_file)
        data_dir1 = os.path.join(diff_dir, diff_resource_type.first_data_dir)
        data_dir2 = os.path.join(diff_dir, diff_resource_type.second_data_dir)

        paths.make_folder(diff_dir, True)
        paths.make_folder(data_dir1, True)
        paths.make_folder(data_dir2, True)

        headers_to_ignore = self.ctx[IGNORED_HEADERS_KEY].split(IGNORED_HEADERS_SEPARATOR)
        headers_to_ignore = set(x.lower() for x in headers_to_ignore)

        meta_file = open(meta_file_path, 'w')
        self.ctx[DIFF_KEY] = False

        printed_requests = set()
        statuses1 = collections.defaultdict(lambda: 0)
        statuses2 = collections.defaultdict(lambda: 0)
        changed_headers = collections.defaultdict(lambda: 0)

        for response1, response2 in itertools.izip_longest(
                self.iterate_responses(self.ctx[FIRST_RESPONSES_KEY], 1),
                self.iterate_responses(self.ctx[SECOND_RESPONSES_KEY], 2), fillvalue=None):
            if response1 is None or response2 is None:
                eh.check_failed("Different number of responses, failing")

            if response1.request != response2.request:
                eh.check_failed("Different requests '{}' and '{}'".format(
                    response1.request,
                    response2.request,
                ))

            def compare_property(description, value1, value2, ignore_func=lambda x, y: False):
                output = None
                if value1 is None:
                    if value2 is None:
                        eh.check_failed("Both responses for request '{}' return None on '{}'".format(
                            response1.request, description
                        ))
                    else:
                        output = "+ {} '{}'".format(description, value2)
                else:
                    if value2 is None:
                        output = "- {} '{}'".format(description, value1)
                    elif value1 != value2 and not ignore_func(value1, value2):
                        output = "* {} '{}' -> '{}'".format(description, value1, value2)

                if output is not None:
                    if response1.request not in printed_requests:
                        meta_file.write("-- {}\n".format(response1.request))
                        self.ctx[DIFF_KEY] = True
                        self.ctx[COMPARE_RESULT] = False
                        printed_requests.add(response1.request)
                    meta_file.write("{}\n".format(output))
                    return False
                return True

            compare_property("Status code", response1.status, response2.status)
            statuses1[response1.status] += 1
            statuses2[response2.status] += 1
            header_names = set(response1.headers.keys() + response2.headers.keys()) - headers_to_ignore
            for header_name in sorted(header_names):
                def ignore_func(value1, value2):
                    if header_name == "date":
                        # Дата меняется каждый раз
                        return True
                    elif header_name == "connection":
                        # nginx и балансер выдают опцию Close в разном регистре
                        # балансер сам закрывает соединение, так что ок
                        return value1.lower() == value2.lower()

                if not compare_property("%s:" % header_name, response1.headers.get(header_name),
                                        response2.headers.get(header_name), ignore_func):
                    changed_headers[header_name] += 1

            if response1.data != response2.data:

                def write_response_data(data_dir, response):
                    data_file_name = os.path.join(data_dir, httpreq.filename_for_request(response.request))
                    fu.write_file(data_file_name, response.data)

                write_response_data(data_dir1, response1)
                write_response_data(data_dir2, response2)
                self.ctx[DIFF_KEY] = True
                self.ctx[COMPARE_RESULT] = False

        meta_file.close()
        stats_file_name = os.path.join(diff_dir, diff_resource_type.stats_file)
        with open(stats_file_name, 'w') as stats_file:

            def serialize_dict(dictionary):
                return "\n".join(("%s %s" % x for x in sorted(dictionary.items(), key=lambda x: x[0])))

            stats_file.write("First statuses:\n{}\n".format(serialize_dict(statuses1)))
            stats_file.write("Second statuses:\n{}\n".format(serialize_dict(statuses2)))
            stats_file.write("Changed headers:\n{}\n".format(serialize_dict(changed_headers)))

    def get_short_task_result(self):
        if self.is_finished():
            return "diff" if self.ctx[DIFF_KEY] else "no diff"
        return None
