# -*- coding: utf-8 -*-

import httplib
import os

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk import process

from sandbox.projects.common.base_search_quality import threadPool

from request import host_and_path
from request import filename_for_request


REQUESTS_RESOURCE_KEY = 'requests_resource_id'
REQUESTS_LIMIT_KEY = 'requests_limit'
REQUIRE_ALL_SUCCESS_KEY = 'require_all_success'
RESPONSES_RESOURCE_KEY = 'responses_resource_id'


def create_parameters(requests_resource_type, default_request_limit):
    group_name = 'Requests'

    class RequestsResourceParameter(LastReleasedResource):
        name = REQUESTS_RESOURCE_KEY
        description = 'Requests'
        group = group_name
        resource_type = requests_resource_type

    class RequestsLimitParameter(SandboxIntegerParameter):
        name = REQUESTS_LIMIT_KEY
        description = 'Requests number limit'
        group = group_name
        default_value = default_request_limit

    class RequireAllSuccessParameter(SandboxBoolParameter):
        name = REQUIRE_ALL_SUCCESS_KEY
        description = 'Fail if any request yields non-success code'
        group = group_name
        default_value = False

    return RequestsResourceParameter, RequestsLimitParameter, RequireAllSuccessParameter


class BaseLocalResponsesMixin:
    """
        Mixin class with methods for getting component responses
    """

    def get_responses_resource_type(self):
        raise NotImplementedError

    def get_data_dir(self):
        return self.abs_path(self.get_responses_resource_type().data_dir)

    def save_responses(self, component, result_dir, on_ready=None):
        requests_resource_path = self.sync_resource(self.ctx[REQUESTS_RESOURCE_KEY])
        with open(requests_resource_path, 'r') as requests_file:
            requests = [line.strip() for line in requests_file]
        if not requests:
            raise SandboxTaskFailureError("Requests file is empty")
        requests = requests[:self.ctx[REQUESTS_LIMIT_KEY]]

        make_folder(self.get_data_dir(), True)
        make_folder(result_dir, True)

        def dump_function(request_list, *args):
            result = []
            # hostname, т.к. некоторые сервисы (improxy) на localhost отвечают админкой
            hostname = os.uname()[1]
            connection = httplib.HTTPConnection(hostname, component.port, timeout=120)
            for request in request_list:
                request_host, request_path = host_and_path(request)
                try:
                    connection.request('GET', request_path, headers={
                        'Host': request_host if request_host else hostname,
                    })
                    response = connection.getresponse()
                except (IOError, httplib.HTTPException) as e:
                    raise SandboxTaskFailureError("Request '%s', failed to retrieve response from server: %s" % (request_path, e))
                data_file_path = os.path.join(self.get_data_dir(), filename_for_request(request_path))
                with open(data_file_path, 'w') as data_file:
                    data_file.write(response.read())
                header_dump = ["%s: %s" % x for x in sorted(response.getheaders(), key=lambda x: x[0])]
                result.append((request, response.status, "\n".join(header_dump)))
            return result

        result_list = threadPool.process_data(dump_function, requests, params=None)
        with open(os.path.join(result_dir, self.get_responses_resource_type().meta_file), 'w') as stats_file:
            for item in result_list:
                stats_file.write("-- %s %d\n%s\n" % item)

        archive_path = os.path.join(result_dir, self.get_responses_resource_type().archive_file)
        process.run_process(['tar', '-czf', archive_path, self.get_responses_resource_type().data_dir],
                            wait=True, check=True)

        if on_ready is not None:
            on_ready()

        if self.ctx[REQUIRE_ALL_SUCCESS_KEY] and any((x[1] != 200 for x in result_list)):
            raise SandboxTaskFailureError('Some responses have non-200 status')


class BaseLocalResponsesTask(SandboxTask, BaseLocalResponsesMixin):
    """
        Таск для сбора http-ответов с локального инстанса.
        Выходной ресурс содержит файл с метаинформацией (запросы, коды возврата, заголовки) по пути meta_file и
        архивированную директорию с ответами по пути archive_file
    """

    def get_component(self):
        raise NotImplementedError

    def get_result_dir(self):
        return self.abs_path('responses')

    def on_enqueue(self):
        self.ctx[RESPONSES_RESOURCE_KEY] = self._create_resource(
            self.descr,
            self.get_result_dir(),
            self.get_responses_resource_type()
        ).id

    def on_execute(self):
        component = self.get_component()
        component.start()
        component.wait()
        try:
            self.save_responses(
                component,
                self.get_result_dir(),
                on_ready=lambda: self.mark_resource_ready(self.ctx[RESPONSES_RESOURCE_KEY])
            )
        finally:
            component.stop()
