import os
import logging
import json
from textwrap import dedent

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import RamDrive, RamDriveType
import sandbox.projects.bitbucket.common.git_helpers as git


def find_resource(search_conf):
    logging.info('lookup resource str: %s' % search_conf)
    try:
        params = json.loads(search_conf)
    except Exception as exc:
        raise ValueError('Error while loading resource search config: {}, {}'
                         .format(search_conf, exc))
    logging.info('find_resource with params: {}'.format(params))
    res = sdk2.Resource.find(**params).order(-sdk2.Resource.id).first()
    logging.info('resource found: id: {}, size: {}, desc: "{}"'.format(res.id,
                                                                       res.size,
                                                                       res.description))
    return res


def get_named_resources(resource_map):
    res = dict()

    for name, some_str in resource_map.iteritems():
        try:
            resource_id = int(some_str)
        except (ValueError, TypeError):
            resource = find_resource(some_str)
        else:
            resource = sdk2.Resource.find(id=resource_id).first()
        if resource is None:
            raise ValueError('resource not found: {!r}'.format(some_str))
        res[name] = resource
    return res


class ExecScriptBase(sdk2.Task):
    """A base task, similar in purpose to RUN_SCRIPT, with sdk2.
    but allow to delay failure untill dedived class collect artifacts produced
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Task.Context):
        out_resource_list = []

    class Parameters(sdk2.Task.Parameters):
        script_url = sdk2.parameters.ArcadiaUrl('Script directory svn url', default='', required=False)
        command = sdk2.parameters.String('Command to execute, example: "dmesg > dmesg.txt; lscpu > lscpu.txt"', required=True, multiline=True)
        vault_env = sdk2.parameters.Dict("Vault items to put in the environment", required=False)
        env_vars = sdk2.parameters.Dict("Values to put in the OS environment", required=False)
        resource = sdk2.parameters.Resource("Resource to make available via {resource} in the command", required=False)
        with sdk2.parameters.Output:
            used_resources = sdk2.parameters.List("List of used resources")

        named_resources = sdk2.parameters.Dict(
            'named_resources',
            description=dedent('''\
            Mapping resource_name => resource_id | search_config,
            resource_name can be used in command line template.
            Search_config example:
            {"type": "BSMR_SCRIPTS_BINARY",
            "state": "READY",
            "attrs": {"released": "stable"}}
            Or explicit id
            {"id": "717656013"}
            ''')
        )
        git_repo_creds = sdk2.parameters.Bool("Add credentials to git_repo_url")
        git_repo_url = sdk2.parameters.String('repo_url', description='Git repo URL', default='', required=True)
        git_ref_id = sdk2.parameters.String('ref_id', description='Git ref id', default='master', required=True)
        git_ref_sha = sdk2.parameters.String('ref_sha', description='Git ref SHA', default='', required=False)
        git_history_depth = sdk2.parameters.Integer("depth", description="Git history depth", default=1)
        git_vault_secret_owner = sdk2.parameters.String('vault_secret_owner',
                                                        description='Vault Secret owner',
                                                        default='',
                                                        required=False
                                                        )
        git_vault_secret_name = sdk2.parameters.String('vault_secret_name',
                                                       description='Vault Secret Name',
                                                       default='bitbucket_token',
                                                       required=False
                                                       )
        git_vault_secret_user = sdk2.parameters.String('vault_secret_user',
                                                       description='Vault Secret User Name',
                                                       default='x-oauth-token',
                                                       required=False
                                                       )
        ramdrive_size = sdk2.parameters.Integer(
            'RAM drive size (in MiB)',
            default=0,
        )

    def on_enqueue(self, **kwargs):
        ramdrive_size = self.Parameters.ramdrive_size
        if ramdrive_size:
            self.Requirements.ramdrive = RamDrive(RamDriveType.TMPFS, ramdrive_size, None)
        return super(ExecScriptBase, self).on_enqueue(**kwargs)

    # Derived class may delay this exception
    def on_exception(self, exception):
        raise exception

    def get_workspace(self):
        return self.path('script')

    def get_cmdline(self, scriptdir):
        if self.Parameters.script_url:
            frozen_svn_url = sdk2.svn.Arcadia.freeze_url_revision(self.Parameters.script_url)
            logging.info('script_url "%s" -> %s', frozen_svn_url, scriptdir)
            sdk2.svn.Arcadia.export(frozen_svn_url, scriptdir)

    def git_fetch(self, work_dir):
        if self.Parameters.git_repo_url:
            pass
        if self.Parameters.git_repo_creds:
            vault_secret_owner = self.Parameters.git_vault_secret_owner
            if vault_secret_owner == "":
                vault_secret_owner = self.owner
            repo_url = git.get_repo_url_with_credentials(self.Parameters.git_repo_url,
                                                         self.Parameters.git_vault_secret_user,
                                                         sdk2.Vault.data(vault_secret_owner,
                                                                         self.Parameters.git_vault_secret_name))
        else:
            repo_url = self.Parameters.git_repo_url
        git.git_init(self, work_dir)
        git.git_fetch(self, work_dir, repo_url,
                      self.Parameters.git_ref_sha or self.Parameters.git_ref_id,
                      depth=self.Parameters.git_history_depth)
        git.git_checkout(self, work_dir, 'FETCH_HEAD')

    def dump_file_tail(self, fname, header, skip_empty=True, log_op=None, num_lines=10):
        if skip_empty:
            if not os.path.exists(fname) or not os.path.getsize(fname):
                return
        if not log_op:
            log_op = self.set_info

        lines = [header + '\n']
        with open(fname) as fp:
            lines += fp.readlines()[-num_lines:]
        log_op("".join(lines))

    def on_execute(self):
        scriptdir = str(self.get_workspace()).strip()
        used_resources = []
        resources = dict()
        if self.Parameters.named_resources:
            res = get_named_resources(self.Parameters.named_resources)
            for name, r in res.iteritems():
                used_resources.append(r)
                resources[name] = str(sdk2.ResourceData(r).path)
        if self.Parameters.resource:
            path = str(sdk2.ResourceData(self.Parameters.resource).path)
            resources['resource'] = path
            used_resources.append(self.Parameters.resource)
        if used_resources:
            ur_list = []
            for r in used_resources:
                ur_list.append(r.id)
            self.Parameters.used_resources = ur_list

        if not os.path.exists(scriptdir):
            os.makedirs(scriptdir)

        self.get_cmdline(scriptdir)
        command_env = os.environ.copy()

        for key, value in self.Parameters.env_vars.items():
            command_env[key] = value

        if self.ramdrive:
            # Note: PosixPath -> bytes
            command_env['RAMDRIVE_PATH'] = str(self.ramdrive.path)

        for vault_item, env_name in self.Parameters.vault_env.items():
            command_env[env_name] = sdk2.Vault.data(vault_item)

        if resources:
            command = self.Parameters.command.format(**resources)
        else:
            command = self.Parameters.command

        logging.info('Run command  "%s"', command)
        with sdk2.helpers.ProcessLog(self, logger="script") as pl:
            proc = sdk2.helpers.subprocess.Popen(command,
                                                 env=command_env,
                                                 cwd=scriptdir,
                                                 shell=True,
                                                 stdout=pl.stdout,
                                                 stderr=pl.stderr,
                                                 close_fds=True)
            url = "%s/%s" % (self.log_resource.http_proxy,
                             pl.stdout.path.relative_to(self.path(self.log_resource.path)))
            err_url = "%s/%s" % (self.log_resource.http_proxy,
                                 pl.stderr.path.relative_to(self.path(self.log_resource.path)))
            tail_sfx = '?tail=1&force_text_mode=1'
            self.set_info('Process started. <a href="{0}" target="_blank">output</a>, <a href="{1}" target="_blank">stderr</a>'.format(url, err_url),
                          do_escape=False)
            self.set_info('Process <a href="{0}{1}">tail log </a>'.format(url, tail_sfx), do_escape=False)
            logging.info('wait task')
            proc.wait()
            logging.error('Script err_code:%d', proc.returncode)
            if proc.returncode:
                pl.stdout.flush()
                pl.stderr.flush()
                self.dump_file_tail(str(pl.stdout.path), "--------------------------- Captured stdout call ---------------------------")
                self.dump_file_tail(str(pl.stderr.path), "--------------------------- Captured stderr call ---------------------------")
                self.on_exception(TaskFailure("Script fail with %d" % proc.returncode))
            else:
                self.set_info('Script succeed')
