import json
import logging
import os
import yaml

from sandbox.common.errors import TaskFailure
from sandbox.projects.common.infra import resource_types  # noqa
from sandbox import sdk2


class ResourceManifest(object):
    """Base task for resource generation
       resource_delcaration:
       - path(String): local path to resouce
       - required(Bool): If true then task will fail if target path was not found
       - type(String): type of resource
       - desc(String): resource description
       - attr(Dict)  : disct of resource's attributes
    """

    def __init__(self, path, r_type='OTHER_RESOURCE', required=False, desc=None, attr=None):
        self.path = path
        self.r_type = r_type
        self.required = required
        self.desc = desc
        self.attr = attr
        if not self.desc:
            self.desc = self.path

    @staticmethod
    def _validate_key(task, dec, key, required=True, default=None):
        if key in dec:
            return
        if required:
            task.on_exception(TaskFailure('Bad resource declaration for %s, key:%s is required' % (dec, key)))

        logging.info('Declaration %s has no key:%s, assigne default:%s', dec, key, default)
        dec[key] = default

    @staticmethod
    def _validate_fmt(task, data):
        # Is it headless manifest?
        if not data.get('apiVersion', None):
            return data

        if data['apiVersion'] != '1':
            task.on_exception(TaskFailure('Unsupported apiVersion: %s' % (data['apiVersion'])))
            return data

        if not (data.get('kind', None) == 'SandboxResource'):
            task.on_exception(TaskFailure('Bad resource datalaration for %s, key:%s is required' % (data, 'kind')))
            return data

        ResourceManifest._validate_key(task, data, 'params', True)
        return data['params']

    @staticmethod
    def make_manifest(task, in_dec):
        logging.debug('Start validate %s', in_dec)
        dec = ResourceManifest._validate_fmt(task, in_dec.copy())
        ResourceManifest._validate_key(task, dec, 'path')
        ResourceManifest._validate_key(task, dec, 'required', False, False)
        ResourceManifest._validate_key(task, dec, 'type', False, 'OTHER_RESOURCE')
        ResourceManifest._validate_key(task, dec, 'desc', False, dec['path'])
        ResourceManifest._validate_key(task, dec, 'attr', False, {})
        logging.debug('Finish validate %s', dec)
        m = ResourceManifest(dec['path'], dec['type'], dec['required'], dec['desc'], dec['attr'])
        return m


def make_reslist_parameter(desc=''):
    sample_res_list = [dict(path='my-path.txt', type='OTHER_RESOURCES', required=False, desc='my-desription', attr={'ttl': 14})]

    if not desc:
        desc = 'JSONList of resources to collect example: %s' % sample_res_list

    return sdk2.parameters.JSON(desc)


def validate_resource_list(task, resource_list):
    ret_list = []
    for dec in resource_list:
        ret_list.append(ResourceManifest.make_manifest(task, dec))
    return ret_list


def create_resource(task, resource_path, resource_type, description, attributes, mark_ready=False):
    logging.info('Create resource path:%s type:%s desc:%s ', resource_path, resource_type, description)
    resource = sdk2.Resource[resource_type](task, description, path=resource_path, **attributes)
    if mark_ready:
        sdk2.ResourceData(resource).ready()
        logging.info('Resource id:%d path:%s ready', resource.id, resource_path)

    return resource


def handle_resource(task, res_man, basedir, xattrs={}, mask_ready=False):
    if xattrs:
        res_man.attr.update(xattrs)

    path = os.path.join(basedir, res_man.path)
    if os.path.exists(path):
        create_resource(task, path, res_man.r_type, res_man.desc, res_man.attr, mask_ready)
    else:
        if res_man.required:
            task.on_exception(TaskFailure('Resource path:%s not found, but declared as mandatory' % (path)))


def collect_resource_list(task, resource_list, basedir, mark_ready=False):
    for dec in resource_list:
        handle_resource(task, dec, basedir)


def deunicodify_hook(pairs):
    new_pairs = []
    for key, value in pairs:
        if isinstance(value, unicode):
            value = value.encode('utf-8')
        if isinstance(key, unicode):
            key = key.encode('utf-8')
        new_pairs.append((key, value))
    return dict(new_pairs)


def collect_resources_from_manifest(task, manifest_file, basedir, pattern='**/*.json'):
    path = os.path.join(basedir, manifest_file)
    if not os.path.exists(path):
        task.on_exception(TaskFailure('Cant file resource manifest file %s' % (path)))

    files = [path]
    if os.path.isdir(path):
        fiter = task.path(path).glob(pattern)
        if not fiter:
            task.on_exception(TaskFailure("Cant find manifest files at :%s" % path))
            return
        files = [str(f) for f in fiter]
    # Finally collect list of manifest files
    for fname in files:
        with open(fname) as f:
            if fname.endswith('.yml') or fname.endswith('.yaml'):
                data = yaml.load(f)
            else:
                data = json.load(f, object_pairs_hook=deunicodify_hook)
            m_list = validate_resource_list(task, data)
            collect_resource_list(task, m_list, basedir, True)
