# coding=utf-8

import os
import logging

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.common.types.task import ReleaseStatus
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.bitbucket.common import git_helpers
from sandbox.projects.common import gnupg
from sandbox.projects.sandbox.resources import LXC_CONTAINER
from sandbox.projects.resource_types import BUILD_LOGS
from sandbox.projects.common.infra.resource_types import DebPackage, DebPackageChanges, DebPackageSource, DebPackageSourceDsc


class BuildDebPackage(sdk2.Task):
    "Build standard Debian package from GIT, Arcadia or Tarball"

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4096
        disk_space = 4096
        client_tags = ctc.Tag.Group.LINUX
        privileged = True
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("Sources"):

            src_origin = sdk2.parameters.String(
                "Sources origin",
                choices=[('Git', 'git'),
                         ('Arcadia', 'arc'),
                         ('Tarball', 'tar')],
                default='git',
                required=True
            )

            with src_origin.value['git']:

                git_url = sdk2.parameters.String(
                    "GIT Repository",
                    description='Example: https://bb.yandex-team.ru/scm/project/repository.git',
                    required=False
                )

                git_head = sdk2.parameters.String(
                    "GIT Head",
                    description="GIT branch, tag or commit",
                    default='HEAD',
                    required=False
                )

                git_shallow = sdk2.parameters.Bool(
                    "GIT Shallow",
                    description="Clone git without history",
                    default=True,
                    required=False
                )

                # FIXME sandbox must have read-only access by default
                git_token = sdk2.parameters.Vault(
                    "GIT Token",
                    description="Format: [owner:]vault",
                    required=False
                )

            with src_origin.value['arc']:

                arc_url = sdk2.parameters.ArcadiaUrl(
                    "Arcadia Url",
                    description='Example: arcadia:/arc/trunc/path[@revision]',
                    required=False
                )

            with src_origin.value['tar']:

                tar_resource = sdk2.parameters.Resource(
                    "Tarball with sources",
                    description='Tarball with debianized sources and single directrory in root',
                    required=False
                )

        with sdk2.parameters.Group("Build"):

            platform = sdk2.parameters.String(
                "Target platfrorm",
                choices=[('linux_ubuntu_18.04_bionic', 'linux_ubuntu_18.04_bionic'),
                         ('linux_ubuntu_16.04_xenial', 'linux_ubuntu_16.04_xenial'),
                         ('linux_ubuntu_14.04_trusty', 'linux_ubuntu_14.04_trusty'),
                         ('linux_ubuntu_12.04_precise', 'linux_ubuntu_12.04_precise')],
                default='linux_ubuntu_16.04_xenial',
                required=True
            )

            container = sdk2.parameters.Container(
                "LXC Container resource to use",
                description="If not set, use latest release of LXC_CONTAINER for target platform",
                resource_type=LXC_CONTAINER,
                required=False,
                default=None,
            )

            install_dependencies = sdk2.parameters.Bool(
                "Install dependencies",
                description="Resolve build requirements",
                default=True,
                required=False
            )

            build_parallel = sdk2.parameters.Bool(
                "Parallel build",
                description="Build with DEB_BUILD_OPTIONS=parallel={cores}",
                default=True,
                required=False
            )

            build_nocheck = sdk2.parameters.Bool(
                "Disable package checks",
                description="Build with DEB_BUILD_OPTIONS=nocheck",
                default=False,
                required=False
            )

            build_noopt = sdk2.parameters.Bool(
                "Disable compiler optimizations",
                description="Build with DEB_BUILD_OPTIONS=noopt",
                default=False,
                required=False
            )

            build_nostrip = sdk2.parameters.Bool(
                "Disable binary strip",
                description="Build with DEB_BUILD_OPTIONS=nostrip",
                default=False,
                required=False
            )

            build_dh_verbose = sdk2.parameters.Bool(
                "Verbose debhelper",
                description="Build with DH_VERBOSE=1",
                default=False,
                required=False
            )

            build_source = sdk2.parameters.Bool(
                "Build source package",
                description="Build both source and binary packages",
                default=False,
                required=False
            )

        with sdk2.parameters.Group("Release"):

            save_packages = sdk2.parameters.Bool(
                "Save packages as resources",
                default=True,
                required=False
            )

            sign_packages = sdk2.parameters.Bool(
                "Sign packages",
                default=False,
                required=False
            )

            with sign_packages.value[True]:

                gpg_public = sdk2.parameters.Vault(
                    "GPG Public key",
                    description="Format: [owner:]vault",
                    required=False
                )

                gpg_private = sdk2.parameters.Vault(
                    "GPG Private key",
                    description="Format: [owner:]vault",
                    required=False
                )

                upload_packages = sdk2.parameters.Bool(
                    "Upload packages",
                    description="Upload packages into dist",
                    default=False,
                    required=False
                )
                release_upload_packages = sdk2.parameters.Bool(
                    "Upload packages on task release",
                    description="Upload packages into dist on release",
                    default=False,
                    required=False
                )

            with upload_packages.value[True]:
                dist_repo = sdk2.parameters.String(
                    "Dist Repository",
                    description="debian repository to upload packages (comma separated list)",
                    required=False
                )

            with release_upload_packages.value[True]:
                release_dist_repo = sdk2.parameters.String(
                    "Dist Repository",
                    description="Debian repository to upload packages on release (comma separated list)",
                    required=False
                )
            release_subscribers = sdk2.parameters.String(
                "Release announce subscribers",
                description="Email to announce on release (comma separated list)",
                required=False
            )

            ssh_login = sdk2.parameters.String(
                "SSH Login",
                description="Login to use with dist",
                required=False
            )

            ssh_private = sdk2.parameters.Vault(
                "SSH Private key",
                description="Format: [owner:]vault",
                required=False
            )

        with sdk2.parameters.Output:
            build_log = sdk2.parameters.Resource(
                "Package building log",
                resource_type=BUILD_LOGS,
                required=True
            )

            deb_changes = sdk2.parameters.Resource(
                "Package .changes file",
                resource_type=DebPackageChanges,
                required=False
            )

            deb_packages = sdk2.parameters.Resource(
                "Binary .deb packages",
                resource_type=DebPackage,
                multiple=True,
                required=False
            )

    def check_call(self, args, **kwargs):
        self.build_log.write(" + '" + "' '".join(args) + "'\n")
        subprocess.check_call(args,
                              stdout=self.build_log,
                              stderr=subprocess.STDOUT,
                              **kwargs)
        self.build_log.write("\n")

    def checkout_sources(self):
        if self.Parameters.src_origin == "git":
            git_url = self.Parameters.git_url

            # FIXME bool(VaultItem)
            if self.Parameters.git_token.name:
                git_token = self.Parameters.git_token.data()
                git_url = git_helpers.get_repo_url_with_credentials(git_url, 'x-oauth-token', git_token)

            git_helpers.git_init(self, str(self.source_path))
            git_helpers.git_fetch(self, str(self.source_path), git_url,
                                  self.Parameters.git_head,
                                  depth=1 if self.Parameters.git_shallow else None)
            git_helpers.git_checkout(self, str(self.source_path), 'FETCH_HEAD')

        elif self.Parameters.src_origin == "arc":
            sdk2.svn.Arcadia.export(self.Parameters.arc_url, str(self.source_path))

        elif self.Parameters.src_origin == "tar":
            tar_path = str(sdk2.ResourceData(self.Parameters.tar_resource).path)
            cmd = ['tar',
                   '--strip-components=1',
                   '-xf', tar_path,
                   '-C', str(self.source_path)]
            if tar_path.endswith('.zst') or tar_path.endswith('.zstd'):
                cmd += ['-I', 'zstd']
            self.check_call(cmd)
        else:
            raise Exception('Unknown src_origin ' + self.Parameters.src_origin)

    def install_tools(self):
        packages = ['build-essential', 'devscripts']
        if self.Parameters.install_dependencies:
            packages += ['equivs']
        if self.Parameters.sign_packages:
            packages += ['gnupg']
        if self.Parameters.src_origin == 'git':
            packages += ['git']
        if self.Parameters.src_origin == 'tar':
            packages += ['zstd']

        self.check_call(['apt-get', 'install', '--yes', '--no-install-recommends'] + packages)

    def install_dependencies(self):
        self.check_call(['mk-build-deps', '-i',
                         '-t', 'apt-get -o Debug::pkgProblemResolver=yes --no-install-recommends --yes --no-remove'],
                        cwd=str(self.source_path))

    def build_packages(self):
        opts = []
        if self.Parameters.build_parallel and self.Requirements.cores:
            opts += ['parallel={}'.format(self.Requirements.cores)]
        if self.Parameters.build_nocheck:
            opts += ['nocheck']
        if self.Parameters.build_noopt:
            opts += ['noopt']
        if self.Parameters.build_nostrip:
            opts += ['nostrip']

        env = os.environ.copy()
        env['DEB_BUILD_OPTIONS'] = ' '.join(opts)

        if self.Parameters.build_dh_verbose:
            env['DH_VERBOSE'] = '1'

        self.check_call(['dpkg-buildpackage', '-us', '-uc',
                         '-F' if self.Parameters.build_source else '-b'],
                        env=env,
                        cwd=str(self.source_path))

    def sign_packages(self):
        with gnupg.GpgKey2(self.Parameters.gpg_public.owner or self.owner,
                           self.Parameters.gpg_private.name,
                           self.Parameters.gpg_public.name):
            # sec::4096:1:8BC06A99F978D4E3:2020-04-03:2021-04-03:::Name (comment) <login@yandex-team.ru>:::\n
            gpg_output = subprocess.check_output(["gpg", "--list-secret-keys", "--with-colons"])
            key_id = gpg_output.split(":")[4]
            self.check_call(["debsign", "-k{}".format(key_id), "-pgpg --batch --no-tty"],
                            cwd=str(self.source_path))

    def parse_changes(self, changes):
        meta = {}
        field = None
        gpg_signature_body = False
        for line in changes.splitlines():
            if line and line == '-----BEGIN PGP SIGNATURE-----':
                gpg_signature_body = True
            if line and line == '-----END PGP SIGNATURE-----':
                gpg_signature_body = False
            if gpg_signature_body:
                continue
            if line and line[0] == ' ':
                meta[field] += line[1:] + '\n'
            elif ":" in line:
                field, value = line.split(':', 1)
                meta[field] = value.strip()

        files = {}
        for line in meta['Files'].splitlines():
            md5, size, section, priority, filename = line.split()
            files[filename] = {
                'package': filename.split('_')[0],
                'version': meta['Version'],
                'size': int(size),
                'section': section,
                'priority': priority,
                'md5': md5,
            }

        for sha in ['Sha1', 'Sha256']:
            if 'Checksums-' + sha in meta:
                for line in meta['Checksums-' + sha].splitlines():
                    csum, size, filename = line.split()
                    assert files[filename]['size'] == int(size)
                    files[filename][sha.lower()] = csum

        return meta, files

    def save_package(self, file_path, attrs, mark_ready=True):

        if file_path.suffix == '.changes':
            resource_type = DebPackageChanges
        elif file_path.suffix == '.dsc':
            resource_type = DebPackageSourceDsc
        elif file_path.suffix in ['.deb', '.udeb']:
            resource_type = DebPackage
        else:
            resource_type = DebPackageSource

        resource = resource_type(self, file_path.name, path=file_path,
                                 deb_package=attrs.get('package'),
                                 deb_version=attrs.get('version'),
                                 deb_md5=attrs.get('md5'),
                                 deb_sha1=attrs.get('sha1'),
                                 deb_sha256=attrs.get('sha256'))
        if mark_ready:
            sdk2.ResourceData(resource).ready()

        return resource

    def upload_packages(self, dist_repo, file_paths):
        logging.info("upload_packages dist_repo:{} pkgs:{}".format(dist_repo, file_paths))
        owner = self.Parameters.ssh_private.owner or self.owner
        login = self.Parameters.ssh_login or owner
        host = 'duploader.yandex.ru'
        incomming = "/repo/" + dist_repo + "/mini-dinstall/incoming/"
        opts = ["-v",
                "-o", "StrictHostKeyChecking=no",
                "-o", "UserKnownHostsFile=/dev/null"]
        with sdk2.ssh.Key(self, owner, self.Parameters.ssh_private.name):
            # -B batch mode for uploading to a single instance
            subprocess.check_call(["scp", "-B"] + opts + map(str, file_paths) +
                                  ['{}@{}:{}'.format(login, host, incomming)])

    def dmove_packages(self, dist_repo, tobranch, package, version, frombranch='unstable'):
        logging.info("dmove  repo:{}, tobranch: {}, package: {}, version: {}, frombranch: {}".format(
            dist_repo, tobranch, package, version, frombranch))
        owner = self.Parameters.ssh_private.owner or self.owner
        login = self.Parameters.ssh_login or owner
        host = 'duploader.yandex.ru'
        opts = ["-v",
                "-o", "StrictHostKeyChecking=no",
                "-o", "UserKnownHostsFile=/dev/null"]
        dmove_cmd = ['dmove', dist_repo, tobranch, package, version]
        if frombranch is not None:
            dmove_cmd.append(frombranch)

        with sdk2.ssh.Key(self, owner, self.Parameters.ssh_private.name):
            subprocess.check_call(["ssh"] + opts +
                                  ['{}@{}'.format(login, host)] +
                                  ["sudo"] + dmove_cmd)

    def on_enqueue(self):
        if self.Parameters.container is None:
            self.Parameters.container = LXC_CONTAINER.find(
                attrs={"platform": self.Parameters.platform,
                       "released": "stable"}).first()
        if self.Parameters.release_upload_packages:
            self.Parameters.save_packages = True

        self.Context.release_types = [ReleaseStatus.UNSTABLE]
        self.Context.release_subscribers = []
        if self.Parameters.release_subscribers:
            self.Context.release_subscribers = self.Parameters.release_subscribers.split(",")

    def on_execute(self):
        self.build_path = self.path("build")
        self.build_path.mkdir()

        self.source_path = self.build_path.joinpath("source")
        self.source_path.mkdir()

        build_log_path = self.build_path.joinpath("build.log")
        self.build_log = build_log_path.open('ab', buffering=0)

        build_log_resource = BUILD_LOGS(self, build_log_path.name, path=build_log_path)
        sdk2.ResourceData(build_log_resource).ready()
        self.Parameters.build_log = build_log_resource
        self.set_info("See : <a href='{}'>build.log</a>".format(self.Parameters.build_log.http_proxy), do_escape=False)

        # dump kernel and os versions
        self.check_call(['uname', '-a'])
        self.check_call(['cat', '/etc/os-release'])

        self.check_call(['apt-get', 'update'])

        self.install_tools()

        self.checkout_sources()

        if self.Parameters.install_dependencies:
            self.install_dependencies()

        # dump installed packages
        self.check_call(['dpkg', '-l'])

        self.build_packages()

        if self.Parameters.sign_packages:
            self.sign_packages()

        changes_path, = self.build_path.glob('*.changes')
        meta, files = self.parse_changes(changes_path.read_text())
        self.Context.deb_meta = meta
        self.Context.deb_files = files
        self.Context.release_changelog = meta["Changes"]

        for file_name in files:
            file_path = self.build_path.joinpath(file_name)
            assert file_path.stat().st_size == files[file_name]['size']
            # packages are just built, no reason for checking checksums

        if self.Parameters.save_packages:
            deb_packages = []
            for file_name in files:
                file_path = self.build_path.joinpath(file_name)
                resource = self.save_package(file_path, files[file_name])
                if file_path.suffix == '.deb':
                    deb_packages.append(resource)

            self.Parameters.deb_packages = deb_packages
            self.Parameters.deb_changes = self.save_package(changes_path,
                                                            {'package': meta['Source'],
                                                             'version': meta['Version']})
        if self.Parameters.upload_packages:
            upload_list = [self.build_path.joinpath(file_name) for file_name in files] + [changes_path]
            for r in self.Parameters.dist_repo.split(","):
                self.upload_packages(r,  upload_list)
                self.set_info("Upload released resources to {}".format(r))

    @property
    def release_template(self):
        return sdk2.ReleaseTemplate(
            cc=self.Context.release_subscribers,
            subject='Release packages version: {}'.format(self.Context.deb_meta['Version']),
            message=self.Context.release_changelog,
            types=self.Context.release_types,
        )

    def on_release(self, parameters):
        second_stage = [ReleaseStatus.TESTING,
                        ReleaseStatus.PRESTABLE,
                        ReleaseStatus.STABLE]

        status = parameters['release_status']
        logging.info("on_relase start enter: status: {} release_upload_packages: {} release_dist_repo:{}".format(
            status, self.Parameters.release_upload_packages, self.Parameters.release_dist_repo))
        super(BuildDebPackage, self).on_release(parameters)

        if not self.Parameters.release_upload_packages or not self.Parameters.release_dist_repo or status == 'cancelled':
            self.set_info("Dist operations not required")
            return

        self.build_log = subprocess.STDOUT
        repos = self.Parameters.release_dist_repo.split(",")
        if status == 'unstable':
            files = []
            for resource in self.Parameters.deb_packages + [self.Parameters.deb_changes]:
                path = sdk2.ResourceData(resource).path
                files.append(str(path))
            if files:
                for r in repos:
                    self.set_info("[BEGIN] Upload released resources to {}".format(r))
                    self.upload_packages(r, files)
                    self.set_info("[DONE]  Upload released resources to {}".format(r))
            # Once package uploaded to dist it can be later released to stable/testing via dmove
            self.Context.release_types = second_stage

        if status in second_stage:
            for r in repos:
                for f in self.Context.deb_files:
                    self.set_info("[BEGIN] Dmove released resources to {} / {} ".format(r, status))
                    self.dmove_packages(r, status,
                                        self.Context.deb_files[f]['package'],
                                        self.Context.deb_files[f]['version'])
                    self.set_info("[DONE]  Dmove released resources to {} / {} ".format(r, status))
