# coding=utf-8

import shutil

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.sdk2.helpers import subprocess
from sandbox.common.types.misc import RamDrive, RamDriveType
from sandbox.projects.sandbox.resources import LXC_CONTAINER
from sandbox.projects.resource_types import BUILD_LOGS
from sandbox.projects.common.infra.resource_types import PBUILDER_CONFIG, GENERIC_PBUILDER_TARBALL


class BUILD_PBUILDER_TARBALL(sdk2.Task):
    "Build pbuilder base tarball"

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        ram = 4096
        disk_space = 4096
        client_tags = ctc.Tag.Group.LINUX
        privileged = True
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group("Sources"):
            distribution = sdk2.parameters.String(
                "distribution",
                choices=[('linux_ubuntu_20.04_focal', 'focal'),
                         ('linux_ubuntu_18.04_bionic', 'bionic'),
                         ('linux_ubuntu_16.04_xenial', 'xenial'),
                         ('linux_ubuntu_14.04_trusty', 'trusty'),
                         ('linux_ubuntu_12.04_precise', 'precise'),
                         ('linux_debian_sid', 'sid')],
                default='xenial',
                required=True)

            platform = sdk2.parameters.String(
                "result resource platform name",
                default="",
                required=False,
            )
            config_url = sdk2.parameters.String(
                "config url",
                required=False,
            )

            config_resource = sdk2.parameters.String(
                "config resource",
                required=False,
            )

        with sdk2.parameters.Group("Build"):
            container = sdk2.parameters.Container(
                "LXC Container resource to use",
                description="If not set, use latest release of LXC_CONTAINER for target platform",
                resource_type=LXC_CONTAINER,
                required=False,
                default=None,
            )
            ramdrive_size = sdk2.parameters.Integer(
                'RAM drive size (in MiB)',
                default=8192,
            )

        with sdk2.parameters.Output:
            base_tarball = sdk2.parameters.Resource(
                "Pbuilder base image",
                resource_type=GENERIC_PBUILDER_TARBALL,
                required=True
            )

            config = sdk2.parameters.Resource(
                "Pbuilder base image",
                resource_type=PBUILDER_CONFIG,
                required=True
            )

            build_log = sdk2.parameters.Resource(
                "Building log",
                resource_type=BUILD_LOGS,
                required=True
            )

    def check_call(self, args, **kwargs):
        self.build_log.write(" + '" + "' '".join(args) + "'\n")
        subprocess.check_call(args,
                              stdout=self.build_log,
                              stderr=subprocess.STDOUT,
                              **kwargs)
        self.build_log.write("\n")

    def run_pbuilder(self, cmd, args=[], **kwargs):
        self.check_call(["pbuilder", cmd,
                         "--configfile", str(self.pbuilderrc),
                         "--override-config",
                         "--distribution", self.Parameters.distribution,
                         "--basetgz", str(self.base_tgz),
                         "--aptcache", str(self.aptcache),
                         "--buildplace", str(self.buildplace),
                         "--buildresult",  str(self.buildresult)] + args,
                        **kwargs)

    def setup_pbuilder(self):
        self.buildresult = self.build_path.joinpath("result")
        self.buildresult.mkdir()

        self.pbuilderrc = self.build_path.joinpath("pbuilderrc")
        self.base_tgz = self.build_path.joinpath("base.tgz")

        self.aptcache = self.ramdrive.path.joinpath("aptcache")
        self.aptcache.mkdir()

        self.buildplace = self.ramdrive.path.joinpath("build")
        self.buildplace.mkdir()

        if self.Parameters.config_resource:
            config_path = str(sdk2.ResourceData(self.Parameters.config_resource).path)
            shutil.copyfileobj(config_path, self.pbuilderrc)
            self.config_origin = self.Parameters.config.http_proxy
        elif self.Parameters.config_url.startswith("arcadia"):
            sdk2.svn.Arcadia.export(self.Parameters.config_url, str(self.pbuilderrc))
            self.config_origin = self.Parameters.config_url
        else:
            raise sdk2.TaskFailure('Non arcadia url is not yet supported')

        self.Parameters.config = PBUILDER_CONFIG(self, self.pbuilderrc.name,
                                                 path=self.pbuilderrc,
                                                 config_origin=self.config_origin,
                                                 distribution=self.Parameters.distribution,
                                                 platform=self.Parameters.platform)
        sdk2.ResourceData(self.Parameters.config).ready()
        self.set_info("See : <a href='{}'>pbuilderrc</a>".format(self.Parameters.config.http_proxy), do_escape=False)

    def install_tools(self):
        packages = ['pbuilder', 'devscripts', 'zstd']
        self.check_call(['apt-get', 'install', '--yes', '--no-install-recommends'] + packages)

    def on_enqueue(self):
        self.Requirements.ramdrive = RamDrive(RamDriveType.TMPFS, self.Parameters.ramdrive_size, None)
        if self.Parameters.container is None:
            self.Parameters.container = LXC_CONTAINER.find(
                attrs={"platform": 'linux_ubuntu_16.04_xenial',
                       "released": "stable"}).first()

    def on_execute(self):
        self.build_path = self.path("pbuilder")
        self.build_path.mkdir()

        build_log_path = self.build_path.joinpath("build.log")
        self.build_log = build_log_path.open('ab', buffering=0)

        self.Parameters.build_log = BUILD_LOGS(self, build_log_path.name, path=build_log_path)
        sdk2.ResourceData(self.Parameters.build_log).ready()
        self.set_info("See : <a href='{}'>build.log</a>".format(self.Parameters.build_log.http_proxy), do_escape=False)

        # dump kernel and os versions
        self.check_call(['uname', '-a'])
        self.check_call(['cat', '/etc/os-release'])

        self.check_call(['apt-get', 'update'])

        self.install_tools()

        self.setup_pbuilder()

        self.run_pbuilder("create")
        self.run_pbuilder("execute", ["/bin/bash", "-c", "dpkg -l"])
        self.check_call(['dpkg', '-l'])
        config_res_id = str(self.Parameters.config.id)
        self.Parameters.base_tarball = GENERIC_PBUILDER_TARBALL(self, self.base_tgz.name,
                                                                path=self.base_tgz,
                                                                distribution=self.Parameters.distribution,
                                                                platform=self.Parameters.platform,
                                                                config_origin=self.config_origin,
                                                                config_resource=config_res_id)
        sdk2.ResourceData(self.Parameters.base_tarball).ready()
        self.set_info("Result: <a href='{}'>base_tarball</a>".format(self.Parameters.base_tarball.http_proxy), do_escape=False)
