import datetime
import gzip
import hashlib
import json
import logging
import shutil
import urllib

import requests

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.projects.common.infra import ya_make_tgz
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class RtcDockerBaseLayerInfo(sdk2.Resource):
    releasable = True
    any_arch = False
    auto_backup = True
    pull_url = sdk2.Attributes.String("Docker registry pull url")


class RtcDockerBaseLayer(sdk2.Task):
    class Parameters(ya_make_tgz.YaMakeTGZ.Parameters):
        with sdk2.parameters.Group('Docker Registry Params') as registry_params:
            registry_url = sdk2.parameters.String(label='Docker registry URL',
                                                  default_value='https://registry.yandex.net/')
            registry_token_secret_name = sdk2.parameters.String(label='Docker registry access token secret name')
            registry_namespace = sdk2.parameters.String(label='Docker registry namespace', required=True)
            registry_image = sdk2.parameters.String(label='Docker registry image name', required=True)
            registry_tag = sdk2.parameters.String(label='Docker registry tag (should be empty in most cases)',
                                                  default_value=None, required=False)
        with sdk2.parameters.Output:
            out_size = sdk2.parameters.Integer(label='Gzipped image size')
            out_tgz_sha256 = sdk2.parameters.String(label='Tgz image sha256')
            out_tar_sha256 = sdk2.parameters.String(label='Tar image sha256')
            out_layer_resource = sdk2.parameters.Resource(label='Resource id to upload to docker registry', required=False,
                                                          default_value=None)
        upload_layer = sdk2.parameters.Resource(label='Resource id to upload to docker registry', required=False,
                                                      default_value=None)

    def on_execute(self):
        if not self.Parameters.upload_layer:
            with self.memoize_stage.build_layer:
                self._build_layer()
            build_task = self.find().first()
            logging.debug('Found build task: {}'.format(build_task))
            if build_task.status != ctt.Status.SUCCESS:
                raise SandboxTaskFailureError('Build in sub-task %s failed' % build_task.id)
            porto_layer = sdk2.Resource[self.Parameters.result_rt].find(task=build_task).first()
            logging.debug('Found porto layer resource: {}'.format(porto_layer))
            self.Parameters.out_layer_resource = porto_layer
        else:
            self.Parameters.out_layer_resource = self.Parameters.upload_layer
        self._upload_layer()

    def _build_layer(self):
        logging.info("Building layer")
        subtask = sdk2.task.Task["YA_MAKE_TGZ"](self, **self._task_params())
        subtask.save().enqueue()
        raise sdk2.WaitTask((subtask,), ctt.Status.Group.FINISH)

    def _task_params(self):
        return {k: v for (k, v) in self.Parameters if not k.startswith('out_')}

    def _upload_layer(self):
        logging.info("Uploading layer")
        token = sdk2.Vault.data(self.owner, self.Parameters.registry_token_secret_name)
        layer_resource = self.Parameters.out_layer_resource
        logging.debug('Found resource to upload: {}'.format(layer_resource))
        # get layer.tar.gz from build task or input params
        layer_data = sdk2.ResourceData(layer_resource)
        tgz_path = str(layer_data.path)
        logging.debug('Got local input resource: {}'.format(tgz_path))
        # create directory for releasable resource
        build_path = self.path("build")
        build_path.mkdir()
        # copy layer.tar.gz from input resource to releasable resource
        local_tgz_path = str(build_path.joinpath('layer.tar.gz'))
        shutil.copy(tgz_path, local_tgz_path)

        # get image digests
        # according to https://docs.docker.com/registry/spec/api/#digest-parameter it is sha256

        # get sha256 of uncompressed layer.tar.gz
        # used in container config blob
        with gzip.open(local_tgz_path, 'rb') as f:
            h = hashlib.sha256()
            data = f.read(8192)
            while data:
                h.update(data)
                data = f.read(8192)
            tar_sha256 = h.hexdigest()
        # get sha256 of compressed layer.tar.gz
        # used for image upload and manifest creation
        with open(local_tgz_path, 'rb') as f:
            h = hashlib.sha256()
            data = f.read(8192)
            while data:
                h.update(data)
                data = f.read(8192)
            tgz_sha256 = h.hexdigest()
            size = f.tell()
            f.seek(0)
            logging.info('Got resource: {} sha256: {}, size: {}', tgz_path, tgz_sha256, size)
            self._put_blob(token, f, tgz_sha256)

        # create releasable resource
        layer_resource = RtcDockerBaseLayerInfo(self, 'RTC Docker layer', local_tgz_path, arch=ctm.OSFamily.LINUX)
        layer_resource.pull_url = '{}/{}:{}'.format(self._registry_url().split('://')[1], self._docker_name(),
                                                    self._docker_tag())
        self._put_manifest(token, size, tgz_sha256, tar_sha256, self._docker_tag())
        self.Parameters.out_size = size
        self.Parameters.out_tar_sha256 = tar_sha256
        self.Parameters.out_tgz_sha256 = tgz_sha256
        sdk2.ResourceData(layer_resource).ready()

    def _docker_tag(self):
        return 'sb-{}'.format(self.id)

    def _put_blob(self, token, blob, sha256):
        request_url = '{}/v2/{}/blobs/uploads/'.format(self._registry_url(), self._docker_name())
        start_resp = requests.post(request_url, headers=self._request_headers(token))
        if start_resp.status_code != 202:
            raise SandboxTaskFailureError('Cannot start docker image blob upload: "{}"'.format(start_resp.content))
        blob_location = start_resp.headers['Location']
        blob_location = urllib.unquote(blob_location)
        upload_uuid = start_resp.headers['Docker-Upload-UUID']
        logging.info('Started upload with uuid: {}, location: {}'.format(upload_uuid, blob_location))
        request_url = '{}&digest=sha256:{}'.format(blob_location, sha256)
        upload_resp = requests.put(request_url, data=blob, headers=self._request_headers(token))
        if upload_resp.status_code != 201:
            msg = 'Cannot upload docker image: {}'.format(upload_resp.content)
            logging.error(msg)
            delete_resp = requests.delete(request_url, headers=self._request_headers(token))
            logging.error('Delete request for {} status: {}'.format(request_url, delete_resp.status_code))
            logging.error('Delete request for {} message: "{}"'.format(request_url, delete_resp.content))
            raise SandboxTaskFailureError(msg)
        logging.info('Successfully finished upload {}'.format(upload_uuid))
        return upload_resp.headers['Location']

    def _put_manifest(self, token, size, tgz_sha256, tar_sha256, tag):
        """
        Put image manifest to registry according to https://docs.docker.com/registry/spec/manifest-v2-2/
        """
        logging.info('Uploading image manifest for tag: {}'.format(tag))
        tgz_digest = 'sha256:{}'.format(tgz_sha256)
        tar_digest = 'sha256:{}'.format(tar_sha256)
        now = '{}Z'.format(datetime.datetime.utcnow().isoformat())
        # docker_image_config used twice with same values in container_image_dict
        docker_image_config = {
            'Hostname': '',
            'Domainname': '',
            'User': '',
            'AttachStdin': False,
            'AttachStdout': False,
            'AttachStderr': False,
            'Tty': False,
            'OpenStdin': False,
            'StdinOnce': False,
            'Env': None,
            'Cmd': None,
            'Image': '',
            'Volumes': None,
            'WorkingDir': '',
            'Entrypoint': None,
            'OnBuild': None,
            'Labels': None
        }
        # image config for config field described in
        # https://docs.docker.com/registry/spec/manifest-v2-2/#image-manifest-field-descriptions
        # unfortunately config structure not described in any docker-registry related docs
        # so to get the structure you have to use docker import && docker tag && docker push
        # to put real docker-forged image to registry and then get image config blob from registry
        container_image_dict = {
            'architecture': 'amd64',
            'comment': 'Imported from Arcadia',
            'config': docker_image_config,
            'container_config': docker_image_config,
            'created': now,
            'docker_version': '19.03.12',
            'history': [
                {
                    'created': now,
                    'comment': 'Imported from Arcadia'
                }
            ],
            'os': 'linux',
            'rootfs': {
                'type': 'layers',
                'diff_ids': [tar_digest]
            }
        }
        container_image_blob = json.dumps(container_image_dict)
        h = hashlib.sha256(bytes(container_image_blob))
        container_image_blob_sha256 = h.hexdigest()
        container_image_blob_digest = 'sha256:{}'.format(container_image_blob_sha256)
        logging.debug('Image manifest sha256: {} blob: "{}"'.format(container_image_blob_sha256, container_image_blob))
        self._put_blob(token, container_image_blob, container_image_blob_sha256)
        # manifest structure defined in
        # https://docs.docker.com/registry/spec/manifest-v2-2/#image-manifest-field-descriptions
        manifest_dict = {
            "schemaVersion": 2,
            "mediaType": "application/vnd.docker.distribution.manifest.v2+json",
            "config": {
                "mediaType": "application/vnd.docker.container.image.v1+json",
                "size": len(container_image_blob),
                "digest": container_image_blob_digest
            },
            "layers": [
                {
                    "mediaType": "application/vnd.docker.image.rootfs.diff.tar.gzip",
                    "size": size,
                    "digest": tgz_digest
                }
            ]
        }
        manifest_blob = json.dumps(manifest_dict)
        request_url = '{}/v2/{}/manifests/{}'.format(self._registry_url(), self._docker_name(), tag)
        logging.debug('Uploading manifest to {}: {}'.format(request_url, manifest_blob))
        manifest_headers = self._request_headers(token)
        manifest_headers['Content-Type'] = 'application/vnd.docker.distribution.manifest.v2+json'
        manifest_resp = requests.put(request_url, data=manifest_blob, headers=manifest_headers)
        if manifest_resp.status_code != 201:
            raise SandboxTaskFailureError('Cannot create image manifest: {}'.format(manifest_resp.content))
        logging.info('Successfully uploaded manifest for tag: {} '
                     'with response: code {}, "{}"'.format(tag, manifest_resp.status_code, manifest_resp.content))

    def _registry_url(self):
        return str(self.Parameters.registry_url).rstrip('/')

    def _docker_name(self):
        return '{}/{}'.format(self.Parameters.registry_namespace, self.Parameters.registry_image)

    def _blob_exists(self, token, sha256):
        request_url = '{}/v2/{}/blobs/sha256:{}'.format(self._registry_url(), self._docker_name(), sha256)
        resp = requests.head(request_url, headers=self._request_headers(token), allow_redirects=True)
        return resp.status_code in [200, 307]

    @staticmethod
    def _request_headers(token):
        return {'Authorization': 'OAuth {}'.format(token)}

    def on_release(self, parameters_):
        logging.info("Releasing layer")
        super(RtcDockerBaseLayer, self).on_release(parameters_)
        token = sdk2.Vault.data(self.owner, self.Parameters.registry_token_secret_name)
        self._put_manifest(token, self.Parameters.out_size, self.Parameters.out_tgz_sha256,
                           self.Parameters.out_tar_sha256, parameters_['release_status'])
