# -*- coding: utf-8 -*-

import os
import logging
import tarfile
import shutil
import lxml.etree as et

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build.YaMake2 import YaMake2, ResultResourceTypeParameter
from sandbox.projects.common.build import parameters as bp
from sandbox.projects.common.infra.ya_make_tgz import junit_noframes_short_xsl
from sandbox.projects import resource_types

logger = logging.getLogger(__name__)
ARCHIVE_DIR = 'parchive'
DEFAULT_RESOURCE_TYPE_NAME = resource_types.ARCADIA_PROJECT_TGZ.name


class ARCADIA_PROJECT_BIN(sdk2.Resource):
    """
        Arcadia project artifact
    """
    releasable = True
    any_arch = False
    executable = True
    auto_backup = True


class YaMakeTGZ(YaMake2):
    """
        Build artifacts via YaMake2 and each artifact as separate resource
    """

    class Parameters(YaMake2.Parameters):
        # Customize default YaMake parameters
        result_rt = ResultResourceTypeParameter(default_value=DEFAULT_RESOURCE_TYPE_NAME)
        junit_report = bp.JUnitReport(default_value=True)

        # Task speciffic parameters
        resource_archive_check = sdk2.parameters.Bool("Do not archive if resource is archive", default=False)
        resource_compress = sdk2.parameters.Bool("Do we need to archive and compress artifacts", default=True)

        release_on_success = sdk2.parameters.Bool("Release task on success", default=False)
        release_status = sdk2.parameters.String('Release status', default='unstable')
        release_comment = sdk2.parameters.String('Release comment', default='Selfrelased after task success')
        release_subject = sdk2.parameters.String('Release subject', default='Selfrelased after task success')

        fail_fast = sdk2.parameters.Bool('Fail immediately if \'ya make\' returned error', default=False)

        with sdk2.parameters.Output:
            out_build_artifacts = sdk2.parameters.List("Build artifacts list",
                                                       value_type=sdk2.parameters.String,
                                                       default=[])
            with YaMake2.Parameters.junit_report.value[True]:
                out_junit_report = sdk2.parameters.Resource(
                    "junit report",
                    resource_type=resource_types.JUNIT_REPORT,
                    required=False
                )

    def on_exception(self, e):
        if self.Parameters.fail_fast:
            raise e
        else:
            self.set_info('Delay exception %s' % str(e))
            self.Context.exceptions = True

    def on_enqueue(self):
        if not self.Parameters.resource_compress and self.Parameters.result_rt == DEFAULT_RESOURCE_TYPE_NAME:
            self.Parameters.result_rt = ARCADIA_PROJECT_BIN.name
        YaMake2.on_enqueue(self)

    def need_archive(self, path):
        if not self.Parameters.resource_compress:
            return False
        if self.Parameters.resource_archive_check:
            if path.endswith('.tar.gz') or path.endswith('.tgz'):
                return False
        return True

    def get_arts(self):
        arts = YaMake2.get_arts(self)
        for a in arts:
            a['dest'] = a.get('dest', os.path.dirname(a['path']))
        return arts

    def get_arts_source(self):
        arts = YaMake2.get_arts_source(self)
        for a in arts:
            a['dest'] = a.get('dest', os.path.dirname(a['path']))
        return arts

    def do_get_resources(self):
        """
        Each artifact saved as dedicated resource
        """
        arts = self.get_arts() + self.get_arts_source()
        logger.debug("gen_resource, arts:%s", arts)

        if not arts:
            return {}

        gen_attr = {}
        def_flags = sdk.parse_flags(self.Parameters.definition_flags)
        for k, v in def_flags.iteritems():
            gen_attr['DF_' + k] = v

        if self.Parameters.result_single_file:
            params = {}
            params['resource_type'] = self.Parameters.result_rt
            params['description'] = 'resource.tar.gz'
            params['resource_path'] = os.path.join(ARCHIVE_DIR, 'resource.tar.gz')
            params['attributes'] = dict(gen_attr)
            params['attributes']['arcadia_path'] = '/'
            return {'.': params}

        resources = {}
        for a in arts:
            params = {}
            fname = os.path.basename(a['path'])
            path = os.path.join(a['dest'], fname)
            params['resource_type'] = self.Parameters.result_rt
            params['description'] = path
            params['attributes'] = dict(gen_attr)
            params['attributes']['arcadia_path'] = path
            if self.need_archive(path):
                params['resource_path'] = os.path.join(ARCHIVE_DIR, path + '.tar.gz')
            else:
                params['resource_path'] = path
            resources[path] = params
        return resources

    def get_resources(self):
        resources = self.do_get_resources()
        logging.debug("get_resources:%s", resources)
        return resources

    def fixup_resource_attr(self, path, attrs):
        logging.debug("append resource_attrs resource:%s attributes:%s", path, attrs)
        resource_id = self.Context.ap_packs[path]
        resource = sdk2.Resource[resource_id]

        self.out_build_artifacts.append(resource_id)
        for attr_name, attr_value in attrs.iteritems():
            try:
                setattr(resource, attr_name, attr_value)
            except AttributeError:
                if attr_value is None:
                    logging.debug('Skip attribute: %s', attr_name)
                else:
                    logging.debug('Try to set name: %s, value: %s', attr_name, attr_value)
                    self.server.resource[resource_id].attribute({"name": attr_name, "value": attr_value})
                    logging.debug('Attribute {"name": %s, "value": %s} is OK', attr_name, attr_value)

    def __post_build(self, source_dir, output_dir, pack_dir):
        """
        Generate tar.gz archives from artifacts
        """
        logging.debug('ap_packs %s', str(self.Context.ap_packs))

        resources = self.get_resources()
        logging.debug("post_build ->get_resources:%s", resources)

        if self.Parameters.junit_report:
            src_junit_filename = str(self.log_path('junit_report.xml'))

            # save original (.xml) 'junit_report' as a resource
            dst_junit_filename = str(os.path.join(pack_dir, 'junit_report.xml'))
            sdk2.Resource[resource_types.JUNIT_REPORT](self, 'junit_report.xml', dst_junit_filename)
            with open(src_junit_filename) as src, open(dst_junit_filename, 'w') as dst:
                shutil.copyfileobj(src, dst)

            # perform XSLT transformation of 'junit_report.xml' to human-readable .html
            dom = et.parse(src_junit_filename)
            xslt = et.fromstring(junit_noframes_short_xsl.blob)
            transform = et.XSLT(xslt)
            newdom = transform(dom)

            # save transformed (.html) 'junit_report' to file
            out_file = str(et.tostring(newdom, pretty_print=True))
            junit_filename_html = str(self.log_path('junit_report.html'))
            with open(junit_filename_html, 'w') as junit:
                junit.write(out_file)

            # save transformed (.html) 'junit_report' as a resource
            dst_junit_filename_html = str(os.path.join(pack_dir, 'junit_report.html'))
            results_resource_html = sdk2.Resource[resource_types.JUNIT_REPORT](self, 'junit_report.html', dst_junit_filename_html)
            with open(junit_filename_html) as src, open(dst_junit_filename_html, 'w') as dst:
                shutil.copyfileobj(src, dst)
            self.set_info("See : <a href='{}'>junit_report.html</a>".format(results_resource_html.http_proxy), do_escape=False)

        self.out_build_artifacts = []
        for archive_path, params in resources.iteritems():
            self.fixup_resource_attr(archive_path, params['attributes'])

            # If Artifact is archive, do nothing
            if not self.need_archive(archive_path):
                logger.debug('Archive {} already created, skip'.format(archive_path))
                continue

            res_path = params['resource_path']
            tar_file = os.path.realpath(os.path.join('pack', res_path))
            tar_dir = os.path.dirname(tar_file)
            if not os.path.exists(tar_dir):
                os.makedirs(tar_dir)

            logger.debug('Archive src: {} to dst: {}'.format(archive_path, tar_file))
            with tarfile.open(tar_file, 'w:gz') as tar:
                tar.add(os.path.join(pack_dir, archive_path), arcname=os.path.basename(archive_path))

        self.Parameters.out_build_artifacts = self.out_build_artifacts

        if self.Parameters.junit_report:
            self.Parameters.out_junit_report = results_resource_html

    def post_build(self, source_dir, output_dir, pack_dir):
        try:
            self.__post_build(source_dir, output_dir, pack_dir)
        except Exception as exc:
            self.on_exception(exc)
        # this allows task to save created resources before failing
        if self.Context.exceptions:
            raise TaskFailure('FORCE_FAILURE: Delayed exceptions found, see logs above')

    def on_success(self, prev_status):
        # don't forget to mark resources as ready
        YaMake2.on_success(self, prev_status)

        if not self.Parameters.release_on_success:
            return
        task_class = sdk2.Task["RELEASE_GIVEN_TASK"]
        sub_task = task_class(
            self,
            description="Autorelease on task success (task id: {})".format(str(self.id)),
            check_already_released=False,
            task=str(self.id),
            type=self.Parameters.release_status,
            subject=self.Parameters.release_subject
        )
        sub_task.save().enqueue()
