# coding: utf-8
import logging
import urllib2
from xml.etree import ElementTree

from sandbox import common
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common.nanny import nanny

ARTIFACTORY_TIMEOUT_SECONDS = 30
VERSION_KEY = 'version'


def create_version_parameter(artifact_dir, artifact_description):
    """
    Parameter with 'choices' field set is displayed as a dropdown.
    If there was an error, we need to set the value manually, so need some conditional logic
    """

    class VersionParameter(SandboxStringParameter):
        name = VERSION_KEY
        description = artifact_description
        required = True

        @common.utils.singleton_classproperty
        def metadata_tree(cls):
            xml_url = artifact_dir + 'maven-metadata.xml'
            metadata_xml = urllib2.urlopen(xml_url, timeout=ARTIFACTORY_TIMEOUT_SECONDS).read()
            return ElementTree.fromstring(metadata_xml)

        @common.utils.singleton_classproperty
        def choices(cls):
            try:
                versions = [element.text for element in cls.metadata_tree.find('versioning').find('versions')]
                return [(x, x) for x in reversed(versions)]
            except:
                pass

        @common.utils.singleton_classproperty
        def default_value(cls):
            try:
                return cls.metadata_tree.find('versioning').find('release').text
            except Exception as e:
                logging.info("Exception retrieving maven metadata: {}".format(e))
                return "Could not retrieve from artifactory, refresh or enter manually"

    return VersionParameter


class BuildFromArtifactoryTask(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Get jar from maven, make it a resource and register in nanny on release
    """

    execution_space = 5000

    def _resource_path(self):
        """
        Basename of resource file on disk
        """
        raise NotImplementedError

    def _artifact_dir(self):
        """
        Directory with artifact versions on HTTP server
        """
        raise NotImplementedError

    def _artifact_basename_template(self):
        raise NotImplementedError

    def _resource_type(self):
        """
        Sandbox type of resource to create (one of classes from resource_types.py)
        """
        raise NotImplementedError

    def _resource_id_context_key(self):
        """
        Context key in which chosen resource version is saved
        """
        raise NotImplementedError

    def _get_resource_ttl(self):
        """
        resource ttl to extend in inherited classes
        """
        return 14

    def _retrieve_jar(self):
        with open(self._resource_path(), 'w') as resource_file:
            version = self.ctx.get(VERSION_KEY)
            jar_url = self._artifact_dir() + ('{version}/' + self._artifact_basename_template()).format(
                version=version)
            logging.info("Jar url is {}".format(jar_url))
            resource_file.write(urllib2.urlopen(jar_url, timeout=ARTIFACTORY_TIMEOUT_SECONDS).read())

    def _make_resource(self):
        return self.create_resource(
            description=self.descr,
            resource_path=self._resource_path(),
            resource_type=self._resource_type(),
            attributes={
                "version": self.ctx.get(VERSION_KEY),
                "ttl": self._get_resource_ttl(),
            }
        ).id

    def arcadia_info(self):
        return None, self.ctx.get(VERSION_KEY), None

    def on_execute(self):
        if self._resource_id_context_key() not in self.ctx:
            self._retrieve_jar()
            self.ctx[self._resource_id_context_key()] = self._make_resource()
