# coding: utf-8

import os
import platform
import contextlib
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common import constants
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.environments import DepotToolsEnvironment
from sandbox.projects.common.build import ArcadiaTask
from sandbox.projects.common.build.parameters import ArcadiaPatch
from sandbox.projects.common.arcadia import sdk

import sandbox.common.types.client as ctc
from sandbox.common.types.task import Status

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import svn

from . import virtualenv
from . import jclient
from . import git
from . import compiler
from . import debian


class GitTagParameter(parameters.SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = 'master'
    required = True


class RepoNameParameter(parameters.SandboxStringParameter):
    name = 'repo_name'
    description = 'Repo name'
    default_value = 'yandex-precise'
    required = True


class VersionPostfixParameter(parameters.SandboxStringParameter):
    name = 'version_postfix'
    description = 'Custom version postfix'
    default_value = ''


class IncludePythonParameter(parameters.SandboxBoolParameter):
    name = 'include_python'
    description = 'Copy skynet python into bundle'
    default_value = False


class TargetPrefixParameter(parameters.SandboxStringParameter):
    name = 'target_prefix'
    description = 'bin/python prefix for shebang'


class ContainerParameter(parameters.Container):
    description = 'container'


class BaseBuildJugglerTask(nanny.ReleaseToNannyTask, task.SandboxTask, object):
    """
    Abstract build class that can clone repo and setup compiler.
    """

    type = 'BASE_BUILD_JUGGLER'
    VAULT_OWNER = 'JUGGLER'
    SSH_KEY_VAULT_NAME = 'robot-juggling-ssh'

    environment = []
    input_parameters = [
        GitTagParameter,
        IncludePythonParameter,
        TargetPrefixParameter
    ]

    REPO_NAME = None
    GIT_URL_TEMPLATE = "ssh://git@bb.yandex-team.ru/juggler/{name}.git"
    CHECKOUT_PATH_TEMPLATE = "{name}_repo"

    DEFAULT_PRESERVE_ENVVAR = ('CC', 'CXX', 'CPATH', 'LIBRARY_PATH')

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get(GitTagParameter.name), None

    @property
    def _checkout_path(self):
        assert self.REPO_NAME is not None, "no REPO_NAME specified"
        return self.path(self.CHECKOUT_PATH_TEMPLATE.format(name=self.REPO_NAME))

    @property
    def _git_url(self):
        assert self.REPO_NAME is not None, "no REPO_NAME specified"
        return self.GIT_URL_TEMPLATE.format(name=self.REPO_NAME)

    def _checkout(self, submodules=True):
        """
        Clone git repository and checkout to given branch/tag.
        """
        DepotToolsEnvironment(version='0.0.1').prepare()
        if 'ref_id' in self.ctx and 'ref_sha' in self.ctx:
            self.ctx[GitTagParameter.name] = self.ctx['ref_id']
            tag = self.ctx['ref_sha']
        else:
            tag = self.ctx['ref_id'] = self.ctx[GitTagParameter.name]
        # remember commit we build
        with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
            commit = self.ctx['ref_sha'] = git.checkout(
                self._git_url, self._checkout_path, tag, update_submodules=submodules)
        return commit

    def _compiler_environ(self):
        """
        Prepare compiler environment.
        """
        return compiler.compiler_context(self.path('bin'))

    def _extract_changelog_version(self):
        return debian.extract_changelog_version(self._checkout_path)

    def _extract_changelog_comment(self):
        return debian.extract_changelog_comment(self._checkout_path)

    def _update_changelog(self, version_postfix, message='Hand-made build with empty changelog message'):
        """
        Write new version with given postfix to debian/changelog.
        """
        return debian.update_changelog(self._checkout_path, version_postfix, message)

    def _check_for_changelog(self):
        """
        Check last commit for changes in debian/changelog.
        """
        return debian.check_for_changelog(self._checkout_path)

    def _build_deb(self, repository_name, version, preserve_envvar=None, make_threads=1):
        """
        Build debian package and upload it to repository.
        """
        preserve_envvar = preserve_envvar if preserve_envvar is not None else self.DEFAULT_PRESERVE_ENVVAR
        debian.build_deb(self, self._checkout_path, repository_name, version,
                         preserve_envvar=preserve_envvar, make_threads=make_threads)

    def _create_conductor_ticket(self, release_status, excluded_packages=[]):
        debian.create_conductor_ticket(self, release_status, excluded_packages)

    def _build_juggler_server(self, checkout_path, archive_path, include_tests, include_python=None,
                              target_prefix=None):
        requirements = ["default.txt", "arcadia.yaml"]
        if include_tests:
            requirements.extend(("linters.txt", "test.txt", "release.txt"))
        version = self._build_venv(
            [os.path.join(checkout_path, "requirements", name) for name in requirements],
            target_path=archive_path,
            ignored_paths={'lib/juggler/skyhelp/skyhelp-bin.py'},
            include_python=include_python,
            target_prefix=target_prefix
        )
        return version

    def _build_venv(self, requirements_list, target_path, ignored_paths=None, include_python=None, target_prefix=None):
        """
        Create virtualenv with specified requirements and return package version.
        """
        venv_path = self.path("build")
        python_path = None
        include_python = self.ctx.get(IncludePythonParameter.name) if include_python is None else include_python
        target_prefix = self.ctx.get(TargetPrefixParameter.name) if target_prefix is None else target_prefix
        checkout_path = self._checkout_path

        if include_python:
            python_path = '/skynet/python'
        with virtualenv.venv_context(self, venv_path, requirements_list, checkout_path, python_path, ignored_paths,
                                     target_prefix) as venv:
            # create archive from virtualenv
            if python_path is not None:
                with contextlib.closing(tarfile.open(target_path, 'w:gz')) as tar_file:
                    for fname in os.listdir(venv.root_dir):
                        if fname in ('pip_wrapper.py', '.pydistutils.cfg'):
                            continue
                        tar_file.add(
                            os.path.join(venv.root_dir, fname),
                            arcname=fname
                        )
            else:
                venv.pack_bundle(target_path)

            # get package version
            proc = process.run_process(
                [os.path.join(venv.root_dir, 'bin', 'python'), 'setup.py', '--version'],
                outs_to_pipe=True, work_dir=checkout_path)
            stdout_data, _ = proc.communicate()
            return stdout_data.strip()

    def _make_resource(self, path, version, resource_type=resource_types.OTHER_RESOURCE, arch='linux', attributes=None,
                       **kwargs):
        """
        Simply create sandbox resource.
        """
        resource_attributes = {'version': version, 'platform': platform.platform()}
        if attributes:
            resource_attributes.update(attributes)
        self.create_resource(
            resource_path=path,
            resource_type=resource_type,
            arch=arch,
            attributes=resource_attributes,
            **kwargs
        )

    def _send_events_to_juggler(self, host, service, status, description):
        with self.current_action("sending event to juggler"):
            jclient.send_events_to_juggler(host, service, status, description)

    def _build_multiple_resources(self, task_type, resource_type_list, input_parameters=None):
        build_key = '_build_task_id_{0}'.format(task_type)
        build_task_id = self.ctx.get(build_key)
        if build_task_id is None:
            input_parameters = input_parameters or {}
            input_parameters.update({
                'tag': self.ctx.get(GitTagParameter.name),
                'ref_id': self.ctx['ref_id'],
                'ref_sha': self.ctx['ref_sha'],
            })
            build_task = self.create_subtask(
                task_type=task_type,
                description='Created by {0}'.format(self.type),
                input_parameters=input_parameters
            )
            self.ctx[build_key] = build_task.id
            self.wait_tasks(
                tasks=[build_task],
                statuses=tuple(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True)

        build_task = channel.channel.sandbox.get_task(build_task_id)
        if not build_task.is_done():
            raise errors.SandboxTaskFailureError('{0} failed'.format(task_type))

        result = []
        for resource_type in resource_type_list:
            resources = channel.channel.sandbox.list_resources(
                resource_type=resource_type,
                task_id=build_task_id)
            if not resources or not all(x.is_ready() for x in resources):
                raise errors.SandboxTaskFailureError('{0} not ready or not found'.format(resource_type))
            result.extend(resources)

        return result

    def _build_resource(self, task_type, resource_type, input_parameters=None):
        resources = self._build_multiple_resources(task_type, [resource_type], input_parameters)
        if len(resources) != 1:
            raise errors.SandboxTaskFailureError('{0} not found'.format(resource_type))
        return self.sync_resource(resources[0].id)


class BaseBuildJugglerDebTask(BaseBuildJugglerTask):
    """
    Abstract build class that can clone repo and setup compiler.
    """

    type = 'BASE_BUILD_JUGGLER_DEB'

    MAKE_THREADS = 1

    input_parameters = BaseBuildJugglerTask.input_parameters + [
        RepoNameParameter,
        VersionPostfixParameter
    ]
    client_tags = ctc.Tag.Group.LINUX

    def on_prepare(self):
        # checkout
        self._checkout()

    def on_execute(self):
        version_postfix = self.ctx.get(VersionPostfixParameter.name)
        if version_postfix:
            package_version = self._update_changelog(version_postfix)
        else:
            package_version = self._check_for_changelog()
        self._build_deb(self.ctx.get(RepoNameParameter.name, RepoNameParameter.default_value),
                        version=package_version,
                        make_threads=self.MAKE_THREADS)
        self.ctx["changelog_last_comment"] = self._extract_changelog_comment()


class BaseBuildJugglerArcadiaTask(nanny.ReleaseToNannyTask, ArcadiaTask.ArcadiaTask):
    """
    Abstract task to build some project from arcadia.
    """
    type = 'BASE_BUILD_JUGGLER_ARCADIA'

    TARGET_DIR = None

    def _build_deb(self, root_dir, upload_to_repo=True):
        """
        Build debian package and upload it to repository.
        """
        version_postfix = self.ctx.get(VersionPostfixParameter.name)
        if version_postfix:
            package_version = debian.update_changelog(
                root_dir, version_postfix, 'Hand-made build with empty changelog message')
        else:
            package_version = debian.extract_changelog_version(root_dir)
        repository_name = self.ctx.get(RepoNameParameter.name, RepoNameParameter.default_value)
        debian.build_deb(self, root_dir, repository_name, package_version, upload_to_repo=upload_to_repo)
        self.ctx["changelog_last_comment"] = debian.extract_changelog_comment(root_dir)
        return package_version

    def _build_arcadia_target(self, target_platform_flags=None):
        build_system = constants.YMAKE_BUILD_SYSTEM
        build_type = constants.RELEASE_BUILD_TYPE
        arcadia_src_dir = self.get_arcadia_src_dir()
        release_dir = self.abs_path(self.LOCAL_RELEASE_DIR)
        target_dirs = [self.TARGET_DIR]
        def_flags = {'SANDBOX_TASK_ID': self.id}
        if self.ctx.get(ArcadiaPatch.name):
            svn.Arcadia.apply_patch(
                arcadia_src_dir,
                self.ctx.get(ArcadiaPatch.name),
                self.abs_path()
            )
        sdk.do_build(build_system, arcadia_src_dir, target_dirs, build_type, clear_build=True,
                     results_dir=release_dir, def_flags=def_flags, target_platform_flags=target_platform_flags)
        return os.path.join(release_dir, self.TARGET_DIR)

    def _make_resource(self, path, version, resource_type=resource_types.OTHER_RESOURCE,
                       attributes={}, **kwargs):
        resource_attributes = {'version': version, 'platform': platform.platform()}
        if attributes:
            resource_attributes.update(attributes)
        if 'description' not in kwargs:
            kwargs['description'] = '{0}, version {1}'.format(str(resource_type), version)
        self.create_resource(
            resource_path=path,
            resource_type=resource_type,
            arch=self.arch,
            attributes=resource_attributes,
            **kwargs)

    def _create_conductor_ticket(self, release_status):
        debian.create_conductor_ticket(self, release_status)
