# coding: utf-8

import re
import os
import mmap
import contextlib
import errno
import shutil
import itertools
import tempfile
import fnmatch

from sandbox.sandboxsdk import process

ACTIVATE_THIS_PATCH = """
--- bin/activate_this.py        2016-02-13 01:01:34.000000000 +0300
+++ activate_this.py    2016-02-21 21:39:49.014729853 +0300
@@ -13,6 +13,20 @@
 import sys
 import os

+reference_path = os.path.abspath(os.path.join(os.path.dirname(os.path.abspath(__file__)), "python"))
+guard_name = "ACTIVATE_THIS_EXEC_FLAG"
+if os.uname()[0] == "Linux":
+    current_path = os.readlink("/proc/self/exe")
+else:
+    current_path = os.path.realpath(sys.executable)
+if os.path.realpath(reference_path) != current_path and ":" not in reference_path:
+    if guard_name in os.environ:
+        raise RuntimeError("execve already called with {0}, break the loop".format(os.environ[guard_name]))
+    env = os.environ.copy()
+    env[guard_name] = reference_path
+    os.execve(reference_path, [reference_path] + sys.argv, env)
+os.environ.pop(guard_name, None)
+
 old_os_path = os.environ['PATH']
 os.environ['PATH'] = os.path.dirname(os.path.abspath(__file__)) + os.pathsep + old_os_path
 base = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
"""

PREFIX_PATCH = """
--- a/activate_this.py
+++ b/activate_this.py
@@ -36,7 +36,7 @@ else:
 prev_sys_path = list(sys.path)
 import site
 site.addsitedir(site_packages)
-sys.real_prefix = sys.prefix
+sys.real_prefix = "{real_prefix}"
 sys.prefix = base
 # Move the added items to the front of the path:
 new_sys_path = []
"""

SITE_PATCH = """
--- lib/python2.7/site.py       2016-02-13 01:01:33.000000000 +0300
+++ site.py     2016-02-22 02:42:24.374941874 +0300
@@ -550,9 +550,7 @@
         pass

 def virtual_install_main_packages():
-    f = open(os.path.join(os.path.dirname(__file__), 'orig-prefix.txt'))
-    sys.real_prefix = f.read().strip()
-    f.close()
+    sys.real_prefix = os.path.abspath(os.path.join(os.path.dirname(os.path.abspath(__file__)), '..', '..', 'python'))
     pos = 2
     hardcoded_relative_dirs = []
     if sys.path[0] == '':
"""

PROLONG_TPL = (
    """import os;"""
    """ activate_this=os.path.join(os.path.dirname(os.path.realpath(__file__)), {0}, 'activate_this.py');"""
    """ exec(compile(open(activate_this).read(), activate_this, 'exec'), dict(__file__=activate_this));"""
    """ del os, activate_this\n"""
)


def replace_all_until(view, target):
    source = view[:view.tell()]
    resulting_size = view.size() - len(source) + len(target)
    if resulting_size > view.size():
        view.resize(resulting_size)
    if len(target) != len(source):
        view.move(len(target), len(source), resulting_size - len(target))
    if resulting_size < view.size():
        view.resize(resulting_size)
    view[:len(target)] = bytes(target)


def iter_files(path):
    for root, dirs, files in os.walk(path):
        for name in itertools.chain(dirs, files):
            yield os.path.join(root, name)


def fix_shebang(venv_path, target_prefix, ignored_paths=None):
    header = "#!"
    venv_path = os.path.abspath(venv_path)
    for file_path in iter_files(venv_path):
        if os.path.islink(file_path) or not os.path.isfile(file_path):
            continue
        elif os.path.join("lib", "python") in file_path:
            continue
        elif os.path.splitext(file_path)[-1] not in ("", ".py", '.test'):
            continue
        elif ignored_paths is not None and trim_path(file_path, venv_path) in ignored_paths:
            continue
        try:
            with open(file_path, "r+") as fd:
                if not os.fstat(fd.fileno()).st_size:
                    continue
                with contextlib.closing(mmap.mmap(fd.fileno(), 0)) as view:
                    if view[:len(header)] != header:
                        continue
                    elif "python" not in view.readline():
                        continue
                    parts = [header + target_prefix + "\n"]
                    while True:
                        prev_pos = view.tell()
                        line = view.readline()
                        if not line.strip() or line.startswith("#") or line.startswith("\"\"\""):
                            # empty line or started with comment
                            parts.append(line)
                        elif "__future__" in line and "import" in line:
                            parts.append(line)
                        elif "del os, activate_this" in line:
                            break
                        else:
                            view.seek(prev_pos, os.SEEK_SET)
                            break
                    level = trim_path(file_path, venv_path).count(os.path.sep)
                    parts.append(PROLONG_TPL.format(", ".join(["'..'"] * level + ["'bin'"])))
                    replace_all_until(view, "".join(parts))
        except IOError as exc:
            if exc.errno == errno.ETXTBSY or exc.errno == errno.EACCES:
                continue
            raise


def find_deps(file_path, source_path, seen=None):
    stack = []
    seen = set() if seen is None else seen
    proc = process.run_process(["ldd", file_path], outs_to_pipe=True)
    stdout, _ = proc.communicate()
    for lib_path in re.findall(r"^.+ => (.+) \(.+\)$", stdout, re.MULTILINE):
        if lib_path.startswith(os.path.join(source_path, "lib")) and lib_path not in seen:
            seen.add(lib_path)
            stack.append(lib_path)
    while stack:
        lib_path = stack.pop()
        find_deps(lib_path, source_path, seen)
    return seen


def find_elfs(path):
    for file_path in iter_files(path):
        if os.path.islink(file_path) or not os.path.isfile(file_path):
            continue
        elif file_path.endswith(".dsym"):
            continue
        try:
            with open(file_path, "rb") as fd:
                if not os.fstat(fd.fileno()).st_size:
                    continue
                with contextlib.closing(mmap.mmap(fd.fileno(), 0, access=mmap.ACCESS_READ)) as view:
                    if view[:4] != b'\x7fELF':
                        continue
        except IOError as exc:
            if exc.errno == errno.ETXTBSY:
                continue
            raise
        else:
            yield file_path


def find_libraries(path, source_path):
    result = set()
    path = os.path.abspath(path)
    for file_path in find_elfs(path):
        find_deps(file_path, source_path, result)
    return result


def create_symlink(source, link_name):
    try:
        os.symlink(source, link_name)
    except OSError as exc:
        if exc.errno != errno.EEXIST:
            raise
        os.unlink(link_name)
        os.symlink(source, link_name)


def trim_path(path, root):
    return path.replace(root, "").strip(os.path.sep)


def fix_symlinks(venv_path, source_path, python_path):
    assert python_path.startswith(venv_path)
    python_dir = trim_path(python_path, venv_path)

    for file_path in iter_files(os.path.join(venv_path, "lib")):
        if not os.path.islink(file_path):
            continue

        orig_path = os.readlink(file_path)
        if not orig_path.startswith(source_path):
            continue

        orig_path = trim_path(orig_path, source_path)
        if os.path.exists(os.path.join(python_path, orig_path)):
            target_path = os.path.join(
                os.path.sep.join([".."] * trim_path(file_path, venv_path).count(os.path.sep)),
                python_dir, orig_path)
            create_symlink(target_path, file_path)
        else:
            os.unlink(file_path)

        pyc_file_path = "{0}c".format(file_path)
        if os.path.exists(pyc_file_path):
            os.unlink(pyc_file_path)


def copy_tree(python_path, libraries, source_path):
    libraries_masks = tuple({"{0}.so*".format(os.path.basename(x).partition(".so")[0]) for x in libraries})
    prefix_map = {
        "": ("lib", ),
        "lib": ("python2.7", "engines") + libraries_masks,
        "lib/engines": ("lib", ),
        "lib/python2.7": ("lib-dynload", "bsddb", "compiler", "ctypes", "curses",
                          "distutils", "email", "encodings", "hotshot", "importlib",
                          "json", "lib-tk", "lib2to3", "logging", "multiprocessing",
                          "plat-linux2", "pydoc_data", "sqlite3", "unittest",
                          "wsgiref", "xml", "*.py", "*.egg-info", "*.doc", "*.txt"),
        "lib/python2.7/lib-dynload": ("*.so", "*.egg-info"),
        "lib/python2.7/bsddb": ("*.py", ),
        "lib/python2.7/compiler": ("*.py", ),
        "lib/python2.7/ctypes": ("macholib", "*.py"),
        "lib/python2.7/ctypes/macholib": ("*.py", "README.ctypes", "fetch_macholib"),
        "lib/python2.7/curses": ("*.py", ),
        "lib/python2.7/distutils": ("command", "*.py", "README"),
        "lib/python2.7/distutils/command": ("*.py", "command_template"),
        "lib/python2.7/email": ("mime", "*.py"),
        "lib/python2.7/email/mime": ("*.py", ),
        "lib/python2.7/encodings": ("*.py", ),
        "lib/python2.7/hotshot": ("*.py", ),
        "lib/python2.7/importlib": ("*.py", ),
        "lib/python2.7/json": ("*.py", ),
        "lib/python2.7/lib-tk": ("*.py", ),
        "lib/python2.7/lib2to3": ("fixes", "pgen2", "*.py", "*.txt", "*.pickle"),
        "lib/python2.7/lib2to3/fixes": ("*.py", ),
        "lib/python2.7/lib2to3/pgen2": ("*.py", ),
        "lib/python2.7/logging": ("*.py", ),
        "lib/python2.7/multiprocessing": ("dummy", "*.py"),
        "lib/python2.7/multiprocessing/dummy": ("*.py", ),
        "lib/python2.7/plat-linux2": ("*.py", ),
        "lib/python2.7/pydoc_data": ("*.py", ),
        "lib/python2.7/sqlite3": ("*.py", ),
        "lib/python2.7/unittest": ("*.py", ),
        "lib/python2.7/wsgiref": ("*.py", ),
        "lib/python2.7/xml": ("dom", "etree", "parsers", "sax", "*.py"),
        "lib/python2.7/xml/dom": ("*.py", ),
        "lib/python2.7/xml/etree": ("*.py", ),
        "lib/python2.7/xml/parsers": ("*.py", ),
        "lib/python2.7/xml/sax": ("*.py", ),
    }

    def get_prefixes(path):
        parts = path.replace(source_path, "").strip(os.path.sep).split(os.path.sep)
        for pos in range(len(parts), 0, -1):
            key = os.path.sep.join(parts[:pos])
            if key in prefix_map:
                return prefix_map[key]
        return ()

    def get_excluded(path, name_list):
        prefix_list = get_prefixes(path)
        included = set()
        for name in name_list:
            for prefix in prefix_list:
                if fnmatch.fnmatch(name, prefix):
                    included.add(name)
        return set(name_list) - included

    shutil.copytree(source_path, python_path, symlinks=True, ignore=get_excluded)


def copy_python(venv_path, source_path):
    python_path = os.path.join(venv_path, "python")

    libraries = find_libraries(venv_path, source_path)
    libraries.update(find_libraries(os.path.join(source_path, "lib", "python2.7", "lib-dynload"), source_path))
    copy_tree(python_path, libraries, source_path)

    return python_path


def fix_rpath(venv_path, python_path):
    assert python_path.startswith(venv_path)
    python_dir = trim_path(python_path, venv_path)

    for file_path in itertools.chain(find_elfs(os.path.join(venv_path, "lib")),
                                     find_elfs(os.path.join(venv_path, "bin"))):
        rpath = os.path.join(
            "$ORIGIN",
            os.path.sep.join([".."] * trim_path(file_path, venv_path).count(os.path.sep)),
            python_dir, "lib"
        )
        process.run_process(["/skynet/python/bin/chrpath", "-r", rpath, file_path],
                            log_prefix="fix_rpath", check=False)


def patch_activate_this(venv_path):
    with tempfile.TemporaryFile() as stream:
        stream.write(ACTIVATE_THIS_PATCH)
        stream.seek(0)
        process.run_process(
            ["patch", "--forward", "-s", os.path.join(venv_path, "bin", "activate_this.py")],
            stdin=stream, log_prefix="patch_activate_this")


def patch_real_prefix(venv_path, real_prefix="/skynet/python"):
    with tempfile.TemporaryFile() as stream:
        stream.write(PREFIX_PATCH.format(real_prefix=real_prefix))
        stream.seek(0)
        process.run_process(
            ["patch", "--forward", "-s", os.path.join(venv_path, "bin", "activate_this.py")],
            stdin=stream, log_prefix="patch_real_prefix")


def patch_site(venv_path):
    with tempfile.TemporaryFile() as stream:
        stream.write(SITE_PATCH)
        stream.seek(0)
        process.run_process(
            ["patch", "--forward", "-s", os.path.join(venv_path, "lib", "python2.7", "site.py")],
            stdin=stream, log_prefix="patch_site")


def remove_pyc(path):
    for file_path in iter_files(path):
        if os.path.islink(file_path) or not os.path.isfile(file_path):
            continue
        elif not file_path.endswith(".pyc"):
            continue
        os.unlink(file_path)


def make_bundle(venv_path, source_path="/skynet/python", ignored_paths=None, target_prefix=None):
    venv_path = os.path.abspath(venv_path)
    fix_shebang(
        venv_path,
        os.path.join(target_prefix if target_prefix else source_path, "bin", "python"),
        ignored_paths,
    )
    python_path = copy_python(venv_path, source_path)
    fix_symlinks(venv_path, source_path, python_path)
    fix_rpath(venv_path, python_path)
    patch_activate_this(venv_path)
    patch_site(venv_path)
    remove_pyc(venv_path)
