# coding: utf-8

import os
import re
import json
import tarfile
import logging
import hashlib
import itertools

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.channel as sdk_channel
from sandbox.projects.common.juggler.s3 import S3Client
from sandbox.projects.common.juggler import jclient


class ForceResourceCreationParameter(sdk_parameters.SandboxBoolParameter):
    name = 'force_resource'
    description = 'Force resource creation'
    default_value = False


class BaseBuildJugglerChecksBundleTask(sdk_task.SandboxTask):
    execution_space = 500
    cores = 1
    required_ram = 1024

    input_parameters = []

    @staticmethod
    def _build_archive(meta, files_list):
        manifest_name = "MANIFEST.json"
        bundle_name = meta["bundle_name"]

        with open(manifest_name, "w") as stream:
            json.dump(meta, stream, indent=4, sort_keys=True)

        archive_path = "{0}.tar.gz".format(bundle_name)
        with tarfile.open(archive_path, "w:gz") as tar:
            for local_path, bundle_path in sorted(files_list):
                tar.add(local_path, arcname=bundle_path)

            tar.add(manifest_name, arcname=manifest_name)

        return bundle_name, archive_path

    @classmethod
    def _build_old_style_remote_archive(cls, meta, files_list):
        meta = meta.copy()
        manifest_name = "meta.json"
        tmp_archive_name = "checks.tar"
        bundle_name = meta["bundle_name"]

        with tarfile.open(tmp_archive_name, "w") as tar:
            for local_path, bundle_path in sorted(files_list):
                tar.add(local_path, arcname=os.path.join("data", bundle_path))

        meta["bundle_hash"] = cls._get_file_hash(tmp_archive_name)
        with open(manifest_name, "w") as stream:
            json.dump(meta, stream, indent=4, sort_keys=True)

        archive_path = "{0}.tar.gz".format(bundle_name)
        with tarfile.open(archive_path, "w:gz") as tar:
            tar.add(tmp_archive_name, arcname="data.tar")
            tar.add(manifest_name, arcname="meta.json")

        return bundle_name, archive_path

    def _make_bundle_resource(self, meta, files_list, resource_type, old_style=False, s3_user=None, s3_endpoint=None):
        if old_style:
            bundle_name, archive_path = self._build_old_style_remote_archive(meta, files_list)
        else:
            bundle_name, archive_path = self._build_archive(meta, files_list)

        unique_hash = self._compute_unique_hash(meta)
        attributes = {
            "build_number": self.id,
            "unique_hash": unique_hash,
            "name": bundle_name
        }

        mds_client = None
        if s3_user and s3_endpoint:
            auth_data = json.loads(self.get_vault_data(s3_user.get('vault_owner'), s3_user.get('vault_name')))
            mds_client = S3Client(s3_endpoint=s3_endpoint, **auth_data)

        if not self.ctx.get(ForceResourceCreationParameter.name):
            # checking only latest resource in case there were svn reverts and whatnot
            old_resource_list = sdk_channel.channel.sandbox.list_resources(
                resource_type=resource_type,
                limit=1,
                attribute_name="name",
                attribute_value=bundle_name,
                order_by="-id") or []
            if old_resource_list:
                resource = old_resource_list[0]
                if self._check_existing_resource(resource, unique_hash, mds_client):
                    logging.info("Found resource {0} with same bundle hash, and it also exists in mds".format(resource.id))
                    self._send_ok_to_juggler(bundle_name, description="Found same bundle resource {0}".format(resource.id))
                    return

        if mds_client:
            try:
                attributes.update(mds_client.upload(
                    local_path=archive_path,
                    mds_path="{0}-{1}.tar.gz".format(bundle_name, self.id)
                ))
            except Exception:
                logging.exception("Can't upload resource to MDS")

        current_resource = self.create_resource(
            description="Juggler {0} checks bundle".format(bundle_name),
            resource_path=archive_path,
            resource_type=resource_type,
            attributes=attributes
        )

        if mds_client:
            # TODO: normal MDS cleanup
            old_resource_list = sdk_channel.channel.sandbox.list_resources(
                resource_type=resource_type,
                limit=10,
                attribute_name="name",
                attribute_value=bundle_name,
                order_by="-id") or []
            for resource in old_resource_list[5:]:
                if resource.id == current_resource.id:
                    continue
                if "mds_key" not in resource.attributes:
                    continue
                try:
                    if not mds_client.delete(resource.attributes["mds_key"]):
                        # let's assume that older resources were deleted
                        break
                except Exception:
                    logging.exception("Can't delete %r from MDS", resource)

        self._send_ok_to_juggler(bundle_name, description="New resource {0} built successfully".format(current_resource.id))

    @staticmethod
    def _compute_unique_hash(meta):
        return hashlib.sha256(json.dumps({
            "files": sorted(meta["checks"], key=lambda x: (x["check_name"], x["check_hash"])),
            "env": meta["env"]
        }, sort_keys=True)).hexdigest()

    @staticmethod
    def _check_existing_resource(resource, unique_hash, mds_client):
        if resource.is_deleted():
            return False
        if resource.attributes.get("unique_hash") != unique_hash:
            return False
        if not mds_client:
            return True
        if "mds_key" not in resource.attributes or not mds_client.exists(resource.attributes["mds_key"]):
            return False
        return True

    @staticmethod
    def _send_ok_to_juggler(bundle_name, description):
        jclient.send_events_to_juggler('juggler.sandbox', 'build_bundle_{0}'.format(bundle_name), 'OK',
                                       description)

    @staticmethod
    def _get_file_hash(file_path, chunk_size=65536):
        hasher = hashlib.sha256()
        with open(file_path, "rb") as stream:
            for chunk in iter(lambda: stream.read(chunk_size), b""):
                hasher.update(chunk)
        return hasher.hexdigest()

    @staticmethod
    def _get_provided_services(file_path):
        with open(file_path) as stream:
            text = stream.read()

        matches_list = re.findall(r"#\s*[Pp]rovides?:\s*([^\n]+)", text)
        services = set()
        for match in matches_list:
            services.update(itertools.ifilter(None, re.split(r"[\s,;]", match)))

        return list(services)
