# coding: utf-8

import os
import logging
import contextlib

from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import environments


def merge_with_environ(key, *args):
    """
    Merge given arguments with existed environment variable using separator.
    """
    values = []
    values.extend(args)
    if key in os.environ:
        values.append(os.environ[key])
    if key in ('LDFLAGS', 'CFLAGS'):
        sep = ' '
    else:
        sep = ':'
    return sep.join(values)


@contextlib.contextmanager
def compiler_context(bin_path, gcc_env=None):
    """
    Prepare compiler environment.
    """
    logging.info('preparing environment (gcc, g++)...')

    if gcc_env is None:
        gcc_env = environments.GCCEnvironment(version='4.9.2')
        gcc_env.prepare()
    compilers = gcc_env.get_compilers()

    environ = {}
    environ['CC'] = gcc_binary = compilers['CC']
    environ['CXX'] = gxx_binary = compilers['CXX']
    environ['PATH'] = merge_with_environ('PATH', bin_path, os.path.dirname(compilers['CC']))
    logging.info('Fetched compilers are: CC=%s CXX=%s', gcc_binary, gxx_binary)

    paths.make_folder(bin_path, delete_content=True)
    try:
        os.symlink(gcc_binary, os.path.join(bin_path, 'gcc'))
        os.symlink(gxx_binary, os.path.join(bin_path, 'g++'))
        with process.CustomOsEnviron(environ):
            yield
    finally:
        paths.remove_path(bin_path)
